package com.ejianc.business.supbusiness.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * 消息发送工具类
 */
public class SendMsgUtils {
    /** 日志 */
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /** 消息类型 msgType:notice */
    private static final String MSG_TYPE_NOTICE = "notice";

    /**
     * 发送系统消息（只发送系统消息，消息类型默认为 notice）
     *
     * @param userList       消息接收用户
     * @param subject        消息标题
     * @param content        消息内容
     * @param tenantId       租户ID
     * @param pushMessageApi 发送消息API
     */
    public void sendSysMsg(List<String> userList, String subject, String content, Long tenantId,
                           IPushMessageApi pushMessageApi,String mobileUrl) {
        List<String> messageTypeList = new ArrayList<>();
        //系统消息
        messageTypeList.add("sys");
        String tenantIdStr = "";
        if (tenantId != null) tenantIdStr = tenantId.toString();
        this.sendMsg(messageTypeList, userList, MSG_TYPE_NOTICE, subject, content, tenantIdStr, null, null, mobileUrl,
                pushMessageApi);
    }

    /**
     * 发送消息
     *
     * @param messageTypeList 消息发送类型
     * @param userList        消息接收用户
     * @param msgType         消息类型
     * @param subject         消息标题
     * @param content         消息内容
     * @param tenantId        租户ID
     * @param cgddAgentid     cgddAgentid（微信消息使用）
     * @param cgddSecret      cgddSecret （微信消息使用）
     * @param url             微信消息URL（微信消息使用）
     * @param pushMessageApi  发送消息API
     */
    public void sendMsg(List<String> messageTypeList, List<String> userList, String msgType, String subject,
                        String content, String tenantId, String cgddAgentid, String cgddSecret, String url,
                        IPushMessageApi pushMessageApi) {
        logger.info("发送消息开始！===========");
        PushMsgParameter parameter = new PushMsgParameter();
        List<String> channel = new ArrayList<>();
        if (messageTypeList.contains("sys")) {
            // 系统消息
            channel.add(PushMsgParameter.CHANNEL_TYPE_SYS);
        }
        if (messageTypeList.contains("smsg")) {
            // 短信消息
//             channel.add(PushMsgParameter.CHANNEL_TYPE_NOTE);
        }
        if (messageTypeList.contains("weChat")) {
            // 系统消息
            channel.add(PushMsgParameter.CHANNEL_TYPE_WEXINEE);
        }
        String[] receivers = userList.toArray(new String[0]);
        parameter.setReceivers(receivers);// 收信人
        parameter.setChannel(channel.toArray(new String[0]));// 消息类型
        parameter.setMsgType(msgType);
        parameter.setSubject(subject);// 标题
        parameter.setContent(content);// 内容
        parameter.setTenantId(tenantId); // 租户ID
        parameter.setSendUserId(InvocationInfoProxy.getUserid());
        parameter.setMobileUrl(url);

        if (messageTypeList.contains("weChat")) {
            JSONObject weixineeParams = new JSONObject();
            weixineeParams.put("agentid", cgddAgentid);
            weixineeParams.put("secret", cgddSecret);
            weixineeParams.put("msgtype", "textcard");
            weixineeParams.put("title", "消息提醒");
            weixineeParams.put("description", subject);
            weixineeParams.put("url", url);
            weixineeParams.put("btntxt", "点击查看");
            parameter.setWeixineeParams(weixineeParams);
        }
        try {
            logger.info("消息发送参数---------------->{}", JSON.toJSONString(parameter));
            CommonResponse<String> result = pushMessageApi.pushMessage(parameter);
            if (result.isSuccess()) {
                logger.info("消息发送成功---------------->" + result.getMsg());
            }
            else {
                logger.error("消息发送失败---------------->" + result.getMsg());
            }
        }
        catch (Exception e) {
            logger.error("调用消息中心RPC服务异常--------------", e);
        }
    }

    /**
     * 发送微信公众号消息
     *
     * @param userList              用户列表
     * @param subject               消息标题
     * @param content               消息内容
     * @param tenantId              租户id
     * @param weixinMap             微信消息内容
     * @param weixinCheckTemplateId 微信消息模板
     * @param mobileUrl             移动端
     * @param pcUrl                 PC端
     * @param wechatUrl             微信消息跳转路径
     * @param pushMessageApi        发送消息API
     */
    public void sendWeiXinMsg(List<String> userList, String subject, String content, String tenantId,
                              HashMap<String, Object> weixinMap, String weixinCheckTemplateId, String mobileUrl,
                              String pcUrl, String wechatUrl,
                              IPushMessageApi pushMessageApi) {
        PushMsgParameter parameter = new PushMsgParameter();
        List<String> channel = new ArrayList<>();
        channel.add("weixinSupplier");
        channel.add(PushMsgParameter.CHANNEL_TYPE_SYS);

        String[] receivers = userList.toArray(new String[0]);
        parameter.setReceivers(receivers);// 收信人
        parameter.setChannel(channel.toArray(new String[0]));// 消息类型
        parameter.setMsgType(MSG_TYPE_NOTICE);
        parameter.setSubject(subject);// 标题
        parameter.setContent(content);// 内容
        parameter.setTenantId(tenantId); // 租户ID
        parameter.setPcUrl(pcUrl);
        parameter.setMobileUrl(mobileUrl);
        parameter.setSendUserId(InvocationInfoProxy.getUserid());

        // 微信消息
        if (StringUtils.isNotBlank(weixinCheckTemplateId)) {
            JSONObject weixinParams = new JSONObject();
            weixinParams.put("template_id", weixinCheckTemplateId);
            JSONObject wxparams = new JSONObject();
            wxparams.putAll(weixinMap);
            weixinParams.put("data", wxparams);
            weixinParams.put("url", wechatUrl);
            parameter.setWeixinParams(weixinParams);
        }
        else {
            logger.info("未配置微信公众号模板，故不发微信公众号消息");
        }

        try {
            logger.info("微信消息发送参数：{}", JSONObject.toJSONString(parameter));
            CommonResponse<String> result = pushMessageApi.pushMessage(parameter);
            logger.info("微信消息发送结果：{}", JSONObject.toJSONString(result));
            if (result.isSuccess()) {
                logger.error("消息发送成功---------------->" + result.getMsg());
            }
            else {
                logger.error("消息发送失败---------------->" + result.getMsg());
            }
        }
        catch (Exception e) {
            logger.error("调用消息中心RPC服务异常--------------" + e);
        }
    }

}
