package com.yycc.common.utils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

/**
 * HTTP客户端请求服务类
 * 
 * @author Dio
 *
 */
public class HttpClientUtils {
	/**
	 * 创建HTTP请求客户端
	 * 
	 * @return
	 */
	public static HttpClient createHttpClient() {
		try {
			// 创建默认的httpClient实例
			HttpClientBuilder builder = HttpClientBuilder.create();
			CloseableHttpClient httpClient = builder.build();
			X509TrustManager xtm = new MyX509TrustManager();
			SSLContext ctx = SSLContext.getInstance("TLS");
			// 使用TrustManager来初始化该上下文，TrustManager只是被SSL的Socket所使用
			ctx.init(null, new TrustManager[] { xtm }, null);
			builder.setSslcontext(ctx);
			return httpClient;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 
	 * 以Post方式执行Http请求
	 * 
	 * @param httpMethod
	 * @throws Exception
	 *             return HttpResponse
	 * @since JDK 1.7
	 */
	public static HttpResponse doHttpPost(HttpPost httpPost) throws Exception {
		HttpClient httpClient = createHttpClient();
		HttpResponse response = httpClient.execute(httpPost);
		return response;
	}

	/**
	 * 
	 * 以Get方式执行Http请求
	 * 
	 * @param httpGet
	 * @throws Exception
	 *             return HttpResponse
	 * @since JDK 1.7
	 */
	public static HttpResponse doHttpGet(HttpGet httpGet) throws Exception {
		// HttpClientBuilder builder = HttpClientBuilder.create();
		// // 创建默认的httpClient实例
		// CloseableHttpClient httpClient = builder.build();
		HttpClient httpClient = createHttpClient();
		HttpResponse response = httpClient.execute(httpGet);
		return response;
	}

	/**
	 * 以Post方式执行Http请求
	 * 
	 * @since JDK 1.7
	 */
	public static String doHttpPostJson(String url, String jsonStr) {
		try {
			HttpPost httpPost = new HttpPost(url);
			// 是否存在传递参数
			if (httpPost != null && jsonStr != null && !"".equals(jsonStr)) {
				StringEntity entity = new StringEntity(jsonStr, "UTF-8");
				// 请求绑定参数
				httpPost.setEntity(entity);
			}
			HttpClient httpClient = createHttpClient();
			// 执行请求
			HttpResponse response = httpClient.execute(httpPost);
			String responseContent = null;
			HttpEntity entity = response.getEntity();
			if (entity != null) {
				responseContent = EntityUtils.toString(entity, "UTF-8");
				EntityUtils.consume(entity);// 关闭流
			}
			return responseContent;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public static String doPost(String url, Map<String, Object> paras) throws Exception {
		HttpClient httpClient = createHttpClient();
		HttpPost httpPost = new HttpPost(url);
		if (paras != null && !paras.isEmpty()) {
			List<NameValuePair> list = new ArrayList<NameValuePair>();
			for (Map.Entry<String, Object> para : paras.entrySet()) {
				String key = para.getKey();
				Object value = para.getValue();
				if (value instanceof String) {
					list.add(new BasicNameValuePair(key, (String) value));
				} else if (value instanceof List) {
					List<Object> values = (List<Object>) value;
					for (Object subva : values) {
						list.add(new BasicNameValuePair(key, String.valueOf(subva)));
					}
				}
			}
			HttpEntity entity = new UrlEncodedFormEntity(list, "UTF-8");
			httpPost.setEntity(entity);
		}
		HttpResponse response = httpClient.execute(httpPost);
		int status = response.getStatusLine().getStatusCode();
		if (response != null && status == HttpStatus.SC_OK) {
			String result = null;
			HttpEntity entity = response.getEntity();
			if (entity != null) {
				result = EntityUtils.toString(entity, "utf-8");
				EntityUtils.consume(entity);
			}
			return result;
		} else {
			throw new Exception("执行Post返回错误信息，错误代号：" + status);
		}
	}

	/**
	 * 以Post方式执行Http请求
	 * 
	 * @since JDK 1.7
	 */
	public static String doHttpPost(String url, Map<String, String> params) {
		try {
			HttpPost httpPost = new HttpPost(url);
			// 是否存在传递参数
			if (httpPost != null && params != null && !params.isEmpty()) {
				// 使用NameValuePair来保存要传递的Post参数
				List<NameValuePair> postParams = new ArrayList<NameValuePair>();
				for (Map.Entry<String, String> param : params.entrySet()) {
					// 添加要传递的参数
					postParams.add(new BasicNameValuePair(param.getKey(), param.getValue()));
				}
				// 设置字符集
				HttpEntity httpentity = new UrlEncodedFormEntity(postParams, "UTF-8");
				// 请求绑定参数
				httpPost.setEntity(httpentity);
			}
			// 创建默认的httpClient实例
			HttpClient httpClient = createHttpClient();
			// 执行请求
			HttpResponse response = httpClient.execute(httpPost);
			String responseContent = null;
			HttpEntity entity = response.getEntity();
			if (entity != null) {
				responseContent = EntityUtils.toString(entity, "UTF-8");
				EntityUtils.consume(entity);// 关闭流
			}
			return responseContent;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 基于HttpClient的发送Html-Get请求获取返回的文本内容
	 * 
	 * @param url
	 * @return
	 */
	public static synchronized String doHttpGet(String url, Map<String, String> params) {
		String httpUrl = getUrlWithParas(url, params);
		String responseText = doHttpGet(httpUrl);
		return responseText;
	}

	/**
	 * 基于HttpClient的发送Html-Get请求获取返回的文本内容
	 * 
	 * @param url
	 * @return
	 */
	public static synchronized String doHttpGet(String url) {
		try {
			HttpGet httpGet = new HttpGet(url);
			HttpClient httpClient = createHttpClient();
			// 执行请求
			HttpResponse response = httpClient.execute(httpGet);
			String responseContent = null;
			HttpEntity entity = response.getEntity();
			if (entity != null) {
				responseContent = EntityUtils.toString(entity, "UTF-8");
				EntityUtils.consume(entity);// 关闭流
			}
			return responseContent;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 用Post发送一串Json字符串
	 * 
	 * @param url
	 *            访问地址
	 * @param data
	 *            Json字符串
	 * @return
	 * @throws Exception
	 */
	public static String doPostJson(String url, String data) throws Exception {
		HttpClient httpClient = createHttpClient();
		HttpPost httpPost = new HttpPost(url);
		if (data != null) {
			StringEntity entity = new StringEntity(data, ContentType.create("application/json", Consts.UTF_8));
			httpPost.setEntity(entity);
		}

		// 执行请求
		HttpResponse response = httpClient.execute(httpPost);
		int status = response.getStatusLine().getStatusCode();
		if (status == HttpStatus.SC_OK) {
			String result = null;
			HttpEntity entity = response.getEntity();
			if (entity != null) {
				result = EntityUtils.toString(entity, "UTF-8");
				EntityUtils.consume(entity);
			}
			return result;
		} else {
			throw new Exception("地址" + url + "访问错误，错误代码：" + status);
		}
	}

	/** url and para separator **/
	public static final String URL_AND_PARA_SEPARATOR = "?";
	/** parameters separator **/
	public static final String PARAMETERS_SEPARATOR = "&";
	/** paths separator **/
	public static final String PATHS_SEPARATOR = "/";
	/** equal sign **/
	public static final String EQUAL_SIGN = "=";

	/**
	 * join url and paras
	 * 
	 * <pre>
	 * getUrlWithParas(null, {(a, b)})                        =   "?a=b";
	 * getUrlWithParas("baidu.com", {})                       =   "baidu.com";
	 * getUrlWithParas("baidu.com", {(a, b), (i, j)})         =   "baidu.com?a=b&i=j";
	 * getUrlWithParas("baidu.com", {(a, b), (i, j), (c, d)}) =   "baidu.com?a=b&i=j&c=d";
	 * </pre>
	 * 
	 * @param url
	 *            url
	 * @param parasMap
	 *            paras map, key is para name, value is para value
	 * @return if url is null, process it as empty string
	 */
	public static String getUrlWithParas(String url, Map<String, String> parasMap) {
		StringBuilder urlWithParas = new StringBuilder(StringUtils.isEmpty(url) ? "" : url);
		String paras = joinParas(parasMap);
		if (!StringUtils.isEmpty(paras)) {
			urlWithParas.append(URL_AND_PARA_SEPARATOR).append(paras);
		}
		return urlWithParas.toString();
	}

	/**
	 * join url and encoded paras
	 * 
	 * @param url
	 * @param parasMap
	 * @return
	 * @see #getUrlWithParas(String, Map)
	 * @see StringUtils#utf8Encode(String)
	 */
	public static String getUrlWithValueEncodeParas(String url, Map<String, String> parasMap) {
		StringBuilder urlWithParas = new StringBuilder(StringUtils.isEmpty(url) ? "" : url);
		String paras = joinParasWithEncodedValue(parasMap);
		if (!StringUtils.isEmpty(paras)) {
			urlWithParas.append(URL_AND_PARA_SEPARATOR).append(paras);
		}
		return urlWithParas.toString();
	}

	/**
	 * join paras
	 * 
	 * @param parasMap
	 *            paras map, key is para name, value is para value
	 * @return join key and value with {@link #EQUAL_SIGN}, join keys with
	 *         {@link #PARAMETERS_SEPARATOR}
	 */
	public static String joinParas(Map<String, String> parasMap) {
		if (parasMap == null || parasMap.size() == 0) {
			return null;
		}

		StringBuilder paras = new StringBuilder();
		Iterator<Map.Entry<String, String>> ite = parasMap.entrySet().iterator();
		while (ite.hasNext()) {
			Map.Entry<String, String> entry = (Map.Entry<String, String>) ite.next();
			paras.append(entry.getKey()).append(EQUAL_SIGN).append(entry.getValue());
			if (ite.hasNext()) {
				paras.append(PARAMETERS_SEPARATOR);
			}
		}
		return paras.toString();
	}

	/**
	 * join paras with encoded value
	 * 
	 * @param parasMap
	 * @return
	 * @see #joinParas(Map)
	 * @see StringUtils#utf8Encode(String)
	 */
	public static String joinParasWithEncodedValue(Map<String, String> parasMap) {
		StringBuilder paras = new StringBuilder("");
		if (parasMap != null && parasMap.size() > 0) {
			Iterator<Map.Entry<String, String>> ite = parasMap.entrySet().iterator();
			try {
				while (ite.hasNext()) {
					Map.Entry<String, String> entry = (Map.Entry<String, String>) ite.next();
					paras.append(entry.getKey()).append(EQUAL_SIGN).append(StringUtils.utf8Encode(entry.getValue()));
					if (ite.hasNext()) {
						paras.append(PARAMETERS_SEPARATOR);
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return paras.toString();
	}

	/**
	 * append a key and value pair to url
	 * 
	 * @param url
	 * @param paraKey
	 * @param paraValue
	 * @return
	 */
	public static String appendParaToUrl(String url, String paraKey, String paraValue) {
		if (url == null || "".equals(url)) {
			return url;
		}
		StringBuilder sb = new StringBuilder(url);
		if (!url.contains(URL_AND_PARA_SEPARATOR)) {
			sb.append(URL_AND_PARA_SEPARATOR);
		} else {
			sb.append(PARAMETERS_SEPARATOR);
		}
		return sb.append(paraKey).append(EQUAL_SIGN).append(paraValue).toString();
	}
}
