package com.yyjz.icop.pub.base.service.impl;

import java.lang.reflect.InvocationTargetException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.persistence.Table;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springside.modules.persistence.DynamicSpecifications;
import org.springside.modules.persistence.SearchFilter;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.yycc.common.exception.BusinessException;
import com.yyjz.icop.pub.base.dao.BaseDao;
import com.yyjz.icop.pub.base.dao.BaseQueryDao;
import com.yyjz.icop.pub.base.entity.BaseIdEntity;
import com.yyjz.icop.pub.base.entity.SuperBillMainEntity;
import com.yyjz.icop.pub.base.entity.SuperBillSubEntity;
import com.yyjz.icop.pub.base.service.IBaseService;
import com.yyjz.icop.pub.base.vo.BpmStateVO;
import com.yyjz.icop.pub.base.vo.SuperBillSubVO;
import com.yyjz.icop.pub.base.vo.SuperVO;
import com.yyjz.icop.pub.utils.AdvanceSearchUtil;
import com.yyjz.icop.pub.utils.SimpleBeanUtils;

/**
 * IBaseService的实现类，提供基本的保存，删除方法，主子表保存需要重新beforeSave和afterSave方法
 *
 * @author zhaowhf
 * @time 2016年7月18日18:25:23
 */
@Service
public class BaseServiceImpl<E extends BaseIdEntity, V extends SuperVO> implements IBaseService<E, V> {
	private final Logger logger = LoggerFactory.getLogger(BaseServiceImpl.class);

	public static final int READ = 0;
	public static final int ADD = 1;
	public static final int UPDATE = 2;
	public static final int DELETE = 3;

	@Autowired
	protected BaseQueryDao queryDao;

	/** 每个业务必须有继承BaseDao并且泛型为业务实体的dao，否则此处会报错 */
	@Autowired
	private BaseDao<E> baseDao;

	public BaseServiceImpl() {
	}

	public BaseDao<E> getBaseDao() {
		return baseDao;
	}

	public void setBaseDao(BaseDao<E> baseDao) {
		this.baseDao = baseDao;
	}

	/** 保存动作前处理 */
	protected E beforeSave(V vo, Class<E> clazzE, Class<V> clazzV) throws BusinessException, Exception {
		E entity = handleSingleV2E(vo, clazzE);// 主子表关系暂时由各业务service自行处理
		return entity;
	}

	@Override
	@Transactional
	public V save(V vo, Class<E> clazzE, Class<V> clazzV) throws BusinessException, Exception {
		E entity = beforeSave(vo, clazzE, clazzV);
//		E entity = handleSingleV2E(vo, clazzE);
		entity = saveEntity(entity, clazzE, clazzV);
//		V returnVO = handleSingleE2V(entity, clazzV);
		V returnVO = afterSave(entity, clazzE, clazzV);
		return returnVO;
	}

	@Transactional
	public V saveEntityReturnVO(E entity, Class<E> clazzE, Class<V> clazzV) throws BusinessException, Exception {
		saveEntity(entity, clazzE, clazzV);
		V returnVO = afterSave(entity, clazzE, clazzV);
		return returnVO;
	}

	@Transactional
	public E saveEntity(E entity, Class<E> clazzE, Class<V> clazzV) throws BusinessException, Exception {
		if (entity.getId() == null && entity instanceof SuperBillMainEntity) {
			((SuperBillMainEntity) entity).setCreatetime(new Timestamp(System.currentTimeMillis()));
			((SuperBillMainEntity) entity).setModifytime(new Timestamp(System.currentTimeMillis()));
		}
		if (entity.getId() != null && entity instanceof SuperBillMainEntity) {
			((SuperBillMainEntity) entity).setModifytime(new Timestamp(System.currentTimeMillis()));
		}
		entity = baseDao.save(entity);
		return entity;
	}

	@Override
	@Transactional
	public void saveBatch(List<V> vos, Class<E> clazzE, Class<V> clazzV) throws BusinessException, Exception {
//		baseDao.save(handleMultiV2E(vos, clazzE));
		for (int i = 0; i < vos.size(); i++) {
			save(vos.get(i), clazzE, clazzV);
		}
	}

	/** 保存动作后处理 */
	protected V afterSave(E entity, Class<E> clazzE, Class<V> clazzV) throws BusinessException, Exception {
		V returnVO = handleSingleE2V(entity, clazzV);// 主子表关系暂时由各业务service自行处理
		return returnVO;
	}

	/** 删除动作前处理 */
	protected void beforeDelete() {
		
	}

	@Override
	@Transactional
	public void delete(V vo) throws Exception {
		delete(vo.getId());
	}

	@Override
	@Transactional
	public void delete(String id) throws Exception {
		beforeDelete();
		E entity = findEntityById(id);
		baseDao.delete(entity);
		afterDelete();
	}

	@Override
	@Transactional
	public void deleteBatch(List<V> vos) throws Exception {
		for (int i = 0; i < vos.size(); i++) {
			delete(vos.get(i));
		}
	}

	@Override
	@Transactional
	public void deleteBatch(String[] ids) throws Exception {
		for (int i = 0; i < ids.length; i++) {
			delete(ids[i]);
		}
	}

	/** 删除动作后处理 */
	protected void afterDelete() {
		
	}

	@Override
	public List<V> findByCondition(final Map<String, Object> searchParams, Class<V> clazzV, Class<E> clazzE) throws Exception {
		Specification<E> spec = buildSpecification(searchParams, clazzE);
		List<V> list = handleMultiE2V(baseDao.findAll(spec), clazzV);
		return list;
	}

	@Override
	public Page<V> queryPage(final Map<String, Object> searchParams, Pageable pageable, Class<V> clazzV, Class<E> clazzE) throws Exception {
		Specification<E> spec = buildSpecification(searchParams, clazzE);
		Page<E> page = baseDao.findAll(spec, pageable);
//		Page<E> page =  baseDao.findAll(new Specification<E>() {
//			@Override
//			public Predicate toPredicate(Root<E> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
//				Predicate criteria = cb.conjunction();
//				criteria = cb.and(criteria, cb.equal(root.get("dr"), 0));
//				query.where(criteria);
//				return query.getRestriction();
//			}
//		}, pageable);
		return handleMultiPage(page, clazzV);
	}

	/**
	 * 构建动态查询条件对象
	 * 
	 * @param searchParams 查询参数映射
	 * @return 条件查询对象
	 */
	private Specification<E> buildSpecification(Map<String, Object> searchParams, Class<E> clazzE) {
		Map<String, SearchFilter> filters = SearchFilter.parse(searchParams);
		Specification<E> spec = DynamicSpecifications.bySearchFilter(filters.values(), clazzE);
		return spec;
	}
	
	/**
	 * 处理多行查询数据结果
	* @Title: handleMultiPage 
	* @Description: 
	* @param it 查询结果列表
	* @param cls VO类型
	* @return VO列表
	* @throws 
	 */
	protected Page<V> handleMultiPage(Page<E> it, Class<V> cls) throws Exception {
		List<V> l = new ArrayList<V>();
		if (it == null) {
			return new PageImpl<V>(l);
		}
		for (E entity : it.getContent()) {
			V vo = cls.newInstance();
			BeanUtils.copyProperties(entity, vo);
			l.add(vo);
		}
		Page<V> pageV = new PageImpl<V>(l);
		return pageV;
	}

	/**
	 * 处理多行查询数据结果
	* @Title: handleMulti 
	* @Description: 
	* @param it 查询结果列表
	* @param cls VO类型
	* @return VO列表
	* @throws 
	 */
	protected List<E> handleMultiV2E(Iterable<V> it, Class<E> cls) throws IllegalAccessException, InvocationTargetException, InstantiationException{
		List<E> l = new ArrayList<E>();
		if(it == null){
			return l;
		}
		for (V vo : it) {
			E entity = cls.newInstance();
			BeanUtils.copyProperties(vo, entity);
			l.add(entity);
		}
		return l;
	}

	/**
	 * 处理多行查询数据结果
	* @Title: handleMulti 
	* @Description: 
	* @param it 查询结果列表
	* @param cls VO类型
	* @return VO列表
	* @throws 
	 */
	protected List<V> handleMultiE2V(Iterable<E> it, Class<V> cls) throws IllegalAccessException, InvocationTargetException, InstantiationException{
		List<V> l = new ArrayList<V>();
		if(it == null){
			return l;
		}
		for (E entity : it) {
			V vo = cls.newInstance();
			BeanUtils.copyProperties(entity, vo);
			l.add(vo);
		}
		return l;
	}
	
	/**
	 * 处理单行查询数据结果
	* @Title: handleSingle 
	* @Description: 
	* @param e 实体
	* @param cls VO类型
	* @return 
	* @throws 
	 */
	protected V handleSingleE2V(E e, Class<V> cls) throws IllegalAccessException, InvocationTargetException, InstantiationException{
		if(e == null)
			return null;
		V vo = cls.newInstance();
		BeanUtils.copyProperties(e, vo, null, "parent");
		return vo;
	}
	
	/**
	 * 处理单行查询数据结果
	* @Title: handleSingle 
	* @Description: 
	* @param e 实体
	* @param cls VO类型
	* @return 
	* @throws 
	 */
	protected E handleSingleV2E(V vo, Class<E> cls) throws IllegalAccessException, InvocationTargetException, InstantiationException{
		if(vo == null)
			return null;
		E entity = cls.newInstance();
		BeanUtils.copyProperties(vo, entity);
		return entity;
	}
	
	/**
	 * 根据id查询实体并处理结果
	* @Title: findById 
	* @Description: 
	* @param cls VO类型
	* @param id 主键
	* @return VO列表
	 * @throws NoSuchMethodException 
	* @throws 
	 */
	public V findVOById(final String id, Class<V> cls) throws Exception {
		E entity = (E) baseDao.findOne(new Specification<E>() {
			@Override
			public Predicate toPredicate(Root<E> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
				return cb.and(cb.equal(root.get("dr").as(Integer.class), 0),
						cb.equal(root.get("id").as(String.class), id));
			}
		});
		// return handleSingleE2V(entity, cls);
		if (entity == null) {
			throw new BusinessException("查询失败，数据已被删除，或丢失！");
		}
		V vo = cls.newInstance();
		handleSingleE2V(entity, vo);//解决copy错误
		return vo;
	}

	/**
	 * 如果实体为主子关系，且子表实体中的主表实体属性名为parent，可以不用重写此类；否则反之
	 * @param entity
	 * @return
	 */
	@Deprecated
	protected V handleSingleE2V(E entity) {
		return null;
	}

	/**
	 * 如果实体为主子关系，且子表实体中的主表实体属性名为parent，可以不用重写此类；否则反之
	 * @param entity
	 * @return
	 */
	protected V handleSingleE2V(E entity, V vo) throws Exception {
		SimpleBeanUtils.copyProperties(entity, vo, new String[] { "parent" });
		return vo;
	}

	/**
	 * 根据id查询实体
	 * @Title: findById 
	 * @Description: 
	 * @param id 主键
	 * @return E 实体
	 * @throws 
	 */
	protected E findEntityById(final String id) throws Exception {
		E entity = (E) baseDao.findOne(new Specification<E>() {
			@Override
			public Predicate toPredicate(Root<E> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
				return cb.and(cb.equal(root.get("dr").as(Integer.class), 0), cb.equal(root.get("id").as(String.class), id));
			}
		});
		return entity;
	}

	/**
	 * 根据id查询实体
	 * @Title: findById 
	 * @Description: 
	 * @param id 主键
	 * @return E 实体
	 * @throws 
	 */
	public E findEntityById(final String id, Class<E> cls) throws Exception {
		E entity = (E) baseDao.findOne(new Specification<E>() {
			@Override
			public Predicate toPredicate(Root<E> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
				return cb.and(cb.equal(root.get("dr").as(Integer.class), 0), cb.equal(root.get("id").as(String.class), id));
			}
		});
		return entity;
	}

	/**
	 * 查询所有实体，并处理结果
	* @Title: findAll 
	* @Description: 
	* @param cls VO类型
	* @return VO列表
	* @throws 
	 */
	public List<V> findAll(Class<V> cls) throws IllegalAccessException, InvocationTargetException, InstantiationException{
		Iterable<E> entity = baseDao.findAll(new Specification<E>() {
			@Override
			public Predicate toPredicate(Root<E> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
				return cb.equal(root.get("dr").as(Integer.class), 0);
			}
		});
		return handleMultiE2V(entity, cls);
	}

	/**
	 * like trim()
	 * null返回空字符串
	 *
	 * @Title: trimString 
	 * @Description: 
	 * @param 
	 * @return 
	 * @throws 
	 */
	protected static String trimString(String s) {
		return s == null ? "" : s.trim();
	}

	@Override
	@Transactional
	public void updateApprovalState(BpmStateVO vo, Class<E> cls){
		if(vo == null )
			throw new NullPointerException("BpmStateVO不能为空");
		if(StringUtils.isBlank(vo.getBillId()) ){
			throw new NullPointerException("单据ID不能为空");
		}
		if(vo.getBillState() == null ){
			throw new NullPointerException("单据状态不能为空");
		}
		E e = baseDao.findOne(vo.getBillId());
		if(e == null || e.getDr() != 0){
			throw new RuntimeException("未找到当前单据，或当前单据已被删除");
		}
		SuperBillMainEntity supEty = (SuperBillMainEntity)e;
		supEty.setBillState(vo.getBillState());
		supEty.setReviewer(vo.getReviewer());
		supEty.setReviewerid(vo.getReviewerid());
		supEty.setReviewtime(vo.getReviewtime());
		baseDao.save(e);
	}

	public List<? extends SuperBillSubVO> deleteSubVOs(List<? extends SuperBillSubVO> vos, Class<? extends SuperBillSubEntity> clazz) throws Exception {
		if (vos.size() <= 0) {
			return vos;
		}
		List<SuperBillSubVO> returnVO = new ArrayList<SuperBillSubVO>();
		Table table = clazz.getAnnotation(javax.persistence.Table.class);
		String tableName = table.name();
		String ids = "";
		for (int i = 0; i < vos.size(); i++) {
			SuperBillSubVO superBillSubVO = vos.get(i);
			if (superBillSubVO.getVostate() == DELETE) {
				if (ids.equals("")) {
					ids += "'" + superBillSubVO.getId() + "'";
				} else {
					ids += ",'" + superBillSubVO.getId() + "'";
				}
			} else {
				returnVO.add(superBillSubVO);
			}
		}
		if (!ids.equals("")){
			String deleteSql = "delete from " + tableName + " where id in (" + ids + ")";
			queryDao.executeUpdateSQL(deleteSql);
		}
		return returnVO;
	}

	/**
	 * 解析高级筛选条件
	 * @param paras
	 * @return
	 */
	public String getSqlByAdvancedvalue(Map<String, Object> paras){
		return getSqlByAdvancedvalue(paras,"");
	}
	/**
	 * 解析高级筛选条件
	 * @param paras
	 * @return
	 */
	public String getSqlByAdvancedvalue(Map<String, Object> paras, String fieldBefore){
		Map<String, String> sqlAdvanced = new HashMap<String, String>();
				
		String stradvancedValue = paras.get("advancedValue")==null?"": paras.get("advancedValue").toString();
		if(StringUtils.isEmpty(stradvancedValue)) 
			return "";
		JSONObject JsonAdvanced = (JSONObject) JSONObject.parse(stradvancedValue);
		
		for(Entry<String,Object> entry:JsonAdvanced.entrySet()){ 
            String fiedName = entry.getKey();              
            
            String sql = getSqlAdvancedByField(fiedName, JsonAdvanced, sqlAdvanced, fieldBefore);
            if(!StringUtils.isEmpty(sql))
            	sqlAdvanced.put(fiedName, sql);
        }
		afterGetAdvancedSql(paras, sqlAdvanced);
		
		StringBuilder sqltotal = new StringBuilder();
        for(Map.Entry<String, String> entry:sqlAdvanced.entrySet()){   
        	if(sqltotal.length()>0)
        		sqltotal.append(" and ");
        	sqltotal.append(entry.getValue());
        }
        return sqltotal.toString();
	}
	
	/**
	 * 解析高级筛选条件后统一处理
	 * @param paras
	 * @param sqlAdvanced
	 */
	public void afterGetAdvancedSql(Map<String, Object> paras, Map<String, String> sqlAdvanced){
		
	}
	
	/**
	 * 解析高级筛选条件， 单字段，特殊处理可重写此方法
	 * @param fieldName
	 * @param JsonAdvanced
	 * @param sqlAdvanced
	 * @return
	 */
	public String getSqlAdvancedByField(String fieldName, JSONObject JsonAdvanced, Map<String, String> sqlAdvanced, String fieldBefore){
		StringBuilder sql = new StringBuilder();
		JSONObject jsonValue = (JSONObject) JSONObject.parse(JsonAdvanced.getString(fieldName));  		
		String type = jsonValue.getString("type");    //条件类型
		String opt = jsonValue.getString("opt");    //操作符
		String hiddenValue = jsonValue.getString("hiddenValue");  
		if(StringUtils.isEmpty(opt) && StringUtils.isEmpty(hiddenValue)){
			logger.error("高级筛选条件字段值为空，字段名：" + fieldName);
			return null;
		}
		
		if(type.equals("refer")){  //参照
			String[] lstInValue = hiddenValue.split(",");			
			sql.append(fieldBefore + fieldName + " in (");
			sql.append(AdvanceSearchUtil.sqlUnionByList(fieldName, "in", lstInValue, fieldBefore));
			sql.append(")");			
		}else if(type.equals("string")){  //字符串
			if(StringUtils.isEmpty(opt)){  //无操作符，即为枚举选择
				String[] lstInValue = hiddenValue.split(",");	
				sql.append("(");
				sql.append(AdvanceSearchUtil.sqlUnionByList(fieldName, "like", lstInValue, fieldBefore));
				sql.append(")");		
			}else{
				sql.append(AdvanceSearchUtil.sqlUnionByopt(fieldName, opt , hiddenValue, fieldBefore));
			}
		}else if(type.equals("date")){  //日期
			if(StringUtils.isEmpty(opt)){  //无操作符，即为枚举选择
				String[] lstInValue = hiddenValue.split(",");	
				List<String> inValues = new ArrayList<String>();
				List<String> likeValues = new ArrayList<String>();
				for(String v : lstInValue){
					if(v.length()>8)
						inValues.add(v);
					else
						likeValues.add(v);						
				}
				if(likeValues.size()>0)
					sql.append(AdvanceSearchUtil.sqlUnionByList(fieldName, "like", likeValues.toArray(new String[0]), fieldBefore));
				if(inValues.size()>0){
					if(sql.length()>0)
						sql.append(" or ");
					sql.append(AdvanceSearchUtil.sqlUnionByList(fieldName, "=", inValues.toArray(new String[0]), fieldBefore));
				}					
				sql.insert(0,"(");
				sql.append(")");		
			}else{
				sql.append(AdvanceSearchUtil.sqlUnionByopt(fieldName, opt , hiddenValue, fieldBefore));
			}
		}else if(type.equals("date-area")){  //日期期间
			JSONArray lstValues = (JSONArray) JSONArray.parse(hiddenValue);
			for(int i=0; i< lstValues.size(); i++){
				if(i==0){
					sql.append(AdvanceSearchUtil.sqlUnionByopt(fieldName, ">=" , lstValues.getString(i), fieldBefore));
					sql.append(" and ");
				}
				else if(i==1)
					sql.append(AdvanceSearchUtil.sqlUnionByopt(fieldName, "<=" , lstValues.getString(i), fieldBefore));
			}			
		}else if(type.equals("money")){  //金额
			if(StringUtils.isEmpty(opt)){  //无操作符，即为枚举选择
				String[] lstInValue = hiddenValue.split(",");	
				sql.append("(");
				sql.append(AdvanceSearchUtil.sqlUnionByList(fieldName, " ", lstInValue, fieldBefore));
				sql.append(")");	
			}else{
				sql.append(fieldBefore + fieldName);
				sql.append(" " + opt + " ");
				sql.append(hiddenValue.replace("'", "''"));
			}			
		}
		return sql.toString();
	}
	
	
	
	
}
