package com.yyjz.icop.servlet;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFDataFormat;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFDataFormat;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import com.yyjz.icop.report.ExcelMetadata;
import com.yyjz.icop.report.MappingType;

/**
 * Created by yangmiao on 2016/11/14.
 * excel工具类
 */
@SuppressWarnings("Duplicates")
public class ExcelUtils {
    private static final String LABEL_SEPARATOR = "-";
    private static final String LINE_TEXT = "行号";

    public enum Suffix {
        XLS, XLSX
    }

    /**
     * 导出excel文件
     *
     * @param headers  表头
     * @param dataList 对象集合
     * @param props    要导出的对象属性集合
     * @param response response输出流
     * @param fileName 导出文件名
     */
    public static OutputStream exportExcel(List<String> headers, List dataList, List<String> props,
                                           HttpServletResponse response, String fileName) {
        return exportExcel(headers, dataList, props, response, fileName, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param headers  表头
     * @param dataList 对象集合
     * @param props    要导出的对象属性集合
     * @param response response输出流
     * @param fileName 导出文件名
     * @param suffix   excel格式（xls或xlsx）
     */
    public static OutputStream exportExcel(List<String> headers, List dataList, List<String> props,
                                           HttpServletResponse response, String fileName, Suffix suffix) {
        serResponse(response, fileName, suffix);
        try {
            return exportExcel(headers, dataList, props, response.getOutputStream(), suffix);
        } catch (IOException e) {
            throw new RuntimeException("返回excel输出流异常", e);
        }
    }

    /**
     * 导出excel文件
     *
     * @param headers  表头
     * @param dataList 对象集合
     * @param props    要导出的对象属性集合
     * @param out      输出流
     */
    public static OutputStream exportExcel(List<String> headers, List dataList, List<String> props, OutputStream out) {
        return exportExcel(headers, dataList, props, out, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param headers  表头
     * @param dataList 对象集合
     * @param props    要导出的对象属性集合
     * @param out      输出流
     * @param suffix   excel格式（xls或xlsx）
     */
    public static OutputStream exportExcel(List<String> headers, List dataList, List<String> props, OutputStream out,
                                           Suffix suffix) {
        /*
            判断excel格式，创建对应的workbook对象
         */
        Workbook wb = createWorkBook(suffix, null);
        Sheet sheet = wb.createSheet();

        /*
            导出表头
         */
        int rowIndex = 0;
        if (!CollectionUtils.isEmpty(headers)) {
            Row row = sheet.createRow(rowIndex);
            for (int i = 0; i < headers.size(); i++) {
                row.createCell(i).setCellValue(headers.get(i));
            }
            rowIndex++;
        }

        /*
            导出数据
         */
        if (dataList != null) {
            for (int i = 0; i < dataList.size(); i++) {
                Row row = sheet.createRow(rowIndex + i);
                if (props != null) {
                    for (int j = 0; j < props.size(); j++) {
                        Cell cell = row.createCell(j);
                        try {
                            cell.setCellValue(BeanUtils.getProperty(dataList.get(i), props.get(j)));
                        } catch (Exception e) {
                            throw new RuntimeException("数据转换异常", e);
                        }
                    }
                }
            }
        }

        try {
            wb.write(out);
        } catch (IOException e) {
            throw new RuntimeException("写入数据流异常", e);
        }

        close(out, wb);

        return out;
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param response response输出流
     * @param fileName 导出文件名
     * @param metadata 元数据
     */
    public static OutputStream exportExcel(List dataList, HttpServletResponse response, String fileName, ExcelMetadata metadata) {
        return exportExcel(dataList, response, fileName, metadata, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param response response输出流
     * @param fileName 导出文件名
     * @param metadata 元数据
     */
    public static OutputStream exportExcel(List dataList, HttpServletResponse response, String fileName, ExcelMetadata metadata,
                                           Suffix suffix) {
        serResponse(response, fileName, suffix);

        try {
            return exportExcel(dataList, response.getOutputStream(), metadata, suffix);
        } catch (IOException e) {
            throw new RuntimeException("返回excel输出流异常", e);
        }
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param out      输出流
     * @param metadata 元数据
     */
    public static OutputStream exportExcel(List dataList, OutputStream out, ExcelMetadata metadata) {
        return exportExcel(dataList, out, metadata, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param out      输出流
     * @param metadata 元数据
     * @param suffix   excel格式（xls或xlsx）
     */
    public static OutputStream exportExcel(List dataList, OutputStream out, ExcelMetadata metadata, Suffix suffix) {
        /*
            判断excel格式，创建对应的workbook对象
         */
        Workbook wb = createWorkBook(suffix, null);
        Sheet sheet = wb.createSheet();

        //设置CELL格式为文本格式  
        XSSFCellStyle cellStyle2 = null;
        HSSFCellStyle  cellStyle3 = null;
        if (suffix == Suffix.XLSX) {
	        cellStyle2 = (XSSFCellStyle) wb.createCellStyle();  
//	        Font ztFont = wb.createFont();
//	        ztFont.setColor(Font.COLOR_RED);
//	        cellStyle2.setFont(ztFont);
//          cellStyle2.setAlignment(HSSFCellStyle.ALIGN_CENTER);    
//          cellStyle2.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);//上下居中  
	        XSSFDataFormat format = (XSSFDataFormat ) wb.createDataFormat();  
	        cellStyle2.setDataFormat(format.getFormat("@"));//(format.getFormat("#,##0.00"));
        }else if(suffix == Suffix.XLS){
	        cellStyle3 = (HSSFCellStyle) wb.createCellStyle();  
	        HSSFDataFormat  format = (HSSFDataFormat ) wb.createDataFormat();  
	        cellStyle3.setDataFormat(format.getFormat("@"));//format.getFormat("#,##0.00")
        }else {
        	throw new RuntimeException("不支持的excel格式");
        }
        

        
        if (metadata != null) {
            /*
                导出主表表头
            */
            List<String> labels = metadata.getLabels();
            int rowIndex = 0;
            Row headRow = sheet.createRow(rowIndex++);
            for (int i = 0; i < labels.size(); i++) {
                headRow.createCell(i).setCellValue(labels.get(i));
            }

            if (!CollectionUtils.isEmpty(dataList)) {
                /*
                    导出主表数据
                 */
                List<String> properties = metadata.getProperties();
                List<String> childNames = metadata.getChildNames();
                Map<String, Map<Integer, List>> childIndexSubList = new LinkedHashMap<>();
                HashMap<String,Class> hashTab = new HashMap<String,Class>();
                if(null!=dataList&&dataList.size()>0){
                	Field[] classs_fields = dataList.get(0).getClass().getDeclaredFields();
                	if(null!=classs_fields && classs_fields.length>0 ){
                		for(Field f : classs_fields ){
                			hashTab.put(f.getName(), f.getType());
                		}
                	}
                }
                for (int i = 0; i < dataList.size(); i++) {
                    Object data = dataList.get(i);
                    Row row = sheet.createRow(rowIndex++);
                    for (int j = 0; j < properties.size(); j++) {
                        Cell cell = row.createCell(j);
                        Class type = hashTab.get(properties.get(j));
                        if(null!=type&&hashTab.size()>0){
                            if("java.lang.Long".equals(type.getName()) || "java.lang.Integer".equals(type.getName()) 
                            		|| "java.lang.Double".equals(type.getName()) || "java.lang.Short".equals(type.getName())
                            		|| "java.math.BigDecimal".equals(type.getName()) || "java.math.BigInteger".equals(type.getName()) ){
                            	//cell.setCellType(XSSFCell.CELL_TYPE_NUMERIC);
                            	if(suffix == Suffix.XLSX && null!=cellStyle2){
                            		cell.setCellStyle(cellStyle2); 
                            	}else if(suffix == Suffix.XLS && null!=cellStyle3){
                            		cell.setCellStyle(cellStyle3); 
                            	}
								try {
									String val = BeanUtils.getProperty(data, properties.get(j));
									cell.setCellValue(val==null?null:Double.parseDouble(val));
									continue;
								} catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
									 throw new RuntimeException("设置excel单元格单元格值异常", e);
								} 
                            }
                        }
                        
                        try {
                            cell.setCellValue(BeanUtils.getProperty(data, properties.get(j)));
                        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                            throw new RuntimeException("设置excel单元格单元格值异常", e);
                        }
                    }

                    if (!CollectionUtils.isEmpty(childNames)) {
                        for (String childName : childNames) {
                            List subList = getSubList(data, childName);
                            if (!CollectionUtils.isEmpty(subList)) {
                                if (childIndexSubList.containsKey(childName)) {
                                    Map<Integer, List> indexSubList = childIndexSubList.get(childName);
                                    indexSubList.put(i, subList);
                                } else {
                                    Map<Integer, List> indexSubList = new LinkedHashMap<>();
                                    indexSubList.put(i, subList);
                                    childIndexSubList.put(childName, indexSubList);
                                }
                            }
                        }
                    }
                }

                /*
                    导出子表
                 */
                List<ExcelMetadata> children = metadata.getChildren();
                for (Map.Entry<String, Map<Integer, List>> entry : childIndexSubList.entrySet()) {
                    Map<Integer, List> indexSubList = entry.getValue();
                    if (indexSubList != null) {
                        rowIndex++; // 导出空白行

                        ExcelMetadata subMetadata = children.get(childNames.indexOf(entry.getKey()));

                        /*
                            导出子表表头
                        */
                        List<String> subLabels = subMetadata.getLabels();
                        Row subHeadRow = sheet.createRow(rowIndex++);
                        subHeadRow.createCell(0).setCellValue("行号");
                        for (int j = 0; j < subLabels.size(); j++) {
                            subHeadRow.createCell(j + 1).setCellValue(subLabels.get(j));
                        }

                        /*
                            导出子表数据
                         */
                        List<String> subProperties = subMetadata.getProperties();
                        for (Map.Entry<Integer, List> subEntry : indexSubList.entrySet()) {
                            List subList = subEntry.getValue();
                            for (int j = 0; j < subList.size(); j++) {
                                Object data = subList.get(j);
                                Row row = sheet.createRow(rowIndex++);
                                row.createCell(0).setCellValue(subEntry.getKey() + 2);
                                for (int k = 0; k < subProperties.size(); k++) {
                                    Cell cell = row.createCell(k + 1);
                                    try {
                                        cell.setCellValue(BeanUtils.getProperty(data, subProperties.get(k)));
                                    } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                                        throw new RuntimeException("设置excel单元格单元格值异常", e);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } else {
            throw new RuntimeException("excel元数据不能为空!");
        }

        try {
            wb.write(out);
        } catch (IOException e) {
            throw new RuntimeException("写入数据流异常", e);
        }

        close(out, wb);

        return out;
    }

    /**
     * 导入excel数据
     *
     * @param in    输入流
     * @param props 要导入的对象属性集合
     * @param type  对象类型class
     * @param <T>   对象类型
     * @return 对象集合
     */
    public static <T> List<T> importExcel(InputStream in, List<String> props, Class<T> type) {
        return importExcel(in, props, type, Suffix.XLSX);
    }

    /**
     * 导入excel数据
     *
     * @param in     输入流
     * @param props  要导入的对象属性集合
     * @param type   对象类型class
     * @param suffix excel格式（xls或xlsx）
     * @param <T>    对象类型
     * @return 对象集合
     */
    public static <T> List<T> importExcel(InputStream in, List<String> props, Class<T> type, Suffix suffix) {
        Workbook wb = createWorkBook(suffix, in);
        Sheet sheet = wb.getSheetAt(0);

        /*
         * 解析excel，将数据转换成对象集合
         */
        List<T> entities = new ArrayList<>();
        for (int i = 1; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            if (isNullRow(row)) {
                T entity;
                try {
                    entity = type.newInstance();
                } catch (InstantiationException | IllegalAccessException e) {
                    throw new RuntimeException("类型转转异常", e);
                }
                for (int j = 0; j < props.size(); j++) {
                    try {
                        BeanUtils.setProperty(entity, props.get(j), getCellValue(row.getCell(j)));
                    } catch (IllegalAccessException | InvocationTargetException e) {
                        throw new RuntimeException("数据转换异常", e);
                    }
                }
                entities.add(entity);
            }
        }
        return entities;
    }

    /**
     * 导入excel数据
     *
     * @param in       输入流
     * @param metadata 元数据
     * @return 对象集合
     */
    public static List<Object> importExcel(InputStream in, ExcelMetadata metadata) {
        return importExcel(in, metadata, Suffix.XLSX);
    }

    /**
     * 导入excel数据
     *
     * @param in       输入流
     * @param metadata 元数据
     * @param suffix   excel格式（xls或xlsx）
     * @return 对象集合
     */
    public static List<Object> importExcel(InputStream in, ExcelMetadata metadata, Suffix suffix) {
        Workbook wb = createWorkBook(suffix, in);
        Sheet sheet = wb.getSheetAt(0);

        /*
         * 解析excel，将数据转换成对象集合
         */
        // 移除最后面多余的空行
        int lastRow = sheet.getLastRowNum();
        for (int i = sheet.getLastRowNum(); i > 0; i--) {
            if (isNullRow(sheet.getRow(i))) {
                lastRow--;
            } else {
                break;
            }
        }

        List<Integer> nullRowNumbers = new ArrayList<>();
        for (int i = 0; i <= lastRow + 1; i++) {
            if (isNullRow(sheet.getRow(i))) {
                nullRowNumbers.add(i);
            }
        }

        List<Integer> indices = getIndices(metadata.getLabels(), sheet.getRow(0), metadata.getMappingType());

        List<String> childNames = metadata.getChildNames();
        List<Object> models = new ArrayList<>();
        int start = 1;
        for (int i = 0; i < nullRowNumbers.size(); i++) {
            int nullRowNumber = nullRowNumbers.get(i);
            if (i == 0) {
                for (int j = start; j < nullRowNumber; j++) {
                    Row row = sheet.getRow(j);
                    Object model = parseObject(row, metadata.getModel(), metadata.getProperties(), indices);
                    models.add(model);
                }
                start = nullRowNumber + 2;
            } else {
                ExcelMetadata subMetadata = metadata.getChildren().get(i - 1);

                List<Integer> subIndices = getIndices(subMetadata.getLabels(), sheet.getRow(start - 1), subMetadata.getMappingType());

                Map<Integer, List<Object>> indexSubModels = new HashMap<>();
                for (int j = start; j < nullRowNumber; j++) {
                    Row row = sheet.getRow(j);
                    Object subModel = parseObject(row, subMetadata.getModel(), subMetadata.getProperties(), subIndices, 1);

                    int index = (int) row.getCell(0).getNumericCellValue();
                    if (indexSubModels.containsKey(index)) {
                        List<Object> subModels = indexSubModels.get(index);
                        subModels.add(subModel);
                        indexSubModels.put(index, subModels);
                    } else {
                        List<Object> subModels = new ArrayList<>();
                        subModels.add(subModel);
                        indexSubModels.put(index, subModels);
                    }
                }
                for (Map.Entry<Integer, List<Object>> entry : indexSubModels.entrySet()) {
                    try {
                        BeanUtils.setProperty(models.get(entry.getKey() - 2), childNames.get(i - 1), entry.getValue());
                    } catch (IllegalAccessException | InvocationTargetException e) {
                        throw new RuntimeException("解析excel行数据异常", e);
                    }
                }

                start = nullRowNumber + 2;
            }
        }
        return models;
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param response response输出流
     * @param fileName 导出文件名
     * @param metadata 元数据
     */
    public static OutputStream export(List dataList, HttpServletResponse response, String fileName,
                                      ExcelMetadata metadata) {
        return export(dataList, response, fileName, metadata, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param response response输出流
     * @param fileName 导出文件名
     * @param metadata 元数据
     */
    public static OutputStream export(List dataList, HttpServletResponse response, String fileName,
                                      ExcelMetadata metadata, Suffix suffix) {
        serResponse(response, fileName, suffix);

        try {
            return export(dataList, response.getOutputStream(), metadata, suffix);
        } catch (IOException e) {
            throw new RuntimeException("返回excel输出流异常", e);
        }
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param out      输出流
     * @param metadata 元数据
     */
    public static OutputStream export(List dataList, OutputStream out, ExcelMetadata metadata) {
        return export(dataList, out, metadata, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param dataList 数据
     * @param out      输出流
     * @param metadata 元数据
     * @param suffix   excel格式（xls或xlsx）
     */
    public static OutputStream export(List dataList, OutputStream out, ExcelMetadata metadata, ExcelUtils.Suffix suffix) {
        Workbook wb = createWorkBook(suffix, null);

        if (metadata != null) {
            exportSingle(wb, dataList, metadata);
        } else {
            throw new RuntimeException("excel元数据不能为空!");
        }

        try {
            wb.write(out);
        } catch (IOException e) {
            throw new RuntimeException("写入数据流异常", e);
        }

        close(out, wb);

        return out;
    }

    /**
     * 导出excel文件
     *
     * @param dataList     数据集合
     * @param response     response输出流
     * @param fileName     导出文件名
     * @param metadataList 元数据集合
     */
    public static OutputStream export(List<List> dataList, HttpServletResponse response, String fileName,
                                      List<ExcelMetadata> metadataList) {
        return export(dataList, response, fileName, metadataList, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param dataList     数据集合
     * @param response     response输出流
     * @param fileName     导出文件名
     * @param metadataList 元数据集合
     */
    public static OutputStream export(List<List> dataList, HttpServletResponse response, String fileName,
                                      List<ExcelMetadata> metadataList, Suffix suffix) {
        serResponse(response, fileName, suffix);

        try {
            return export(dataList, response.getOutputStream(), metadataList, suffix);
        } catch (IOException e) {
            throw new RuntimeException("返回excel输出流异常", e);
        }
    }

    /**
     * 导出excel文件
     *
     * @param dataList     数据集合
     * @param out          输出流
     * @param metadataList 元数据集合
     */
    public static OutputStream export(List<List> dataList, OutputStream out, List<ExcelMetadata> metadataList) {
        return export(dataList, out, metadataList, Suffix.XLSX);
    }

    /**
     * 导出excel文件
     *
     * @param dataList     数据集合
     * @param out          输出流
     * @param metadataList 元数据集合
     * @param suffix       excel格式（xls或xlsx）
     */
    public static OutputStream export(List<List> dataList, OutputStream out, List<ExcelMetadata> metadataList,
                                      ExcelUtils.Suffix suffix) {
        Workbook wb = createWorkBook(suffix, null);

        if (!CollectionUtils.isEmpty(metadataList)) {
            IntStream.range(0, metadataList.size()).forEach(index -> {
                ExcelMetadata metadata = metadataList.get(index);
                if (metadata != null) {
                    exportSingle(wb, dataList.get(index), metadata);
                } else {
                    throw new RuntimeException("excel元数据不能为空!");
                }
            });
        } else {
            throw new RuntimeException("excel元数据不能为空!");
        }

        try {
            wb.write(out);
        } catch (IOException e) {
            throw new RuntimeException("写入数据流异常", e);
        }

        close(out, wb);

        return out;
    }

    /**
     * 导出单独实体数据
     *
     * @param wb       {@link Workbook}
     * @param dataList 数据集合
     * @param metadata 元数据
     */
    private static void exportSingle(Workbook wb, List dataList, ExcelMetadata metadata) {
        Sheet sheet = metadata.getSheetName() == null ? wb.createSheet() : wb.createSheet(metadata.getSheetName());
        int headerRowNum = exportHeader(sheet, metadata.getLabels());
        exportData(sheet, dataList, metadata.getProperties(), -1);

        List<String> childNames = metadata.getChildNames();
        if (!CollectionUtils.isEmpty(childNames)) {
            List<ExcelMetadata> children = metadata.getChildren();
            IntStream.range(0, childNames.size()).forEach(index -> {
                ExcelMetadata childMetadata = children.get(index);
                Sheet childSheet = childMetadata.getSheetName() == null ? wb.createSheet() :
                        wb.createSheet(childMetadata.getSheetName());
                List<String> childLabels = new ArrayList<>();
                childLabels.add(LINE_TEXT);
                childLabels.addAll(childMetadata.getLabels());
                exportHeader(childSheet, childLabels);
                if (!CollectionUtils.isEmpty(dataList)) {
                    IntStream.range(0, dataList.size()).forEach(dataIndex -> {
                        if (childMetadata.isCollection()) {
                            List childList = getSubList(dataList.get(dataIndex), childNames.get(index));
                            exportData(childSheet, childList, childMetadata.getProperties(), headerRowNum + dataIndex + 1);
                        } else {
                            Object childObj = getSubObj(dataList.get(dataIndex), childNames.get(index));
                            exportData(childSheet, childObj, childMetadata.getProperties(), headerRowNum + dataIndex + 1);
                        }
                    });
                }
            });
        }
    }

    /**
     * 导出数据集合
     *
     * @param sheet      excel sheet
     * @param dataList   数据集合
     * @param properties 导出的属性
     * @param parentNum  父数据行号
     */
    @SuppressWarnings("unchecked")
    private static void exportData(Sheet sheet, List dataList, List<String> properties, int parentNum) {
        if (!CollectionUtils.isEmpty(dataList)) {
            dataList.forEach(data -> exportData(sheet, data, properties, parentNum));
        }
    }

    /**
     * 导出单行数据
     *
     * @param sheet      excel sheet
     * @param data       数据
     * @param properties 导出的属性
     * @param parentNum  父数据行号
     */
    private static void exportData(Sheet sheet, Object data, List<String> properties, int parentNum) {
        if (data != null) {
            Row row = sheet.createRow(sheet.getLastRowNum() + 1);
            if (!CollectionUtils.isEmpty(properties)) {
                if (parentNum != -1) {
                    row.createCell(0).setCellValue(parentNum);
                }
                IntStream.range(0, properties.size()).forEach(cellNum -> {
                    Cell cell = row.createCell(parentNum == -1 ? cellNum : cellNum + 1);
                    try {
                        cell.setCellValue(BeanUtils.getProperty(data, properties.get(cellNum)));
                    } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                        throw new RuntimeException("设置excel单元格单元格值异常", e);
                    }
                });
            }
        }
    }

    /**
     * 导入excel数据
     *
     * @param in       输入流
     * @param metadata 元数据
     * @return 对象集合
     */
    public static List imports(InputStream in, ExcelMetadata metadata) {
        return imports(in, metadata, Suffix.XLSX);
    }

    /**
     * 导入excel数据
     *
     * @param in       输入流
     * @param metadata 元数据
     * @param suffix   excel格式（xls或xlsx）
     * @return 对象集合
     */
    public static List imports(InputStream in, ExcelMetadata metadata, Suffix suffix) {
        Workbook wb = createWorkBook(suffix, in);

        if (metadata != null) {
            return importsSingle(wb, metadata);
        } else {
            throw new RuntimeException("excel元数据不能为空!");
        }
    }

    /**
     * 导入excel数据
     *
     * @param in           输入流
     * @param metadataList 元数据集合
     * @return 对象集合
     */
    public static List<List> imports(InputStream in, List<ExcelMetadata> metadataList) {
        return imports(in, metadataList, Suffix.XLSX);
    }

    /**
     * 导入excel数据
     *
     * @param in           输入流
     * @param metadataList 元数据集合
     * @param suffix       excel格式（xls或xlsx）
     * @return 对象集合
     */
    public static List<List> imports(InputStream in, List<ExcelMetadata> metadataList, Suffix suffix) {
        Workbook wb = createWorkBook(suffix, in);

        if (!CollectionUtils.isEmpty(metadataList)) {
            return metadataList.stream().map(metadata -> importsSingle(wb, metadata)).collect(Collectors.toList());
        } else {
            throw new RuntimeException("excel元数据不能为空!");
        }
    }

    /**
     * 导入单独实体数据
     *
     * @param wb       {@link Workbook}
     * @param metadata 元数据
     */
    private static List importsSingle(Workbook wb, ExcelMetadata metadata) {
        Sheet sheet = wb.getSheet(metadata.getSheetName());
        int headerRowNum = metadata.getHeadRowNum() == null ?
                (metadata.getLabels().stream().anyMatch(label -> label.contains(LABEL_SEPARATOR)) ? 2 : 1) : metadata.getHeadRowNum();

        List<Integer> indices = getIndices(metadata.getLabels(), sheet.getRow(0), metadata.getMappingType());

        List dataList = IntStream.range(headerRowNum, sheet.getLastRowNum() + 1)
                .mapToObj(index -> parseObject(sheet.getRow(index), metadata.getModel(), metadata.getProperties(), indices))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        List<String> childNames = metadata.getChildNames();
        if (!CollectionUtils.isEmpty(childNames)) {
            List<ExcelMetadata> children = metadata.getChildren();
            IntStream.range(0, childNames.size()).forEach(index -> {
                ExcelMetadata childMetadata = children.get(index);
                Sheet childSheet = wb.getSheet(childMetadata.getSheetName());
                int childHeaderRowNum = childMetadata.getHeadRowNum() == null ?
                        (childMetadata.getLabels().stream().anyMatch(label -> label.contains(LABEL_SEPARATOR)) ? 2 : 1) : childMetadata.getHeadRowNum();

                List<Integer> subIndices = getIndices(childMetadata.getLabels(), childSheet.getRow(0), childMetadata.getMappingType());

                final Map<Integer, Object> indexSublist = new HashMap<>();
                IntStream.range(childHeaderRowNum, childSheet.getLastRowNum() + 1).forEach(subIndex -> {
                    Object subObj = parseObject(childSheet.getRow(subIndex), childMetadata.getModel(), childMetadata.getProperties(), subIndices, 1);
                    if (subObj != null) {
                        int parentIndex = Integer.parseInt(String.valueOf(getCellValue(childSheet.getRow(subIndex).getCell(0))));
                        if (childMetadata.isCollection()) {
                            @SuppressWarnings("unchecked")
                            List subList = indexSublist.containsKey(parentIndex) ? (List) indexSublist.get(parentIndex) : new ArrayList<>();
                            //noinspection unchecked
                            subList.add(subObj);
                            indexSublist.put(parentIndex, subList);
                        } else {
                            indexSublist.put(parentIndex, subObj);
                        }
                    }
                });
                indexSublist.forEach((key, value) -> {
                    try {
                        BeanUtils.setProperty(dataList.get(key - headerRowNum - 1), childNames.get(index), value);
                    } catch (IllegalAccessException | InvocationTargetException e) {
                        throw new RuntimeException("解析excel行数据异常", e);
                    }
                });
            });
        }
        return dataList;
    }

    /**
     * 设置输出流
     *
     * @param response 输出流
     * @param fileName 文件名
     * @param suffix   文件格式
     */

    private static void serResponse(HttpServletResponse response, String fileName, Suffix suffix) {
        try {
            fileName = new String(fileName.getBytes("gb2312"), "iso8859-1");
        } catch (UnsupportedEncodingException e) {
            fileName = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        }

        fileName += "." + suffix.name().toLowerCase();

        response.setContentType("application/vnd.ms-excel;charset=UTF-8");
        response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
        response.setCharacterEncoding("utf-8");
    }

    /**
     * 判断excel格式，创建对应的workbook对象
     *
     * @param suffix excel格式（03或07）
     */
    private static Workbook createWorkBook(ExcelUtils.Suffix suffix, InputStream in) {
        Workbook wb;
        if (suffix == Suffix.XLS) {
            try {
                wb = in == null ? new HSSFWorkbook() : new HSSFWorkbook(in);
            } catch (IOException e) {
                throw new RuntimeException("解析文件流异常", e);
            }
        } else if (suffix == Suffix.XLSX) {
            try {
                wb = in == null ? new XSSFWorkbook() : new XSSFWorkbook(in);
            } catch (IOException e) {
                throw new RuntimeException("解析文件流异常", e);
            }
        } else {
            throw new RuntimeException("不支持的excel格式");
        }
        return wb;
    }

    /**
     * 导出数据表头
     *
     * @param sheet  {@link Sheet}
     * @param labels 属性标签集合
     */
    private static int exportHeader(Sheet sheet, List<String> labels) {
        Row headerRow1 = sheet.createRow(0);
        List<Integer> unMergeNumbers = IntStream.range(0, labels.size())
                .filter(index -> !labels.get(index).contains(LABEL_SEPARATOR))
                .mapToObj(index -> index)
                .collect(Collectors.toList());
        if (unMergeNumbers.size() == labels.size()) {
            unMergeNumbers.forEach(index -> headerRow1.createCell(index).setCellValue(labels.get(index)));
        } else {
            Row headerRow2 = sheet.createRow(1);
            unMergeNumbers.forEach(index -> {
                headerRow1.createCell(index).setCellValue(labels.get(index));
                sheet.addMergedRegion(new CellRangeAddress(0, 1, index, index));
            });

            final Map<String, List<Integer>> mergeCells = new LinkedHashMap<>();
            IntStream.range(0, labels.size())
                    .filter(index -> !unMergeNumbers.contains(index))
                    .forEach(index -> {
                        String label = labels.get(index).split(LABEL_SEPARATOR)[0];
                        List<Integer> cellNumbers = mergeCells.containsKey(label) ? mergeCells.get(label) : new ArrayList<>();
                        cellNumbers.add(index);
                        mergeCells.put(label, cellNumbers);
                    });

            mergeCells.entrySet().forEach(entry -> {
                List<Integer> cellNumbers = entry.getValue();
                cellNumbers.forEach(index -> headerRow2.createCell(index).setCellValue(labels.get(index).split(LABEL_SEPARATOR)[1]));

                headerRow1.createCell(cellNumbers.get(0)).setCellValue(entry.getKey());
                if (cellNumbers.size() > 1) {
                    sheet.addMergedRegion(new CellRangeAddress(0, 0, cellNumbers.get(0), cellNumbers.get(cellNumbers.size() - 1)));
                }
            });
        }
        return sheet.getLastRowNum() + 1;
    }

    /**
     * 关闭流
     *
     * @param out 输出流
     * @param wb  {@link Workbook}
     */
    private static void close(OutputStream out, Workbook wb) {
        try {
            wb.close();
            out.flush();
            out.close();
        } catch (IOException e) {
            throw new RuntimeException("流关闭异常", e);
        }
    }

    /**
     * 解析row，获取row对应的对象值
     *
     * @param row        excel行
     * @param modelName  class类名
     * @param properties 属性集合
     * @return row对应的对象值
     */
    private static Object parseObject(Row row, String modelName, List<String> properties, List<Integer> indices) {
        return parseObject(row, modelName, properties, indices, 0);
    }

    /**
     * 解析row，获取row对应的对象值
     *
     * @param row          excel行
     * @param modelName    class类名
     * @param properties   属性集合
     * @param startCellNum 开始列（主子表时，子表第一列为行号，需要排除）
     * @return row对应的对象值
     */
    private static Object parseObject(Row row, String modelName, List<String> properties, List<Integer> indices, int startCellNum) {
        if (isNullRow(row)) {
            return null;
        }

        Object model;
        try {
            Class modelClass = Class.forName(modelName);
            model = modelClass.newInstance();
            System.out.println("indices: " + indices);
            System.out.println("startCellNum: " + startCellNum);
            for (int i = 0; i < properties.size(); i++) {
                System.out.println("cell index: " + (indices == null ? i : indices.get(i) + startCellNum));
                BeanUtils.setProperty(model, properties.get(i), getCellValue(
                        row.getCell(indices == null ? (i + startCellNum): indices.get(i))));
            }
        } catch (ClassNotFoundException | InstantiationException | IllegalAccessException | InvocationTargetException e) {
            throw new RuntimeException("解析excel行数据异常", e);
        }
        return model;
    }

    /**
     * 获取cell单元格的值
     *
     * @param cell 单元格
     * @return 单元格的值
     */
    private static Object getCellValue(Cell cell) {
        if (cell == null) {
            return null;
        }

        switch (CellType.forInt(cell.getCellType())) {
            case STRING: // 字符串
                return cell.getStringCellValue().trim();
            case NUMERIC: // 数值型
                if (HSSFDateUtil.isCellDateFormatted(cell)) { // date类型
                    Date date = HSSFDateUtil.getJavaDate(cell.getNumericCellValue());
                    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    return format.format(date);
                } else { // 纯数字
                    return new DecimalFormat("#.######").format(cell.getNumericCellValue());
                }
                // 布尔类型
            case BOOLEAN:
                return cell.getBooleanCellValue();
            case BLANK: // 空值
                return "";
            case ERROR: // 故障
                return null;
            default:
                return null;
        }
    }

    /**
     * 获取对象指定属性值（集合）
     *
     * @param object   对象
     * @param property 属性
     * @return 属性值
     */
    @SuppressWarnings("unchecked")
    private static List getSubList(Object object, String property) {
        try {
            Class cls = object.getClass();
            Field field = cls.getDeclaredField(property);
            field.setAccessible(true);
            return (List) field.get(object);
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new RuntimeException("获取子对象数据集合异常", e);
        }
    }

    /**
     * 获取对象指定属性值（对象）
     *
     * @param object   对象
     * @param property 属性
     * @return 属性值
     */
    private static Object getSubObj(Object object, String property) {
        try {
            Class cls = object.getClass();
            Field field = cls.getDeclaredField(property);
            field.setAccessible(true);
            return field.get(object);
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new RuntimeException("获取嵌套对象数据异常", e);
        }
    }

    /**
     * 判断是否为空行
     *
     * @param row 行
     * @return boolean
     */
    private static boolean isNullRow(Row row) {
        if (row == null) {
            return true;
        } else {
            int cellNum = row.getLastCellNum();
//            if (cellNum < 2) {
//                return true;
//            }

            for (int i = 0; i < row.getLastCellNum(); i++) {
                Cell cell = row.getCell(i);
                if (cell != null && !String.valueOf(cell).equals("") && CellType.forInt(cell.getCellType()) != CellType.BLANK) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * 获取labels对应于row中的索引
     *
     * @param labels      labels
     * @param headRow     header row
     * @param mappingType 值映射类型
     * @return labels对应于row中的索引
     */
    private static List<Integer> getIndices(List<String> labels, Row headRow, MappingType mappingType) {
        if (MappingType.BY_LABEL.equals(mappingType)) {
            List<String> excelLabels = IntStream.range(0, headRow.getLastCellNum())
                    .mapToObj(i -> headRow.getCell(i).getStringCellValue())
                    .collect(Collectors.toList());
            System.out.println("excelLabels:" + excelLabels);
            System.out.println("labels:" + labels);
            System.out.println("indecis:" + labels.stream().map(excelLabels::indexOf).collect(Collectors.toList()));
            return labels.stream().map(excelLabels::indexOf).collect(Collectors.toList());
        }
        return null;
    }
}
