/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sshd.common.scp;

import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StreamCorruptedException;
import java.nio.charset.StandardCharsets;
import java.nio.file.AccessDeniedException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.sshd.common.SshException;
import org.apache.sshd.common.file.util.MockPath;
import org.apache.sshd.common.scp.LocalFileScpSourceStreamResolver;
import org.apache.sshd.common.scp.LocalFileScpTargetStreamResolver;
import org.apache.sshd.common.scp.ScpReceiveLineHandler;
import org.apache.sshd.common.scp.ScpSourceStreamResolver;
import org.apache.sshd.common.scp.ScpTargetStreamResolver;
import org.apache.sshd.common.scp.ScpTimestamp;
import org.apache.sshd.common.scp.ScpTransferEventListener;
import org.apache.sshd.common.util.DirectoryScanner;
import org.apache.sshd.common.util.GenericUtils;
import org.apache.sshd.common.util.SelectorUtils;
import org.apache.sshd.common.util.io.IoUtils;
import org.apache.sshd.common.util.io.LimitInputStream;
import org.apache.sshd.common.util.logging.AbstractLoggingBean;

public class ScpHelper
extends AbstractLoggingBean {
    public static final String SCP_COMMAND_PREFIX = "scp";
    public static final int OK = 0;
    public static final int WARNING = 1;
    public static final int ERROR = 2;
    public static final int DEFAULT_COPY_BUFFER_SIZE = 8192;
    public static final int DEFAULT_RECEIVE_BUFFER_SIZE = 8192;
    public static final int DEFAULT_SEND_BUFFER_SIZE = 8192;
    public static final int MIN_COPY_BUFFER_SIZE = 127;
    public static final int MIN_RECEIVE_BUFFER_SIZE = 127;
    public static final int MIN_SEND_BUFFER_SIZE = 127;
    public static final int S_IRUSR = 256;
    public static final int S_IWUSR = 128;
    public static final int S_IXUSR = 64;
    public static final int S_IRGRP = 32;
    public static final int S_IWGRP = 16;
    public static final int S_IXGRP = 8;
    public static final int S_IROTH = 4;
    public static final int S_IWOTH = 2;
    public static final int S_IXOTH = 1;
    protected final FileSystem fileSystem;
    protected final InputStream in;
    protected final OutputStream out;
    protected final ScpTransferEventListener listener;

    public ScpHelper(InputStream in, OutputStream out, FileSystem fileSystem, ScpTransferEventListener eventListener) {
        this.in = in;
        this.out = out;
        this.fileSystem = fileSystem;
        this.listener = eventListener == null ? ScpTransferEventListener.EMPTY : eventListener;
    }

    public void receiveFileStream(final OutputStream local, final int bufferSize) throws IOException {
        this.receive(new ScpReceiveLineHandler(){

            @Override
            public void process(final String line, boolean isDir, ScpTimestamp timestamp) throws IOException {
                if (isDir) {
                    throw new StreamCorruptedException("Cannot download a directory into a file stream: " + line);
                }
                final MockPath path = new MockPath(line);
                ScpHelper.this.receiveStream(line, new ScpTargetStreamResolver(){

                    @Override
                    public OutputStream resolveTargetStream(String name, long length, Set<PosixFilePermission> perms) throws IOException {
                        if (ScpHelper.this.log.isDebugEnabled()) {
                            ScpHelper.this.log.debug("resolveTargetStream(" + name + ")[" + perms + "][len=" + length + "] started local stream download");
                        }
                        return local;
                    }

                    @Override
                    public Path getEventListenerFilePath() {
                        return path;
                    }

                    @Override
                    public void postProcessReceivedData(String name, boolean preserve, Set<PosixFilePermission> perms, ScpTimestamp time) throws IOException {
                        if (ScpHelper.this.log.isDebugEnabled()) {
                            ScpHelper.this.log.debug("postProcessReceivedData(" + name + ")[" + perms + "][time=" + time + "] ended local stream download");
                        }
                    }

                    public String toString() {
                        return line;
                    }
                }, timestamp, false, bufferSize);
            }
        });
    }

    public void receive(final Path path, final boolean recursive, boolean shouldBeDir, final boolean preserve, final int bufferSize) throws IOException {
        if (shouldBeDir) {
            LinkOption[] options = IoUtils.getLinkOptions(false);
            Boolean status = IoUtils.checkFileExists(path, options);
            if (status == null) {
                throw new SshException("Target directory " + path + " is most like inaccessible");
            }
            if (!status.booleanValue()) {
                throw new SshException("Target directory " + path + " does not exist");
            }
            if (!Files.isDirectory(path, options)) {
                throw new SshException("Target directory " + path + " is not a directory");
            }
        }
        this.receive(new ScpReceiveLineHandler(){

            @Override
            public void process(String line, boolean isDir, ScpTimestamp time) throws IOException {
                if (recursive && isDir) {
                    ScpHelper.this.receiveDir(line, path, time, preserve, bufferSize);
                } else {
                    ScpHelper.this.receiveFile(line, path, time, preserve, bufferSize);
                }
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void receive(ScpReceiveLineHandler handler) throws IOException {
        this.ack();
        ScpTimestamp time = null;
        block10: while (true) {
            String line;
            boolean isDir = false;
            int c = this.readAck(true);
            switch (c) {
                case -1: {
                    return;
                }
                case 68: {
                    isDir = true;
                    line = String.valueOf((char)c) + this.readLine();
                    this.log.debug("Received header: " + line);
                    break;
                }
                case 67: {
                    line = String.valueOf((char)c) + this.readLine();
                    this.log.debug("Received header: " + line);
                    break;
                }
                case 84: {
                    line = String.valueOf((char)c) + this.readLine();
                    this.log.debug("Received header: " + line);
                    time = ScpTimestamp.parseTime(line);
                    this.ack();
                    continue block10;
                }
                case 69: {
                    line = String.valueOf((char)c) + this.readLine();
                    this.log.debug("Received header: " + line);
                    this.ack();
                    return;
                }
                default: {
                    continue block10;
                }
            }
            try {
                handler.process(line, isDir, time);
                continue;
            }
            finally {
                time = null;
                continue;
            }
            break;
        }
    }

    public void receiveDir(String header, Path path, ScpTimestamp time, boolean preserve, int bufferSize) throws IOException {
        if (this.log.isDebugEnabled()) {
            this.log.debug("Receiving directory {}", (Object)path);
        }
        if (!header.startsWith("D")) {
            throw new IOException("Expected a D message but got '" + header + "'");
        }
        Set<PosixFilePermission> perms = ScpHelper.parseOctalPermissions(header.substring(1, 5));
        int length = Integer.parseInt(header.substring(6, header.indexOf(32, 6)));
        String name = header.substring(header.indexOf(32, 6) + 1);
        if (length != 0) {
            throw new IOException("Expected 0 length for directory but got " + length);
        }
        LinkOption[] options = IoUtils.getLinkOptions(false);
        Boolean status = IoUtils.checkFileExists(path, options);
        if (status == null) {
            throw new AccessDeniedException("Receive directory existence status cannot be determined: " + path);
        }
        Path file = null;
        if (status.booleanValue() && Files.isDirectory(path, options)) {
            String localName = name.replace('/', File.separatorChar);
            file = path.resolve(localName);
        } else if (!status.booleanValue()) {
            Path parent = path.getParent();
            status = IoUtils.checkFileExists(parent, options);
            if (status == null) {
                throw new AccessDeniedException("Receive directory parent (" + parent + ") existence status cannot be determined for " + path);
            }
            if (status.booleanValue() && Files.isDirectory(parent, options)) {
                file = path;
            }
        }
        if (file == null) {
            throw new IOException("Cannot write to " + path);
        }
        status = IoUtils.checkFileExists(file, options);
        if (status == null) {
            throw new AccessDeniedException("Receive directory file existence status cannot be determined: " + file);
        }
        if (!status.booleanValue() || !Files.isDirectory(file, options)) {
            Files.createDirectory(file, new FileAttribute[0]);
        }
        if (preserve) {
            this.updateFileProperties(file, perms, time);
        }
        this.ack();
        time = null;
        try {
            block19: {
                this.listener.startFolderEvent(ScpTransferEventListener.FileOperation.RECEIVE, path, perms);
                while (true) {
                    header = this.readLine();
                    if (this.log.isDebugEnabled()) {
                        this.log.debug("Received header: " + header);
                    }
                    if (header.startsWith("C")) {
                        this.receiveFile(header, file, time, preserve, bufferSize);
                        time = null;
                        continue;
                    }
                    if (header.startsWith("D")) {
                        this.receiveDir(header, file, time, preserve, bufferSize);
                        time = null;
                        continue;
                    }
                    if (header.equals("E")) break block19;
                    if (!header.startsWith("T")) break;
                    time = ScpTimestamp.parseTime(header);
                    this.ack();
                }
                throw new IOException("Unexpected message: '" + header + "'");
            }
            this.ack();
        }
        catch (IOException | RuntimeException e) {
            this.listener.endFolderEvent(ScpTransferEventListener.FileOperation.RECEIVE, path, perms, e);
            throw e;
        }
    }

    public void receiveFile(String header, Path path, ScpTimestamp time, boolean preserve, int bufferSize) throws IOException {
        if (this.log.isDebugEnabled()) {
            this.log.debug("Receiving file {}", (Object)path);
        }
        this.receiveStream(header, new LocalFileScpTargetStreamResolver(path), time, preserve, bufferSize);
    }

    public void receiveStream(String header, ScpTargetStreamResolver resolver, ScpTimestamp time, boolean preserve, int bufferSize) throws IOException {
        int bufSize;
        if (!header.startsWith("C")) {
            throw new IOException("receiveStream(" + resolver + ") Expected a C message but got '" + header + "'");
        }
        if (bufferSize < 127) {
            throw new IOException("receiveStream(" + resolver + ") buffer size (" + bufferSize + ") below minimum (" + 127 + ")");
        }
        Set<PosixFilePermission> perms = ScpHelper.parseOctalPermissions(header.substring(1, 5));
        long length = Long.parseLong(header.substring(6, header.indexOf(32, 6)));
        String name = header.substring(header.indexOf(32, 6) + 1);
        if (length < 0L) {
            this.log.warn("receiveStream(" + resolver + ") bad length in header: " + header);
        }
        if (length == 0L) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("receiveStream(" + resolver + ") zero file size (perhaps special file) using copy buffer size=" + 127);
            }
            bufSize = 127;
        } else {
            bufSize = (int)Math.min(length, (long)bufferSize);
        }
        if (bufSize < 0) {
            this.log.warn("receiveFile(" + resolver + ") bad buffer size (" + bufSize + ") using default (" + 127 + ")");
            bufSize = 127;
        }
        try (LimitInputStream is = new LimitInputStream(this.in, length);
             OutputStream os = resolver.resolveTargetStream(name, length, perms);){
            this.ack();
            Path file = resolver.getEventListenerFilePath();
            try {
                this.listener.startFileEvent(ScpTransferEventListener.FileOperation.RECEIVE, file, length, perms);
                IoUtils.copy(is, os, bufSize);
                this.listener.endFileEvent(ScpTransferEventListener.FileOperation.RECEIVE, file, length, perms, null);
            }
            catch (IOException | RuntimeException e) {
                this.listener.endFileEvent(ScpTransferEventListener.FileOperation.RECEIVE, file, length, perms, e);
                throw e;
            }
        }
        resolver.postProcessReceivedData(name, preserve, perms, time);
        this.ack();
        this.readAck(false);
    }

    protected void updateFileProperties(Path file, Set<PosixFilePermission> perms, ScpTimestamp time) throws IOException {
        if (this.log.isTraceEnabled()) {
            this.log.trace("updateFileProperties(" + file + ") permissions: " + perms);
        }
        IoUtils.setPermissions(file, perms);
        if (time != null) {
            BasicFileAttributeView view = Files.getFileAttributeView(file, BasicFileAttributeView.class, new LinkOption[0]);
            FileTime lastModified = FileTime.from(time.getLastModifiedTime(), TimeUnit.MILLISECONDS);
            FileTime lastAccess = FileTime.from(time.getLastAccessTime(), TimeUnit.MILLISECONDS);
            if (this.log.isTraceEnabled()) {
                this.log.trace("updateFileProperties(" + file + ") last-modified=" + lastModified + ", last-access=" + lastAccess);
            }
            view.setTimes(lastModified, lastAccess, null);
        }
    }

    public String readLine() throws IOException {
        return this.readLine(false);
    }

    public String readLine(boolean canEof) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        int c;
        while ((c = this.in.read()) != 10) {
            if (c == -1) {
                if (!canEof) {
                    throw new EOFException("EOF while await end of line");
                }
                return null;
            }
            baos.write(c);
        }
        return baos.toString();
    }

    public void send(Collection<String> paths, boolean recursive, boolean preserve, int bufferSize) throws IOException {
        this.readAck(false);
        LinkOption[] options = IoUtils.getLinkOptions(false);
        for (String pattern : paths) {
            int idx = (pattern = pattern.replace('/', File.separatorChar)).indexOf(42);
            if (idx >= 0) {
                String[] included;
                String basedir = "";
                String fixedPart = pattern.substring(0, idx);
                int lastSep = fixedPart.lastIndexOf(File.separatorChar);
                if (lastSep >= 0) {
                    basedir = pattern.substring(0, lastSep);
                    pattern = pattern.substring(lastSep + 1);
                }
                for (String path : included = new DirectoryScanner(basedir, pattern).scan()) {
                    Path file = this.resolveLocalPath(basedir, path);
                    if (Files.isRegularFile(file, options)) {
                        this.sendFile(file, preserve, bufferSize);
                        continue;
                    }
                    if (Files.isDirectory(file, options)) {
                        if (!recursive) {
                            this.out.write(1);
                            this.out.write((path.replace(File.separatorChar, '/') + " not a regular file\n").getBytes(StandardCharsets.UTF_8));
                            continue;
                        }
                        this.sendDir(file, preserve, bufferSize);
                        continue;
                    }
                    this.out.write(1);
                    this.out.write((path.replace(File.separatorChar, '/') + " unknown file type\n").getBytes(StandardCharsets.UTF_8));
                }
                continue;
            }
            this.send(this.resolveLocalPath(pattern), recursive, preserve, bufferSize, options);
        }
    }

    public void sendPaths(Collection<? extends Path> paths, boolean recursive, boolean preserve, int bufferSize) throws IOException {
        this.readAck(false);
        LinkOption[] options = IoUtils.getLinkOptions(false);
        for (Path path : paths) {
            this.send(path, recursive, preserve, bufferSize, options);
        }
    }

    protected void send(Path file, boolean recursive, boolean preserve, int bufferSize, LinkOption ... options) throws IOException {
        Boolean status = IoUtils.checkFileExists(file, options);
        if (status == null) {
            throw new AccessDeniedException("Send file existence status cannot be determined: " + file);
        }
        if (!status.booleanValue()) {
            throw new IOException(file + ": no such file or directory");
        }
        if (Files.isRegularFile(file, options)) {
            this.sendFile(file, preserve, bufferSize);
        } else if (Files.isDirectory(file, options)) {
            if (!recursive) {
                throw new IOException(file + " not a regular file");
            }
            this.sendDir(file, preserve, bufferSize);
        } else {
            throw new IOException(file + ": unknown file type");
        }
    }

    public Path resolveLocalPath(String basedir, String subpath) throws IOException {
        if (GenericUtils.isEmpty(basedir)) {
            return this.resolveLocalPath(subpath);
        }
        return this.resolveLocalPath(basedir + File.separator + subpath);
    }

    public Path resolveLocalPath(String commandPath) throws IOException, InvalidPathException {
        String path = SelectorUtils.translateToLocalFileSystemPath(commandPath, File.separatorChar, this.fileSystem);
        Path lcl = this.fileSystem.getPath(path, new String[0]);
        Path abs = lcl.isAbsolute() ? lcl : lcl.toAbsolutePath();
        Path p = abs.normalize();
        if (this.log.isTraceEnabled()) {
            this.log.trace("resolveLocalPath({}) {}", (Object)commandPath, (Object)p);
        }
        return p;
    }

    public void sendFile(Path path, boolean preserve, int bufferSize) throws IOException {
        if (this.log.isDebugEnabled()) {
            this.log.debug("Sending file {}", (Object)path);
        }
        this.sendStream(new LocalFileScpSourceStreamResolver(path), preserve, bufferSize);
    }

    public void sendStream(ScpSourceStreamResolver resolver, boolean preserve, int bufferSize) throws IOException {
        int bufSize;
        if (bufferSize < 127) {
            throw new IOException("sendStream(" + resolver + ") buffer size (" + bufferSize + ") below minimum (" + 127 + ")");
        }
        long fileSize = resolver.getSize();
        if (fileSize <= 0L) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("sendStream(" + resolver + ") unknown file size (" + fileSize + ")  perhaps special file - using copy buffer size=" + 127);
            }
            bufSize = 127;
        } else {
            bufSize = (int)Math.min(fileSize, (long)bufferSize);
        }
        if (bufSize < 0) {
            this.log.warn("sendStream(" + resolver + ") bad buffer size (" + bufSize + ") using default (" + 127 + ")");
            bufSize = 127;
        }
        ScpTimestamp time = resolver.getTimestamp();
        if (preserve && time != null) {
            String cmd = "T" + TimeUnit.MILLISECONDS.toSeconds(time.getLastModifiedTime()) + ' ' + '0' + ' ' + TimeUnit.MILLISECONDS.toSeconds(time.getLastAccessTime()) + ' ' + '0' + '\n';
            this.out.write(cmd.getBytes(StandardCharsets.UTF_8));
            this.out.flush();
            this.readAck(false);
        }
        EnumSet<PosixFilePermission> perms = EnumSet.copyOf(resolver.getPermissions());
        String octalPerms = preserve ? ScpHelper.getOctalPermissions(perms) : "0644";
        String fileName = resolver.getFileName();
        String cmd = "C" + octalPerms + ' ' + fileSize + ' ' + fileName + '\n';
        this.out.write(cmd.getBytes(StandardCharsets.UTF_8));
        this.out.flush();
        this.readAck(false);
        try (InputStream in = resolver.resolveSourceStream();){
            Path path = resolver.getEventListenerFilePath();
            try {
                this.listener.startFileEvent(ScpTransferEventListener.FileOperation.SEND, path, fileSize, perms);
                IoUtils.copy(in, this.out, bufSize);
                this.listener.endFileEvent(ScpTransferEventListener.FileOperation.SEND, path, fileSize, perms, null);
            }
            catch (IOException | RuntimeException e) {
                this.listener.endFileEvent(ScpTransferEventListener.FileOperation.SEND, path, fileSize, perms, e);
                throw e;
            }
        }
        this.ack();
        this.readAck(false);
    }

    public void sendDir(Path path, boolean preserve, int bufferSize) throws IOException {
        if (this.log.isDebugEnabled()) {
            this.log.debug("Sending directory {}", (Object)path);
        }
        BasicFileAttributes basic = Files.getFileAttributeView(path, BasicFileAttributeView.class, new LinkOption[0]).readAttributes();
        if (preserve) {
            this.out.write(("T" + basic.lastModifiedTime().to(TimeUnit.SECONDS) + " " + "0" + " " + basic.lastAccessTime().to(TimeUnit.SECONDS) + " " + "0" + "\n").getBytes(StandardCharsets.UTF_8));
            this.out.flush();
            this.readAck(false);
        }
        LinkOption[] options = IoUtils.getLinkOptions(false);
        Set<PosixFilePermission> perms = IoUtils.getPermissions(path, options);
        this.out.write(("D" + (preserve ? ScpHelper.getOctalPermissions(perms) : "0755") + " " + "0" + " " + path.getFileName().toString() + "\n").getBytes(StandardCharsets.UTF_8));
        this.out.flush();
        this.readAck(false);
        try (DirectoryStream<Path> children = Files.newDirectoryStream(path);){
            this.listener.startFolderEvent(ScpTransferEventListener.FileOperation.SEND, path, perms);
            try {
                for (Path child : children) {
                    if (Files.isRegularFile(child, options)) {
                        this.sendFile(child, preserve, bufferSize);
                        continue;
                    }
                    if (!Files.isDirectory(child, options)) continue;
                    this.sendDir(child, preserve, bufferSize);
                }
                this.listener.endFolderEvent(ScpTransferEventListener.FileOperation.SEND, path, perms, null);
            }
            catch (IOException | RuntimeException e) {
                this.listener.endFolderEvent(ScpTransferEventListener.FileOperation.SEND, path, perms, e);
                throw e;
            }
        }
        this.out.write("E\n".getBytes(StandardCharsets.UTF_8));
        this.out.flush();
        this.readAck(false);
    }

    public static String getOctalPermissions(Path path, LinkOption ... options) throws IOException {
        return ScpHelper.getOctalPermissions(IoUtils.getPermissions(path, options));
    }

    public static String getOctalPermissions(Collection<PosixFilePermission> perms) {
        int pf = 0;
        for (PosixFilePermission p : perms) {
            switch (p) {
                case OWNER_READ: {
                    pf |= 0x100;
                    break;
                }
                case OWNER_WRITE: {
                    pf |= 0x80;
                    break;
                }
                case OWNER_EXECUTE: {
                    pf |= 0x40;
                    break;
                }
                case GROUP_READ: {
                    pf |= 0x20;
                    break;
                }
                case GROUP_WRITE: {
                    pf |= 0x10;
                    break;
                }
                case GROUP_EXECUTE: {
                    pf |= 8;
                    break;
                }
                case OTHERS_READ: {
                    pf |= 4;
                    break;
                }
                case OTHERS_WRITE: {
                    pf |= 2;
                    break;
                }
                case OTHERS_EXECUTE: {
                    pf |= 1;
                    break;
                }
            }
        }
        return String.format("%04o", pf);
    }

    public static Set<PosixFilePermission> setOctalPermissions(Path path, String str) throws IOException {
        Set<PosixFilePermission> perms = ScpHelper.parseOctalPermissions(str);
        IoUtils.setPermissions(path, perms);
        return perms;
    }

    public static Set<PosixFilePermission> parseOctalPermissions(String str) {
        int perms = Integer.parseInt(str, 8);
        EnumSet<PosixFilePermission> p = EnumSet.noneOf(PosixFilePermission.class);
        if ((perms & 0x100) != 0) {
            p.add(PosixFilePermission.OWNER_READ);
        }
        if ((perms & 0x80) != 0) {
            p.add(PosixFilePermission.OWNER_WRITE);
        }
        if ((perms & 0x40) != 0) {
            p.add(PosixFilePermission.OWNER_EXECUTE);
        }
        if ((perms & 0x20) != 0) {
            p.add(PosixFilePermission.GROUP_READ);
        }
        if ((perms & 0x10) != 0) {
            p.add(PosixFilePermission.GROUP_WRITE);
        }
        if ((perms & 8) != 0) {
            p.add(PosixFilePermission.GROUP_EXECUTE);
        }
        if ((perms & 4) != 0) {
            p.add(PosixFilePermission.OTHERS_READ);
        }
        if ((perms & 2) != 0) {
            p.add(PosixFilePermission.OTHERS_WRITE);
        }
        if ((perms & 1) != 0) {
            p.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return p;
    }

    public void ack() throws IOException {
        this.out.write(0);
        this.out.flush();
    }

    public int readAck(boolean canEof) throws IOException {
        int c = this.in.read();
        switch (c) {
            case -1: {
                if (canEof) break;
                throw new EOFException("readAck - EOF before ACK");
            }
            case 0: {
                break;
            }
            case 1: {
                this.log.warn("Received warning: " + this.readLine());
                break;
            }
            case 2: {
                throw new IOException("Received nack: " + this.readLine());
            }
        }
        return c;
    }
}

