package com.ejianc.business.voucher.transfer;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.accplat.bean.BillAccbookSetEntity;
import com.ejianc.business.accplat.bean.BillAuxiliarySetEntity;
import com.ejianc.business.accplat.bean.BillInfluenceSetEntity;
import com.ejianc.business.accplat.service.IBillAccbookSetService;
import com.ejianc.business.accplat.service.IBillAuxiliarySetService;
import com.ejianc.business.accplat.service.IBillInfluenceSetService;
import com.ejianc.business.accplat.vo.BillAccbookSetVO;
import com.ejianc.business.accplat.vo.BillAuxiliarySetVO;
import com.ejianc.business.accplat.vo.BillInfluenceSetVO;
import com.ejianc.business.accplat.vo.VoucherTemplateVO;
import com.ejianc.business.voucher.service.IFinanceVoucherService;
import com.ejianc.business.voucher.service.VoucherTemplateCacheProvider;
import com.ejianc.business.voucher.vo.VoucherParams;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @description: 公共凭证转化服务
 *  * 1、依据凭证模板转化原始凭证
 *  * 2、依据原始凭证转化财务凭证
 * @author songlx
 * @date 2023/11/28
 * @version 1.0
 */
@Slf4j
public class BaseVoucherTransferService {

    @Autowired
    private VoucherServiceFactory voucherServiceFactory;

    @Autowired
    private VoucherTemplateCacheProvider voucherTemplateCacheProvider;

    @Autowired
    private IBillAccbookSetService billAccbookSetService;

    @Autowired
    private IBillInfluenceSetService billInfluenceSetService;

    @Autowired
    private IBillAuxiliarySetService billAuxiliarySetService;

    @Autowired
    private IOrgApi orgApi;


    /**
     * @description: 获取辅助核算项关联
     *
     * @param voucherParams
     * @return {@link List< BillAuxiliarySetVO>}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillAuxiliarySetVO> getBillAuxiliarySet(VoucherParams voucherParams) {
        String billTypeCode = voucherParams.getBillTypeCode();
        LambdaQueryWrapper<BillAuxiliarySetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillAuxiliarySetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillAuxiliarySetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BillAuxiliarySetEntity> list = billAuxiliarySetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【辅助核算项关联】");
        }
        return BeanMapper.mapList(list, BillAuxiliarySetVO.class);
    }


    /**
     * @description: 获取影响因素关联
     *
     * @param voucherParams
     * @return {@link List< BillInfluenceSetVO>}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillInfluenceSetVO> getBillInfluenceSet(VoucherParams voucherParams) {
        String billTypeCode = voucherParams.getBillTypeCode();
        LambdaQueryWrapper<BillInfluenceSetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillInfluenceSetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillInfluenceSetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BillInfluenceSetEntity> list = billInfluenceSetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【影响因素关联】");
        }
        return BeanMapper.mapList(list, BillInfluenceSetVO.class);
    }


    /**
     * @description: 获取账簿设置
     *
     * @param voucherParams
     * @return {@link List< BillAccbookSetVO>}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillAccbookSetVO> getBillAccbookSet(VoucherParams voucherParams) {
        String billTypeCode = voucherParams.getBillTypeCode();
        LambdaQueryWrapper<BillAccbookSetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillAccbookSetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillAccbookSetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BillAccbookSetEntity> list = billAccbookSetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【账簿映射规则】");
        }
        return BeanMapper.mapList(list, BillAccbookSetVO.class);
    }


    /**
     * @description: 获取当前组织的账簿，递归最近上级获取
     *
     * @param orgId
     * @param billTypeCode
     * @return {@link BillAccbookSetVO}
     * @author songlx
     * @date: 2023/12/18
     */
    protected BillAccbookSetVO getBillAccbookSet(Long orgId, String billTypeCode) {
        CommonResponse<OrgVO> oneById = orgApi.getOneById(orgId);
        OrgVO orgVO = oneById.getData();
        String[] parentOrgIds = orgVO.getInnerCode().split("\\|");

        LambdaQueryWrapper<BillAccbookSetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillAccbookSetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillAccbookSetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        wrapper.in(BillAccbookSetEntity::getSourceOrgId, parentOrgIds);
        List<BillAccbookSetEntity> list = billAccbookSetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【账簿映射规则】, orgId:" + orgId);
        }
        Map<Long, BillAccbookSetEntity> setMap = list.stream().collect(Collectors.toMap(BillAccbookSetEntity::getSourceOrgId, Function.identity(), (key1, key2) -> key2));
        BillAccbookSetVO billAccbookSetVO = null;
        for (int i = parentOrgIds.length - 1; i >= 0; i--) {
            BillAccbookSetEntity billAccbookSetEntity = setMap.get(Long.valueOf(parentOrgIds[i]));
            if(billAccbookSetEntity != null){
                billAccbookSetVO = BeanMapper.map(billAccbookSetEntity, BillAccbookSetVO.class);
                break;
            }
        }
        return billAccbookSetVO;
    }


    /**
     * @description: 转化方法
     * @author songlx
     * @date: 2023/11/29
     */
    public void transfer(VoucherParams voucherParams) {
        voucherParams.setIsGenerateOriginVoucherFlag(getIsGenerateOriginVoucherFlag(voucherParams));
        voucherParams.setIsGenerateFinanceVoucherFlag(getIsGenerateFinanceVoucherFlag(voucherParams));
        if (voucherParams.getIsGenerateOriginVoucherFlag()) {
            transferOriginVoucher(voucherParams);
        }
        if (voucherParams.getIsGenerateFinanceVoucherFlag()) {
            transferFinanceVoucher(voucherParams);
        }
    }


    /**
     * @description: 转化原始凭证，子类复写，有以下两种情况
     * 1、公共CommonVoucherTransferService 根据凭证模板转化原始凭证，可以直接调用transferOriginVoucherByVoucherTemplate
     * 2、自定义转化服务（如SubSettleVoucherCustomTransferService）可以根据定义分录转化原始凭证，
     *    如不需要复写getIsGenerateOriginVoucherFlag返回false
     * @param voucherParams
     * @return
     * @author songlx
     * @date: 2023/12/1
     */
    public void transferOriginVoucher(VoucherParams voucherParams) {
        this.transferOriginVoucherByVoucherTemplate(voucherParams);
    }


    /**
     * @description: 转化财务凭证，子类复写，有以下两种情况
     * 1、公共CommonVoucherTransferService 根据原始凭证转化财务凭证，可以直接调用transferFinanceVoucherByOriginVoucher
     * 2、自定义转化服务（如SubSettleVoucherCustomTransferService）可以根据定义分录直接转化财务凭证（或者自定义也转化了原始凭证，可以直接调用transferFinanceVoucherByOriginVoucher），
     *    如不需要复写getIsGenerateFinanceVoucherFlag返回false
     * @param voucherParams
     * @return
     * @author songlx
     * @date: 2023/12/1
     */
    public void transferFinanceVoucher(VoucherParams voucherParams) {
        this.transferFinanceVoucherByOriginVoucher(voucherParams);
    }


    /**
     * @description: 依据凭证模板转化原始凭证
     * @author songlx
     * @date: 2023/11/29
     */
    private void transferOriginVoucherByVoucherTemplate(VoucherParams voucherParams) {
        VoucherTemplateVO voucherTemplateVO = voucherTemplateCacheProvider.loadTemplateFromCache(InvocationInfoProxy.getTenantid(), voucherParams.getBillTypeCode());
        voucherParams.setOriginVoucher(null);
    }


    /**
     * @description: 依据原始凭证转化财务凭证
     *
     * @author songlx
     * @date: 2023/11/29
     */
    public void transferFinanceVoucherByOriginVoucher(VoucherParams voucherParams) {
        String systemCode = voucherParams.getSystemCode();
        IFinanceVoucherService financeVoucherService = voucherServiceFactory.getFinanceVoucherService(systemCode);
        financeVoucherService.convertToFinanceVoucherByOriginVoucher(voucherParams);
    }


    /**
     * @description: 是否生成原始凭证，如为true，请在transferOriginVoucher中转化
     * @author songlx
     * @date: 2023/11/29
     */
    public Boolean getIsGenerateOriginVoucherFlag(VoucherParams voucherParams) {
        return Boolean.TRUE;
    }


    /**
     * @description: 是否生成财务凭证，如为true，请在transferFinanceVoucher中转化
     * 后续可能改成配置参数获取
     * @author songlx
     * @date: 2023/11/29
     */
    public Boolean getIsGenerateFinanceVoucherFlag(VoucherParams voucherParams) {
        return Boolean.TRUE;
    }


}
