package com.ejianc.business.yonyou.base.module.auth.tenant.impl;

import com.ejianc.business.yonyou.base.datacenter.DataCenterUrlProvider;
import com.ejianc.business.yonyou.base.exception.BusinessException;
import com.ejianc.business.yonyou.base.network.OpenApiRequestEncryptor;
import com.ejianc.business.yonyou.base.properties.OpenApiProperties;
import com.ejianc.business.yonyou.base.module.token.OpenApiTokenCacheProvider;
import com.ejianc.business.yonyou.base.module.auth.pojo.OpenApiAccessTokenResponse;
import com.ejianc.business.yonyou.base.module.auth.tenant.TenantAuthProvider;
import com.ejianc.business.yonyou.base.module.common.OpenApiAccessToken;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.TreeMap;

/**
 * 本类主要用于
 *
 * @author liuhaoi
 * @since Created At 2020/5/11 11:00 上午
 */
@Slf4j
@RequiredArgsConstructor
public class TenantAuthProviderImpl implements TenantAuthProvider {

    private static final String URL_TOKEN = "open-auth/selfAppAuth/getAccessToken?signature=%s&appKey=%s&timestamp=%s";

    private final OpenApiProperties properties;
    private final OpenApiRequestEncryptor encryptor;

    private final RestTemplate restTemplate;

    @Resource
    OpenApiTokenCacheProvider cacheProvider;

    @Resource
    DataCenterUrlProvider dataCenterUrlProvider;

    @Override
    public OpenApiAccessToken buildTenantAuthToken() {
        OpenApiAccessToken token = cacheProvider.loadTokenFromCache(dataCenterUrlProvider.queryTenantId());
        if (token != null && token.getExpiredAt() > System.currentTimeMillis()) {
            return token;
        }
        return buildTenantAuthTokenFromRemote();

    }

    public OpenApiAccessToken buildTenantAuthTokenFromRemote() {

        long timestamp = System.currentTimeMillis();
        String appKey = properties.getAppKey();
        Map<String, String> params = new TreeMap<>();
        params.put("appKey", appKey);
        params.put("timestamp", String.valueOf(timestamp));
        String signature = encryptor.signature(params, properties.getAppSecret());

        String url = properties.concatURL(dataCenterUrlProvider.buildTokenUrl(), String.format(URL_TOKEN, signature, appKey, timestamp));

        URI uri;
        try {
            uri = new URI(url);
        } catch (URISyntaxException e) {
            throw new BusinessException("invalid open api uri format", e);
        }

        ResponseEntity<OpenApiAccessTokenResponse> responseEntity = restTemplate
                .getForEntity(uri, OpenApiAccessTokenResponse.class);

        OpenApiAccessTokenResponse body = responseEntity.getBody();
        if (body != null) {
            body.check();
            OpenApiAccessToken accessToken = body.getData().build();
            cacheProvider.saveTokenToCache(dataCenterUrlProvider.queryTenantId(), accessToken);
            return accessToken;
        }

        throw new BusinessException("Unexpected empty response encountered when request open api access token");
    }

    @Override
    public OpenApiAccessToken buildISVAuthToken() {
        return buildTenantAuthToken();
    }

}
