package com.ejianc.business.voucher.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.accplat.config.bean.BillVoucherSetEntity;
import com.ejianc.business.accplat.config.service.IBillVoucherSetService;
import com.ejianc.business.accplat.config.vo.BillVoucherSetVO;
import com.ejianc.business.accplat.originvoucher.vo.OriginVoucherVO;
import com.ejianc.business.voucher.consts.VoucherBusinessConsts;
import com.ejianc.business.voucher.transfer.IVoucherTransferService;
import com.ejianc.business.voucher.transfer.VoucherServiceFactory;
import com.ejianc.business.voucher.utils.DataConvertUtil;
import com.ejianc.business.voucher.vo.VoucherInfo;
import com.ejianc.business.voucher.vo.VoucherParams;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Map;
import java.util.Optional;

/**
 * @description: VoucherBusinessService
 * @author songlx
 * @date 2023/11/28
 * @version 1.0
 */
@Slf4j
@Service
public class VoucherBusinessService {


    @Autowired
    private VoucherServiceFactory voucherServiceFactory;


    @Autowired
    private IBillVoucherSetService billVoucherSetService;

    /**
     * @description: 保存凭证
     * 1、构建凭证参数
     * 2、保存原始凭证，可以根据参数动态判断是否需要保存原始凭证
     * 3、推送财务凭证
     * @param voucherParams
     * @return {@link CommonResponse<VoucherInfo>}
     * @author songlx
     * @date: 2023/11/28
     */
    public CommonResponse<VoucherInfo> save(VoucherParams voucherParams) {
        // 1、构建凭证参数、转化凭证
        String billTypeCode = voucherParams.getBillTypeCode();
        // 1.1 根据tenantid 查询该租户用的什么财务系统
        Long tenantid = InvocationInfoProxy.getTenantid();
        String financeSystemCode = VoucherBusinessConsts.KINGDEE_GALAXY;
        voucherParams.setFinanceSystemCode(financeSystemCode);

        BillTypeVO billTypeVO = voucherParams.getBillTypeVO();
        Map<String, Object> billData = voucherParams.getBillData();
        billData.put("billTypeCode", billTypeVO.getBillCode());
        billData.put("billTypeName", billTypeVO.getBillName());

        // 1.2 根据单据类型编码查询【单据转换设置】在当前租户是根据单据凭证模板转化还是自定义转化规则，转化后生成原始凭证 or 财务凭证
        // ① 根据凭证模板，查询凭证模板转化规则，转化数据
        // ② 无凭证模板，走自定义转化
        BillVoucherSetVO billVoucherSetVO = voucherParams.getBillVoucherSetVO();
        String serviceName = Optional.ofNullable(billVoucherSetVO.getCustomTransferServiceName()).orElse(VoucherBusinessConsts.COMMON_VOUCHER_TRANSFER_SERVICE);
        IVoucherTransferService voucherTransferService = voucherServiceFactory.getVoucherTransferService(serviceName);
        voucherTransferService.transfer(voucherParams);

        // 1.3 校验转化后的参数正确性
        checkVoucherParams(voucherParams);

        // 2、保存原始凭证
        if (voucherParams.getIsGenerateOriginVoucherFlag()) {
            OriginVoucherVO originVoucherVO = voucherParams.getOriginVoucherVO();
        }
        log.info(" ============================= ");
        log.info("金蝶云星空接口save:" + DataConvertUtil.objToString(voucherParams.getFinanceVoucher()));
        log.info(" ============================= ");
        // 3、推送财务凭证
        if (voucherParams.getIsGenerateFinanceVoucherFlag()) {
            return getFinanceVoucherService(financeSystemCode).save(voucherParams);
        }
        return CommonResponse.success();
    }


    /**
     * @description: 查询凭证
     *
     * @param voucherInfo
     * @return {@link CommonResponse< Map< String, Object>>}
     * @author songlx
     * @date: 2023/11/30
     */
    public CommonResponse<Map<String, Object>> get(VoucherInfo voucherInfo) {
        return getFinanceVoucherService(voucherInfo.getFinanceSystemCode()).get(voucherInfo);
    }

    /**
     * @description: 删除凭证
     *
     * @param voucherInfo
     * @return {@link CommonResponse}
     * @author songlx
     * @date: 2023/11/30
     */
    public CommonResponse del(VoucherInfo voucherInfo) {
        return getFinanceVoucherService(voucherInfo.getFinanceSystemCode()).del(voucherInfo);
    }


    /**
     * @description: 根据单据类型编码和单据数据查询单据设置判断是否需要传凭证
     *
     * @param voucherParams
     * @return {@link Boolean}
     * @author songlx
     * @date: 2023/12/1
     */
    public Boolean isGenerateVoucher(VoucherParams voucherParams) {
        String billTypeCode = voucherParams.getBillTypeCode();
        Long tenantid = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<BillVoucherSetEntity> voucherSetEntityLambdaQueryWrapper = new LambdaQueryWrapper<>();
        voucherSetEntityLambdaQueryWrapper.eq(BillVoucherSetEntity::getTenantId, tenantid);
        voucherSetEntityLambdaQueryWrapper.eq(BillVoucherSetEntity::getBillTypeCode, billTypeCode);
        BillVoucherSetEntity voucherSetEntity = billVoucherSetService.getOne(voucherSetEntityLambdaQueryWrapper);
        if (voucherSetEntity != null) {
            voucherParams.setBillVoucherSetVO(BeanMapper.map(voucherSetEntity, BillVoucherSetVO.class));
            return voucherSetEntity.getEnabled().intValue() == 1;
        }
        return Boolean.FALSE;
    }


    /**
     * @description: 根据财务系统标识获取对应的财务会计服务
     * @author songlx
     * @date: 2023/11/30
     */
    private IFinanceVoucherService getFinanceVoucherService(String systemCode) {
        return voucherServiceFactory.getFinanceVoucherService(systemCode);
    }

    private void checkVoucherParams(VoucherParams voucherParams) {
        Boolean isGenerateOriginVoucherFlag = voucherParams.getIsGenerateOriginVoucherFlag();
        Boolean isGenerateFinanceVoucherFlag = voucherParams.getIsGenerateFinanceVoucherFlag();
        OriginVoucherVO originVoucherVO = voucherParams.getOriginVoucherVO();
        Map<String, Object> financeVoucher = voucherParams.getFinanceVoucher();
        if (isGenerateOriginVoucherFlag && originVoucherVO == null) {
            throw new BusinessException("凭证构建参数校验错误：生成原始凭证标识为是，但缺乏原始凭证数据[originVoucher]");
        }
        if (isGenerateFinanceVoucherFlag && financeVoucher == null) {
            throw new BusinessException("凭证构建参数校验错误：生成财务凭证标识为是，缺乏财务凭证参数[financeVoucher]");
        }
    }

}
