package com.ejianc.business.voucher.transfer;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.accplat.config.bean.BillAccbookSetEntity;
import com.ejianc.business.accplat.config.bean.BillAuxiliarySetEntity;
import com.ejianc.business.accplat.config.bean.BillInfluenceSetEntity;
import com.ejianc.business.accplat.config.bean.SubjectEntity;
import com.ejianc.business.accplat.config.service.*;
import com.ejianc.business.accplat.config.vo.*;
import com.ejianc.business.accplat.consts.AccplatConsts;
import com.ejianc.business.accplat.originvoucher.vo.*;
import com.ejianc.business.voucher.service.IFinanceVoucherService;
import com.ejianc.business.voucher.utils.AviatorUtil;
import com.ejianc.business.voucher.utils.DataConvertUtil;
import com.ejianc.business.voucher.vo.VoucherParams;
import com.ejianc.foundation.metadata.api.IMdAttributeApi;
import com.ejianc.foundation.metadata.vo.MdAttributeVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.refer.util.ReferObjectUtil;
import com.googlecode.aviator.AviatorEvaluator;
import com.googlecode.aviator.Expression;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @description: 公共凭证转化服务
 *  * 1、依据凭证模板转化原始凭证
 *  * 2、依据原始凭证转化财务凭证
 * @author songlx
 * @date 2023/11/28
 * @version 1.0
 */
@Slf4j
public class BaseVoucherTransferService {

    @Autowired
    private VoucherServiceFactory voucherServiceFactory;

    @Autowired
    private IBillAccbookSetService billAccbookSetService;

    @Autowired
    private IBillInfluenceSetService billInfluenceSetService;

    @Autowired
    private IBillAuxiliarySetService billAuxiliarySetService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private ISubjectContrastService subjectContrastService;

    @Autowired
    private ISubjectService subjectService;

    @Autowired
    private IMdAttributeApi mdAttributeApi;

    @Autowired
    private CacheManager cacheManager;


    /**
     * @description: 业务表达式统一解析入口
     *
     * @param express
     * @param billData
     * @param billTypeCode
     * @return {@link T}
     * @author songlx
     * @date: 2023/12/29
     */
    public <T> T executeBusinessExpression(String express, Map<String, Object> billData, String billTypeCode) {
        // 疑似存在关联或者子表字段特殊解析处理
        if (express.contains("_") || express.contains(".")) {
            List<String> columnNameList = DataConvertUtil.getColumnNameList(express);
            if (CollectionUtils.isNotEmpty(columnNameList)) {
                List<String> subTableKeyList = new ArrayList<>();
                Map<String, MdAttributeVO> mdAttributeVOMap = null;
                String cacheKey = "mdattr" + billTypeCode;
                if (cacheManager.exists(cacheKey)) {
                    mdAttributeVOMap = cacheManager.get(cacheKey);
                } else {
                    CommonResponse<Map<String, MdAttributeVO>> attrMapRes = mdAttributeApi.getAttrMap(billTypeCode);
                    mdAttributeVOMap = attrMapRes.getData();
                }
                for (String column : columnNameList) {
                    MdAttributeVO mdAttributeVO = mdAttributeVOMap.get(column);
                    if (mdAttributeVO == null) continue;
                    String columnPos = mdAttributeVO.getColumnPos();
                    // 头部关联字段处理
                    if (AccplatConsts.ColumnPos.HR.equals(columnPos)) {
                        this.setRelationEntityValue(column, billData, billTypeCode, mdAttributeVOMap);
                    } else if (AccplatConsts.ColumnPos.B.equals(columnPos)) {
                        // detailList.code
                        String[] subKeys = column.split(".");
                        subTableKeyList.add(subKeys[0]);
                    }
                }
                if (subTableKeyList.size() > 1) {
                    throw new BusinessException("单据类型编码【" + billTypeCode + "】的表达式【" + express + "】中还有至少两个子表键值，不能正常解析数据！");
                }
                if (subTableKeyList.size() == 1) {
                    String subTableKey = subTableKeyList.get(0);
                    String _subTableKey = "subData-" + subTableKey;
                    ArrayList<Map<String, Object>> subDatas = new ArrayList<>();
                    // 是否存在镜像遍历数据
                    if (billData.containsKey(_subTableKey)) {
                        subDatas = (ArrayList<Map<String, Object>>) billData.get(_subTableKey);
                    } else {
                        Object sub = billData.get(subTableKey);
                        if (sub != null) {
                            subDatas = (ArrayList<Map<String, Object>>) sub;
                            // 放入备份镜像子表数据，避免拆分覆盖丢失子表数据
                            billData.put(_subTableKey, subDatas);
                        }
                    }
                    // 拆分遍历单个子表执行表达式结果组成合成表达式
                    int a = 1;
                    StringBuilder subExpress = new StringBuilder();
                    Map<String, Object> subExecResMap = new HashMap<>();
                    for (Map<String, Object> subData : subDatas) {
                        billData.put(subTableKey, subData);
                        Object oneSubExecObj = AviatorUtil.exec(express, billData);
                        String key = "oneSubRes" + a;
                        subExecResMap.put(key, oneSubExecObj);
                        subExpress.append(key);
                        if (a != subDatas.size()) {
                            if (oneSubExecObj instanceof Boolean) {
                                subExpress.append(" && ");
                            } else {
                                // 数字和字符串都可以加
                                subExpress.append(" + ");
                            }
                        }
                        a++;
                    }
                    express = subExpress.toString();
                }
            }
        }
        Object exec = AviatorUtil.exec(express, billData);
        log.info("表达式执行：{}, 结果：{}", express, exec);
        return (T) exec;
    }

    /**
     * @description: 设置关联实体值
     * @return
     * @author songlx
     * @date: 2023/12/28
     */
    private void setRelationEntityValue(String columnExpression, Map<String, Object> billData, String billTypeCode, Map<String, MdAttributeVO> mdAttributeVOMap) {
        // 关联字段格式_projectId.type
        String _headCol = columnExpression.substring(0, columnExpression.indexOf("."));
        String headCol = columnExpression.substring(1, columnExpression.indexOf("."));
        if (!billData.containsKey(_headCol)) {
            MdAttributeVO sourceAttributeVO = mdAttributeVOMap.get(headCol);
            Object execVal = AviatorUtil.exec(headCol, billData);
            CommonResponse<Map<String, Object>> mapCommonResponse = mdAttributeApi.queryRelationEntityValue(billTypeCode, sourceAttributeVO.getEntityId(), sourceAttributeVO.getId(), String.valueOf(execVal), null);
            billData.put(_headCol, mapCommonResponse.getData());
        }
    }


    /**
     * @description: 依据凭证模板转化原始凭证
     * @author songlx
     * @date: 2023/11/29
     */
    public OriginVoucherVO transferOriginVoucherByVoucherTemplate(VoucherParams voucherParams, VoucherTemplateVO voucherTemplateVO) {
        String billTypeCode = voucherParams.getBillTypeCode();
        Map<String, Object> billData = voucherParams.getBillData();

        Long orgId = Long.valueOf(String.valueOf(billData.get(voucherTemplateVO.getSrcBillOrgColumn())));
        Date businessDate = DataConvertUtil.getDate(billData.get(voucherTemplateVO.getBusinessDateColumn()));

        OriginVoucherVO originVoucherVO = new OriginVoucherVO();
        // 1 设置账簿
        BillAccbookSetVO billAccbookSetVO = this.getBillAccbookSet(orgId, billTypeCode);
        log.info("billAccbookSetVO:{}", DataConvertUtil.toPrettyFormat(billAccbookSetVO));
        originVoucherVO.setAccbookId(billAccbookSetVO.getAccbookId());
        originVoucherVO.setAccbookCode(billAccbookSetVO.getAccbookCode());
        originVoucherVO.setAccbookName(billAccbookSetVO.getAccbookName());

        // 2 日期
        originVoucherVO.setBusinessDate(businessDate);

        // 3 分录
        ArrayList<OriginVoucherEntryVO> entryList = new ArrayList<>();

        List<VoucherTemplateDetailVO> voucherTemplateDetailList = voucherTemplateVO.getVoucherTemplateDetailList();
        for (VoucherTemplateDetailVO voucherTemplateDetailVO : voucherTemplateDetailList) {
            String subjectRuleFlag = voucherTemplateDetailVO.getSubjectRuleFlag();
            List<OriginVoucherEntryVO> list = null;
            if (AccplatConsts.SubjectRule.SUBJECT_CONTRAST.equals(subjectRuleFlag)) {
                list = this.analysisEntryBySubjectContrast(billTypeCode, billData, voucherTemplateDetailVO, voucherTemplateVO);
            } else {
                list = this.analysisEntryBySubject(billTypeCode, billData, voucherTemplateDetailVO, voucherTemplateVO);
            }
            entryList.addAll(list);
        }

        // 构建辅助核算项
        this.analysisAuxiliary(billTypeCode, billData, entryList);
        originVoucherVO.setOriginVoucherEntryList(entryList);
        return originVoucherVO;
    }


    /**
     * @description: 根据固定科目构建分录
     *
     * @return {@link List<  OriginVoucherEntryVO >}
     * @author songlx
     * @date: 2023/12/26
     */
    public List<OriginVoucherEntryVO> analysisEntryBySubject(String billTypeCode, Map<String, Object> billData, VoucherTemplateDetailVO voucherTemplateDetailVO, VoucherTemplateVO voucherTemplateVO) {
        String subjectCode = voucherTemplateDetailVO.getSubjectRuleCode();
        String directionFlag = voucherTemplateDetailVO.getDirectionFlag();

        String description = voucherTemplateDetailVO.getDescriptionExpressionView();
        String descriptionExpression = voucherTemplateDetailVO.getDescriptionExpression();
        if (StringUtils.isNotBlank(descriptionExpression)) {
            description = executeBusinessExpression(descriptionExpression, billData, billTypeCode);
        }

        String mnyExpression = voucherTemplateDetailVO.getMnyExpression();
        Object mnyObj = executeBusinessExpression(mnyExpression, billData, billTypeCode);
        BigDecimal mny = ComputeUtil.toBigDecimal(mnyObj);

        log.info("==========解析分录【固定科目编码：" + subjectCode + "】=====start====================================");
        List<OriginVoucherEntryVO> entryVOList = new LinkedList<>();

        SubjectVO subject = this.getSubject(subjectCode);
        OriginVoucherEntryVO originVoucherEntryVO = new OriginVoucherEntryVO();

        originVoucherEntryVO.setEntryDescription(description);
        originVoucherEntryVO.setSubjectId(subject.getId());
        originVoucherEntryVO.setSubjectCode(subject.getSubjectCode());
        originVoucherEntryVO.setSubjectName(subject.getSubjectName());
        originVoucherEntryVO.setDirectionFlag(directionFlag);
        if (AccplatConsts.DirectionFlag.DEBIT.equals(directionFlag)) {
            originVoucherEntryVO.setDebitOriginalMny(mny);
        } else {
            originVoucherEntryVO.setCrebitOriginalMny(mny);
        }
        originVoucherEntryVO.setRateType(voucherTemplateVO.getRateType());
        originVoucherEntryVO.setCurrencyCode(voucherTemplateVO.getCurrencyCode());
        entryVOList.add(originVoucherEntryVO);
        log.info("==========解析分录【固定科目编码：" + subjectCode + "】=====end======================================");
        return entryVOList;
    }


    /**
     * @description: 根据科目分类构建分录
     *
     * @return {@link List<  OriginVoucherEntryVO >}
     * @author songlx
     * @date: 2023/12/26
     */
    public List<OriginVoucherEntryVO> analysisEntryBySubjectContrast(String billTypeCode, Map<String, Object> billData, VoucherTemplateDetailVO voucherTemplateDetailVO, VoucherTemplateVO voucherTemplateVO) {
        String subjectCategoryCode = voucherTemplateDetailVO.getSubjectRuleCode();
        String directionFlag = voucherTemplateDetailVO.getDirectionFlag();

        String description = voucherTemplateDetailVO.getDescriptionExpressionView();
        String descriptionExpression = voucherTemplateDetailVO.getDescriptionExpression();
        if (StringUtils.isNotBlank(descriptionExpression)) {
            description = executeBusinessExpression(descriptionExpression, billData, billTypeCode);
            //description = String.valueOf(execDescription);
        }

        String mnyExpression = voucherTemplateDetailVO.getMnyExpression();
        Object mnyObj = executeBusinessExpression(mnyExpression, billData, billTypeCode);
        BigDecimal mny = ComputeUtil.toBigDecimal(mnyObj);
        log.info("==========解析分录【科目分类编码：" + subjectCategoryCode + "】=====start====================================");
        List<OriginVoucherEntryVO> entriesList = new LinkedList<>();
        SubjectContrastVO subjectContrast = this.getSubjectContrast(null, subjectCategoryCode);
        List<SubjectContrastDetailVO> subjectContrastDetailList = subjectContrast.getSubjectContrastDetailList();
        String influenceIdsStr = subjectContrast.getInfluenceIds();
        if (StringUtils.isBlank(influenceIdsStr)) {
            throw new BusinessException("该科目分类【编码：" + subjectContrast.getSubjectCategoryCode() + "】未定义影响因素！");
        }
        String[] influenceIdsArr = influenceIdsStr.split(",");
        List<Long> influenceIds = Arrays.stream(influenceIdsArr).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());
        List<BillInfluenceSetVO> billInfluenceSetList = this.getBillInfluenceSet(billTypeCode, influenceIds);
        Map<Long, BillInfluenceSetVO> billInfluenceSetMap = billInfluenceSetList.stream().collect(Collectors.toMap(BillInfluenceSetVO::getInfluenceId, t -> t));

        if (CollectionUtils.isNotEmpty(subjectContrastDetailList)) {
            for (SubjectContrastDetailVO subjectContrastDetailVO : subjectContrastDetailList) {
                Long accountSubjectId = subjectContrastDetailVO.getAccountSubjectId();
                String accountSubjectCode = subjectContrastDetailVO.getAccountSubjectCode();
                String accountSubjectName = subjectContrastDetailVO.getAccountSubjectName();
                List<SubjectContrastDetailInfluenceVO> subjectContrastDetailInfluenceList = subjectContrastDetailVO.getSubjectContrastDetailInfluenceList();
                List expressionList = new LinkedList<String>();
                String expressionString = null;
                for (SubjectContrastDetailInfluenceVO subjectContrastDetailInfluenceVO : subjectContrastDetailInfluenceList) {
                    Long influenceId = subjectContrastDetailInfluenceVO.getInfluenceId();
                    String influenceValue = subjectContrastDetailInfluenceVO.getInfluenceValue();
                    BillInfluenceSetVO billInfluenceSetVO = billInfluenceSetMap.get(influenceId);
                    if (billInfluenceSetVO == null) {
                        throw new BusinessException("会计平台配置-影响因素【ID：" + subjectContrastDetailInfluenceVO.getInfluenceId() + "】未配置关联字段");
                    }
                    String columnExpression = billInfluenceSetVO.getColumnExpression();
                    String eqStringExpression = columnExpression + "==" + influenceValue;
                    expressionList.add(eqStringExpression);

                }
                expressionString = String.join(" && ", expressionList);
                // 编译表达式
                boolean b = executeBusinessExpression(expressionString, billData, billTypeCode);
                if (b) {
                    OriginVoucherEntryVO entryVO = OriginVoucherEntryVO.getInstanceVO(directionFlag, accountSubjectId, accountSubjectCode, accountSubjectName);
                    if (AccplatConsts.DirectionFlag.DEBIT.equals(directionFlag)) {
                        entryVO.setDebitOriginalMny(mny);
                    } else {
                        entryVO.setCrebitOriginalMny(mny);
                    }
                    entryVO.setEntryDescription(description);
                    entryVO.setRateType(voucherTemplateVO.getRateType());
                    entryVO.setCurrencyCode(voucherTemplateVO.getCurrencyCode());
                    entriesList.add(entryVO);
                }
            }
        }
        // 如果没有符合条件的，放入默认分录
        if (CollectionUtils.isEmpty(entriesList)) {
            Long defaultSubjectId = subjectContrast.getDefaultSubjectId();
            String defaultSubjectCode = subjectContrast.getDefaultSubjectCode();
            String defaultSubjectName = subjectContrast.getDefaultSubjectName();
            OriginVoucherEntryVO entryVO = OriginVoucherEntryVO.getInstanceVO(directionFlag, defaultSubjectId, defaultSubjectCode, defaultSubjectName);
            if (AccplatConsts.DirectionFlag.DEBIT.equals(directionFlag)) {
                entryVO.setDebitOriginalMny(mny);
            } else {
                entryVO.setCrebitOriginalMny(mny);
            }
            entryVO.setEntryDescription(description);
            entryVO.setRateType(voucherTemplateVO.getRateType());
            entryVO.setCurrencyCode(voucherTemplateVO.getCurrencyCode());
            entriesList.add(entryVO);
        }
        log.info("==========解析分录【科目分类编码：" + subjectCategoryCode + "】=====end======================================");
        return entriesList;
    }


    public void analysisAuxiliary(String billTypeCode, Map<String, Object> billData, List<OriginVoucherEntryVO> entriesList) {
        List<BillAuxiliarySetVO> billAuxiliarySet = this.getBillAuxiliarySet(billTypeCode);
        if (CollectionUtils.isEmpty(billAuxiliarySet)) {
            throw new BusinessException("未获取到【单据类型编码：" + billTypeCode + "】会计平台配置-辅助核算项关联");
        }
        Map<Long, BillAuxiliarySetVO> auxiliarySetMap = billAuxiliarySet.stream().collect(Collectors.toMap(BillAuxiliarySetVO::getAuxiliaryId, dto2 -> dto2));

        for (OriginVoucherEntryVO entryVO : entriesList) {
            Long accsubjectId = entryVO.getSubjectId();
            SubjectEntity subjectEntity = subjectService.selectById(accsubjectId);
            String auxiliaryIds = subjectEntity.getAuxiliaryIds();
            if (StringUtils.isNotBlank(auxiliaryIds)) {
                List<OriginVoucherAuxiliaryVO> originVoucherAuxiliaryVOS = new LinkedList<>();
                for (String auxiliaryId : auxiliaryIds.split(",")) {
                    Long _auxiliaryId = Long.valueOf(auxiliaryId);
                    BillAuxiliarySetVO billAuxiliarySetVO = auxiliarySetMap.get(_auxiliaryId);
                    String columnExpression = billAuxiliarySetVO.getColumnExpression();
                    String referCode = billAuxiliarySetVO.getReferCode();
                    Object auxiliaryIdObj = executeBusinessExpression(columnExpression, billData, billTypeCode);
                    JSONObject auxiliaryValJson = getReferCodeValue(String.valueOf(auxiliaryIdObj), referCode);

                    OriginVoucherAuxiliaryVO originVoucherAuxiliaryVO = new OriginVoucherAuxiliaryVO();
                    originVoucherAuxiliaryVO.setAuxiliaryId(billAuxiliarySetVO.getAuxiliaryId());
                    originVoucherAuxiliaryVO.setAuxiliaryCode(billAuxiliarySetVO.getAuxiliaryCode());
                    originVoucherAuxiliaryVO.setAuxiliaryName(billAuxiliarySetVO.getAuxiliaryName());
                    originVoucherAuxiliaryVO.setAuxiliaryValueId(auxiliaryValJson.getLong("id"));
                    String codeVal = auxiliaryValJson.getString("code");
                    originVoucherAuxiliaryVO.setAuxiliaryValueCode(codeVal);
                    originVoucherAuxiliaryVO.setAuxiliaryValueName(auxiliaryValJson.getString("name"));
                    // TODO 这里默认放入工程云辅助项编码和值编码， 后续需要转化财务系统档案编码和值编码
                    originVoucherAuxiliaryVO.setAuxiliaryFsCode(billAuxiliarySetVO.getAuxiliaryCode());
                    originVoucherAuxiliaryVO.setAuxiliaryFsValue(codeVal);
                    originVoucherAuxiliaryVOS.add(originVoucherAuxiliaryVO);
                }
                entryVO.setOriginVoucherAuxiliaryVOList(originVoucherAuxiliaryVOS);
            }
        }
    }


    public JSONObject getReferCodeValue(String value, String referCode) {
        try {
            JSONArray jArray = ReferObjectUtil.getReferEntityValue(value, referCode);
            if (jArray != null && jArray.size() > 0) {
                JSONObject json = (JSONObject) jArray.get(0);
                return json;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * @description: 依据原始凭证转化财务凭证
     *
     * @author songlx
     * @date: 2023/11/29
     */
    public Map<String, Object> transferFinanceVoucherByOriginVoucher(VoucherParams voucherParams, OriginVoucherVO originVoucherVO) {
        String systemCode = voucherParams.getFinanceSystemCode();
        IFinanceVoucherService financeVoucherService = voucherServiceFactory.getFinanceVoucherService(systemCode);
        return financeVoucherService.convertToFinanceVoucherByOriginVoucher(voucherParams, originVoucherVO);
    }


    /**
     * @description: 根据科目表编码和科目分类编码获取科目对照
     *
     * @param subjectChartCode
     * @param subjectCategoryCode
     * @return {@link SubjectContrastVO}
     * @author songlx
     * @date: 2023/12/20
     */
    public SubjectContrastVO getSubjectContrast(String subjectChartCode, String subjectCategoryCode) {
        return subjectContrastService.getOneBySubjectCategoryCode(subjectChartCode, subjectCategoryCode);
    }

    /**
     * @description: 获取固定科目
     *
     * @return {@link SubjectVO}
     * @author songlx
     * @date: 2023/12/20
     */
    public SubjectVO getSubject(String subjectCode) {
        LambdaQueryWrapper<SubjectEntity> subjectEntityLambdaQueryWrapper = new LambdaQueryWrapper<>();
        subjectEntityLambdaQueryWrapper.eq(SubjectEntity::getSubjectCode, subjectCode);
        subjectEntityLambdaQueryWrapper.eq(SubjectEntity::getTenantId, InvocationInfoProxy.getTenantid());
        SubjectEntity one = subjectService.getOne(subjectEntityLambdaQueryWrapper);
        return BeanMapper.map(one, SubjectVO.class);
    }


    /**
     * @description: 获取辅助核算项关联
     *
     * @param billTypeCode
     * @return {@link List< BillAuxiliarySetVO>}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillAuxiliarySetVO> getBillAuxiliarySet(String billTypeCode) {
        LambdaQueryWrapper<BillAuxiliarySetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillAuxiliarySetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillAuxiliarySetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BillAuxiliarySetEntity> list = billAuxiliarySetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【辅助核算项关联】");
        }
        return BeanMapper.mapList(list, BillAuxiliarySetVO.class);
    }


    /**
     * @description: 获取影响因素关联
     *
     * @param billTypeCode
     * @return {@link List< BillInfluenceSetVO>}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillInfluenceSetVO> getBillInfluenceSet(String billTypeCode, List<Long> influenceIds) {
        LambdaQueryWrapper<BillInfluenceSetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillInfluenceSetEntity::getBillTypeCode, billTypeCode);
        wrapper.in(CollectionUtils.isNotEmpty(influenceIds), BillInfluenceSetEntity::getInfluenceId, influenceIds);
        wrapper.eq(BillInfluenceSetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BillInfluenceSetEntity> list = billInfluenceSetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【影响因素关联】");
        }
        return BeanMapper.mapList(list, BillInfluenceSetVO.class);
    }


    /**
     * @description: 获取当前组织的账簿，递归最近上级获取
     *
     * @param orgId
     * @param billTypeCode
     * @return {@link BillAccbookSetVO}
     * @author songlx
     * @date: 2023/12/18
     */
    protected BillAccbookSetVO getBillAccbookSet(Long orgId, String billTypeCode) {
        CommonResponse<OrgVO> oneById = orgApi.getOneById(orgId);
        OrgVO orgVO = oneById.getData();
        String[] parentOrgIds = orgVO.getInnerCode().split("\\|");

        LambdaQueryWrapper<BillAccbookSetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillAccbookSetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillAccbookSetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        wrapper.in(BillAccbookSetEntity::getSourceOrgId, parentOrgIds);
        List<BillAccbookSetEntity> list = billAccbookSetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【账簿映射规则】, orgId:" + orgId);
        }
        Map<Long, BillAccbookSetEntity> setMap = list.stream().collect(Collectors.toMap(BillAccbookSetEntity::getSourceOrgId, Function.identity(), (key1, key2) -> key2));
        BillAccbookSetVO billAccbookSetVO = null;
        for (int i = parentOrgIds.length - 1; i >= 0; i--) {
            BillAccbookSetEntity billAccbookSetEntity = setMap.get(Long.valueOf(parentOrgIds[i]));
            if (billAccbookSetEntity != null) {
                billAccbookSetVO = BeanMapper.map(billAccbookSetEntity, BillAccbookSetVO.class);
                break;
            }
        }
        return billAccbookSetVO;
    }


    /**
     * @description: 是否生成原始凭证，如为true，请在transferOriginVoucher中转化
     * @author songlx
     * @date: 2023/11/29
     */
    public Boolean getIsGenerateOriginVoucherFlag(VoucherParams voucherParams) {
        return Boolean.TRUE;
    }


    /**
     * @description: 是否生成财务凭证，如为true，请在transferFinanceVoucher中转化
     * 后续可能改成配置参数获取
     * @author songlx
     * @date: 2023/11/29
     */
    public Boolean getIsGenerateFinanceVoucherFlag(VoucherParams voucherParams) {
        return Boolean.TRUE;
    }


    public static void main(String[] args) {
        // 创建 Aviator 表达式
        String expressionString = "'x:' + x";

        // 编译表达式
        Expression expression = AviatorEvaluator.getInstance().compile(expressionString);

        // 定义变量
        Map<String, Object> variables = new HashMap<>();
        variables.put("x", 10);
        variables.put("y", "10");

        // 第一次执行表达式
        Object result1 = expression.execute(variables);
        System.out.println("Result 1: " + result1); // 输出 Result 1: 15

        // 修改变量值
        variables.put("x", 20);
        variables.put("y", 8);

        // 第二次执行表达式
        Object result2 = expression.execute(variables);
        System.out.println("Result 2: " + result2); // 输出 Result 2: 28
    }

}
