package com.ejianc.business.voucher.service;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.accplat.config.bean.*;
import com.ejianc.business.accplat.config.service.*;
import com.ejianc.business.accplat.config.vo.*;
import com.ejianc.business.accplat.consts.AccplatConsts;
import com.ejianc.business.accplat.originvoucher.vo.*;
import com.ejianc.business.voucher.transfer.VoucherTransferFactory;
import com.ejianc.business.voucher.utils.AviatorUtil;
import com.ejianc.business.voucher.utils.DataConvertUtil;
import com.ejianc.business.voucher.vo.VoucherParams;
import com.ejianc.foundation.dataCompare.api.IDataCompareApi;
import com.ejianc.foundation.metadata.api.IMdAttributeApi;
import com.ejianc.foundation.metadata.vo.MdAttributeVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.refer.util.ReferObjectUtil;
import com.googlecode.aviator.AviatorEvaluator;
import com.googlecode.aviator.Expression;
import com.googlecode.aviator.lexer.token.OperatorType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 公共凭证业务服务父类
 *
 * @author songlx
 * @date 2023/11/28
 * @version 1.0
 */
@Slf4j
public class BaseVoucherBusinessService {

    @Autowired
    private VoucherTransferFactory voucherTransferFactory;

    @Autowired
    private IBillAccbookSetService billAccbookSetService;

    @Autowired
    private IBillInfluenceSetService billInfluenceSetService;

    @Autowired
    private IBillAuxiliarySetService billAuxiliarySetService;

    @Autowired
    private IAuxiliaryService auxiliaryService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IAccbookService accbookService;

    @Autowired
    private ISubjectContrastService subjectContrastService;

    @Autowired
    private ISubjectService subjectService;

    @Autowired
    private IMdAttributeApi mdAttributeApi;

    @Autowired
    private CacheManager cacheManager;

    @Autowired
    private IDataCompareApi dataCompareApi;


    /**
     * @description: 解析单个表达式并获取结果。可以处理主表、主表关联、子表字段
     *
     * @param express 表达式
     * @param billData 单据数据
     * @param billTypeCode 单据类型
     * @param logic 逻辑连接符(AviatorUtil里有定义符号常量），若表达式涉及子表，则会出现子表维度的结果集。此连接符表示多个结果集的逻辑关系，其他情况不需要可不传
     * @return {@link T}
     * @author songlx
     * @date: 2024/1/4
     */
    public <T> T executeOneExpression(String express, Map<String, Object> billData, String billTypeCode, String logic) {
        // 疑似存在关联或者子表字段特殊解析处理
        String subTableKey = this.getSubTableKey(billTypeCode, billData, express);
        if (subTableKey != null) {
            String _subTableKey = "subData-" + subTableKey;
            JSONArray subDatas = new JSONArray();
            // 是否存在镜像遍历数据
            if (billData.containsKey(_subTableKey)) {
                subDatas = (JSONArray) billData.get(_subTableKey);
            } else {
                Object sub = billData.get(subTableKey);
                if (sub != null) {
                    subDatas = (JSONArray) sub;
                    // 放入备份镜像子表数据，避免拆分覆盖丢失子表数据
                    billData.put(_subTableKey, subDatas);
                }
            }
            // 拆分遍历单个子表执行表达式结果组成合成表达式
            int a = 1;
            LinkedList subExpress = new LinkedList<String>();
            Map<String, Object> subExecResMap = new HashMap<>();
            for (Object subData : subDatas) {
                billData.remove(subTableKey);
                billData.put(subTableKey, subData);
                Object oneSubExecObj = AviatorUtil.exec(express, billData);
                String key = "oneSubRes" + a;
                subExecResMap.put(key, oneSubExecObj);
                subExpress.push(key);
                a++;
            }
            express = String.join(logic, subExpress);
            Object execSub = AviatorUtil.exec(express, subExecResMap);
            log.info("executeOneExpression(B)表达式执行：{}, 结果：{}", express, execSub);
            return (T) execSub;

        }
        Object exec = AviatorUtil.exec(express, billData);
        log.info("executeOneExpression(H)表达式执行：{}, 结果：{}", express, exec);
        return (T) exec;
    }

    /**
     * @description: 设置关联实体值
     * @return
     * @author songlx
     * @date: 2023/12/28
     */
    private void setRelationEntityValue(String columnExpression, Map<String, Object> billData, String billTypeCode, Map<String, MdAttributeVO> mdAttributeVOMap) {
        // 关联字段格式_projectId.type
        String _headCol = columnExpression.substring(0, columnExpression.indexOf("."));
        String headCol = columnExpression.substring(1, columnExpression.indexOf("."));
        if (!billData.containsKey(_headCol)) {
            MdAttributeVO sourceAttributeVO = mdAttributeVOMap.get(headCol);
            Object execVal = AviatorUtil.exec(headCol, billData);
            CommonResponse<Map<String, Object>> mapCommonResponse = mdAttributeApi.queryRelationEntityValue(billTypeCode, sourceAttributeVO.getEntityId(), sourceAttributeVO.getId(), String.valueOf(execVal), null);
            if (mapCommonResponse.isSuccess() && mapCommonResponse.getData() != null) {
                billData.put(_headCol, mapCommonResponse.getData());
            }
        }
    }


    /**
     * @description: 依据凭证模板转化原始凭证
     * @author songlx
     * @date: 2023/11/29
     */
    public OriginVoucherVO transferOriginVoucherByVoucherTemplate(VoucherParams voucherParams, VoucherTemplateVO voucherTemplateVO) {
        String billTypeCode = voucherParams.getBillTypeCode();
        Map<String, Object> billData = voucherParams.getBillData();
        Long billId = Long.valueOf(String.valueOf(billData.get("id")));
        voucherParams.setBillId(billId);
        Long orgId = voucherParams.getOrgId();
        Date businessDate = DataConvertUtil.getDate(executeOneExpression(voucherTemplateVO.getBusinessDateColumn(), billData, billTypeCode, null));

        OriginVoucherVO originVoucherVO = new OriginVoucherVO();
        // 来源单据类型
        BillTypeVO billTypeVO = voucherParams.getBillTypeVO();
        originVoucherVO.setSrcBillTypeId(billTypeVO.getId());
        originVoucherVO.setSrcBillTypeCode(billTypeVO.getBillCode());
        originVoucherVO.setSrcBillTypeName(billTypeVO.getBillName());
        // 来源系统编码
        originVoucherVO.setSrcSystemCode(voucherParams.getSrcSystemCode());
        // 来源系统名称
        originVoucherVO.setSrcSystemName(voucherParams.getSrcSystemName());
        // 来源业务单据
        originVoucherVO.setSrcBillId(billId);
        String billCode = voucherParams.getBillCode();
        String billCodeColumn = voucherTemplateVO.getBillCodeColumn();
        if (StringUtils.isEmpty(billCode) && StringUtils.isNotEmpty(billCodeColumn)) {
            billCode = executeOneExpression(billCodeColumn, billData, billTypeCode, null);
        }
        originVoucherVO.setSrcBillCode(billCode);
        // 来源业务单据URL-PC
        String srcBillPcUrl = voucherTemplateVO.getSrcBillPcUrl();
        if (StringUtils.isNotBlank(srcBillPcUrl) && srcBillPcUrl.contains("#")) {
            String pcExpression = "'" + srcBillPcUrl + "'";
            srcBillPcUrl = executeOneExpression(pcExpression, billData, billTypeCode, null);
        }
        originVoucherVO.setSrcBillPcUrl(srcBillPcUrl);

        // 来源业务单据主组织
        JSONObject orgObj = getReferCodeValue(String.valueOf(orgId), "idm-org");
        originVoucherVO.setSrcBillOrgId(orgId);
        if (orgObj != null) {
            originVoucherVO.setSrcBillOrgCode(orgObj.getString("code"));
            originVoucherVO.setSrcBillOrgName(orgObj.getString("name"));
        }
        // 凭证生成时间
        originVoucherVO.setVoucherCreateTime(new Date());
        originVoucherVO.setVoucherDate(new Date());
        // 业务日期
        originVoucherVO.setBusinessDate(businessDate);
        // 制单人
        String markerColumn = voucherTemplateVO.getMarkerColumn();
        if (StringUtils.isNotEmpty(markerColumn)) {
            String markerId = executeOneExpression(markerColumn, billData, billTypeCode, null);
            JSONObject markerObj = getReferCodeValue(markerId, "idm-employee");
            originVoucherVO.setMarkerId(Long.valueOf(markerId));
            originVoucherVO.setMarkerCode(markerObj.getString("code"));
            originVoucherVO.setMarkerName(markerObj.getString("name"));

        }

        // 凭证类型
        String voucherType = voucherTemplateVO.getVoucherType();
        if (StringUtils.isNotEmpty(voucherType)) {
            String[] voucherTypeArr = voucherType.split(":|：");
            originVoucherVO.setVoucherTypeCode(voucherTypeArr[0]);
            originVoucherVO.setVoucherTypeName(voucherTypeArr[1]);
        }
        // 借方金额合计debitTotalMny、贷方金额合计creditTotalMny、摘要description

        // 设置账簿
        originVoucherVO.setAccbookId(voucherTemplateVO.getAccbookId());
        originVoucherVO.setAccbookCode(voucherTemplateVO.getAccbookCode());
        originVoucherVO.setAccbookName(voucherTemplateVO.getAccbookName());

        // 目标财务系统标识
        originVoucherVO.setFinanceSystemCode(voucherParams.getFinanceSystemCode());
        // 转换模版ID
        originVoucherVO.setVoucherTemplateId(voucherTemplateVO.getId());

        // 3 构建分录、辅助核算项
        Map<Long, BillAuxiliarySetVO> auxiliarySetMap = new HashMap<>();
        List<BillAuxiliarySetVO> billAuxiliarySet = this.getBillAuxiliarySet(billTypeCode);
        if (CollectionUtils.isNotEmpty(billAuxiliarySet)) {
            auxiliarySetMap = billAuxiliarySet.stream().collect(Collectors.toMap(BillAuxiliarySetVO::getAuxiliaryId, dto2 -> dto2));
        }
        List<OriginVoucherEntryVO> entryList = new ArrayList<>();
        List<VoucherTemplateDetailVO> voucherTemplateDetailList = voucherTemplateVO.getVoucherTemplateDetailList();
        for (VoucherTemplateDetailVO voucherTemplateDetailVO : voucherTemplateDetailList) {
            String subjectRuleFlag = voucherTemplateDetailVO.getSubjectRuleFlag();
            List<OriginVoucherEntryVO> list = null;
            if (AccplatConsts.SubjectRule.SUBJECT_CONTRAST.equals(subjectRuleFlag)) {
                list = this.analysisEntryBySubjectContrast(billTypeCode, billData, voucherTemplateDetailVO, voucherTemplateVO, auxiliarySetMap);
            } else {
                list = this.analysisEntryBySubject(billTypeCode, billData, voucherTemplateDetailVO, voucherTemplateVO, auxiliarySetMap);
            }
            entryList.addAll(list);
        }

        // 合并分录、金额为空过滤
        List<OriginVoucherEntryVO> mergeEntryList = this.mergeEntry(voucherTemplateVO, originVoucherVO, entryList);
        originVoucherVO.setOriginVoucherEntryList(mergeEntryList);
        return originVoucherVO;
    }

    /**
     * @description: 合并分录、金额为空过滤
     *
     * @param entryList
     * @return {@link List< OriginVoucherEntryVO>}
     * @author songlx
     * @date: 2024/1/9
     */
    private List<OriginVoucherEntryVO> mergeEntry(VoucherTemplateVO voucherTemplateVO, OriginVoucherVO originVoucherVO, List<OriginVoucherEntryVO> entryList) {
        String entryMergeRule = voucherTemplateVO.getEntryMergeRule();
        // 金额为空过滤
        Boolean mnyNullFilterFlag = voucherTemplateVO.getMnyNullFilterFlag();

        BigDecimal debitTotal = null;
        BigDecimal creditTotal = null;

        List<OriginVoucherAuxiliaryVO> auxiliaryList = new ArrayList<>();

        Map<String, OriginVoucherEntryVO> map = new HashMap<>();
        for (int i = 0; i < entryList.size(); i++) {
            OriginVoucherEntryVO originVoucherEntryVO = entryList.get(i);
            List<OriginVoucherAuxiliaryVO> auxiliaryVOList = originVoucherEntryVO.getOriginVoucherAuxiliaryVOList();
            if (CollectionUtils.isNotEmpty(auxiliaryVOList)) {
                auxiliaryList.addAll(auxiliaryVOList);
            }

            BigDecimal debitOriginalMny = originVoucherEntryVO.getDebitOriginalMny();
            BigDecimal creditOriginalMny = originVoucherEntryVO.getCreditOriginalMny();
            if (mnyNullFilterFlag && ComputeUtil.isEmpty(debitOriginalMny) && ComputeUtil.isEmpty(creditOriginalMny)) {
                continue;
            }
            debitTotal = ComputeUtil.safeAdd(debitTotal, debitOriginalMny);
            creditTotal = ComputeUtil.safeAdd(creditTotal, creditOriginalMny);

            Long subjectId = originVoucherEntryVO.getSubjectId();
            String entryDescription = originVoucherEntryVO.getEntryDescription();
            String auxiliaryIds = originVoucherEntryVO.getAuxiliaryIds();
            String directionFlag = originVoucherEntryVO.getDirectionFlag();
            String key = String.valueOf(i);
            if (AccplatConsts.EntryMergeRule.SUBJECT_DESCRIPTION_AUXILIARY_DIRECTION.equals(entryMergeRule)) {
                // 暂只支持一种合并规则：科目+摘要+辅助核算项+方向一致
                key = subjectId + entryDescription + auxiliaryIds + directionFlag;
            }
            OriginVoucherEntryVO _originVoucherEntryVO = map.get(key);
            if (_originVoucherEntryVO != null) {
                BigDecimal debitMny = ComputeUtil.safeAdd(_originVoucherEntryVO.getDebitOriginalMny(), debitOriginalMny);
                _originVoucherEntryVO.setDebitOriginalMny(debitMny);
                BigDecimal creditMny = ComputeUtil.safeAdd(_originVoucherEntryVO.getCreditOriginalMny(), creditOriginalMny);
                _originVoucherEntryVO.setCreditOriginalMny(creditMny);
            } else {
                map.put(key, originVoucherEntryVO);
            }
        }

        originVoucherVO.setDebitTotalMny(debitTotal);
        originVoucherVO.setCreditTotalMny(creditTotal);
        List<OriginVoucherEntryVO> entryVOList = map.values().stream().collect(Collectors.toList());
        // 辅助核算项财务编码转换
        getDataCompareMap(entryVOList, auxiliaryList);

        return entryVOList;
    }

    /**
     * @description: 辅助核算项财务编码转换
     *
     * @param entryVOList
     * @param auxiliaryList
     * @return
     * @author songlx
     * @date: 2024/1/23
     */
    private void getDataCompareMap(List<OriginVoucherEntryVO> entryVOList, List<OriginVoucherAuxiliaryVO> auxiliaryList) {
        if (CollectionUtils.isNotEmpty(auxiliaryList)) {
            Map<String, String> dataCompareMap = new HashMap<>();
            Map<Long, List<String>> auxiliaryCodeMap = auxiliaryList.stream().filter(t -> t.getCompareAppId() != null).collect(Collectors.groupingBy(OriginVoucherAuxiliaryVO::getCompareAppId, Collectors.mapping(OriginVoucherAuxiliaryVO::getAuxiliaryFsValue, Collectors.toList())));
            if (MapUtils.isNotEmpty(auxiliaryCodeMap)) {
                CommonResponse<Map<String, String>> appDataMapRes = dataCompareApi.getAppDataMap(auxiliaryCodeMap);
                if (appDataMapRes.isSuccess()) {
                    dataCompareMap = appDataMapRes.getData();
                }
                for (OriginVoucherEntryVO originVoucherEntryVO : entryVOList) {
                    List<OriginVoucherAuxiliaryVO> originVoucherAuxiliaryVOList = originVoucherEntryVO.getOriginVoucherAuxiliaryVOList();
                    for (OriginVoucherAuxiliaryVO auxiliaryVO : originVoucherAuxiliaryVOList) {
                        Long compareAppId = auxiliaryVO.getCompareAppId();
                        if (compareAppId != null) {
                            String auxiliaryFsValue = auxiliaryVO.getAuxiliaryFsValue();
                            String fsCodeVal = dataCompareMap.get(compareAppId + auxiliaryFsValue);
                            if (StringUtils.isNotBlank(fsCodeVal)) {
                                auxiliaryVO.setAuxiliaryFsValue(fsCodeVal);
                            }
                        }
                    }
                }
            }
        }
    }


    /**
     * @description: 依据分录金额表达式设置主表关联字段值并判断是否含有子表键值（如果有返回子表键值，无则返回null）
     *
     * @param billTypeCode
     * @param billData
     * @param mnyExpression
     * @return {@link String}
     * @author songlx
     * @date: 2024/1/12
     */
    public String getSubTableKey(String billTypeCode, Map<String, Object> billData, String mnyExpression) {
        List<String> columnNameList = DataConvertUtil.getRelaionOrSubColumnList(mnyExpression);
        List<String> subTableKeyList = new ArrayList<>();
        Map<String, MdAttributeVO> mdAttributeVOMap = null;
        String cacheKey = "mdattr" + billTypeCode;
        if (cacheManager.exists(cacheKey)) {
            mdAttributeVOMap = cacheManager.get(cacheKey);
        } else {
            CommonResponse<Map<String, MdAttributeVO>> attrMapRes = mdAttributeApi.getAttrMap(billTypeCode);
            mdAttributeVOMap = attrMapRes.getData();
        }
        for (String column : columnNameList) {
            MdAttributeVO mdAttributeVO = mdAttributeVOMap.get(column);
            if (mdAttributeVO == null) continue;
            String columnPos = mdAttributeVO.getColumnPos();
            // 头部关联字段处理
            if (AccplatConsts.ColumnPos.HR.equals(columnPos)) {
                this.setRelationEntityValue(column, billData, billTypeCode, mdAttributeVOMap);
            } else if (AccplatConsts.ColumnPos.B.equals(columnPos)) {
                // detailList.code
                String[] subKeys = column.split("\\.");
                subTableKeyList.add(subKeys[0]);
            }
        }
        if (subTableKeyList.size() > 1) {
            throw new BusinessException("单据类型编码【" + billTypeCode + "】的表达式【" + mnyExpression + "】中含有至少两个子表键值，不能正常解析数据！");
        }
        return CollectionUtils.isNotEmpty(subTableKeyList) ? subTableKeyList.get(0) : null;
    }


    /**
     * @description: 根据固定科目构建分录
     *
     * @return {@link List<  OriginVoucherEntryVO >}
     * @author songlx
     * @date: 2023/12/26
     */
    public List<OriginVoucherEntryVO> analysisEntryBySubject(String billTypeCode, Map<String, Object> billData, VoucherTemplateDetailVO voucherTemplateDetailVO, VoucherTemplateVO voucherTemplateVO, Map<Long, BillAuxiliarySetVO> auxiliarySetMap) {
        String subjectCode = voucherTemplateDetailVO.getSubjectRuleCode();
        log.info("==========解析分录【固定科目编码：" + subjectCode + "】=====start====================================");
        String directionFlag = voucherTemplateDetailVO.getDirectionFlag();
        String mnyExpression = voucherTemplateDetailVO.getMnyExpression();
        String descriptionFlag = voucherTemplateDetailVO.getDescriptionFlag();
        String conditionExpression = voucherTemplateDetailVO.getConditionExpression();
        String descriptionExpression = voucherTemplateDetailVO.getDescriptionExpression();

        List<OriginVoucherEntryVO> entryVOList = new LinkedList<>();
        String subTableKey = this.getSubTableKey(billTypeCode, billData, mnyExpression);
        // 主表金额、摘要解析
        if (subTableKey == null) {
            Boolean headbool = Boolean.TRUE;
            if (StringUtils.isNotEmpty(conditionExpression)) {
                headbool = AviatorUtil.exec(conditionExpression, billData);
            }
            log.info("==========分录条件(H)：{}，结果：{}", conditionExpression, headbool);
            if (headbool) {
                OriginVoucherEntryVO originVoucherEntryVO = new OriginVoucherEntryVO();
                String description = descriptionExpression;
                if (StringUtils.isNotEmpty(descriptionExpression) && AccplatConsts.DescriptionFlag.BILL_COLUMN.equals(descriptionFlag)) {
                    description = executeOneExpression(descriptionExpression, billData, billTypeCode, OperatorType.AND.getToken());
                }
                originVoucherEntryVO.setEntryDescription(description);
                originVoucherEntryVO.setSubjectId(voucherTemplateDetailVO.getSubjectRuleId());
                originVoucherEntryVO.setSubjectCode(voucherTemplateDetailVO.getSubjectRuleCode());
                originVoucherEntryVO.setSubjectName(voucherTemplateDetailVO.getSubjectRuleName());
                originVoucherEntryVO.setDirectionFlag(directionFlag);
                BigDecimal mny = ComputeUtil.toBigDecimal((Object) AviatorUtil.exec(mnyExpression, billData));
                if (AccplatConsts.DirectionFlag.DEBIT.equals(directionFlag)) {
                    originVoucherEntryVO.setDebitOriginalMny(mny);
                } else {
                    originVoucherEntryVO.setCreditOriginalMny(mny);
                }
                originVoucherEntryVO.setRateType(voucherTemplateVO.getRateType());
                originVoucherEntryVO.setCurrencyCode(voucherTemplateVO.getCurrencyCode());
                this.analysisAuxiliary(originVoucherEntryVO, auxiliarySetMap, billData);
                entryVOList.add(originVoucherEntryVO);
            }
        } else {
            String _subTableKey = "subData-" + subTableKey;
            JSONArray subDatas = new JSONArray();
            // 是否存在镜像遍历数据
            if (billData.containsKey(_subTableKey)) {
                subDatas = (JSONArray) billData.get(_subTableKey);
            } else {
                Object sub = billData.get(subTableKey);
                if (sub != null) {
                    subDatas = (JSONArray) sub;
                    // 放入备份镜像子表数据，避免拆分覆盖丢失子表数据
                    billData.put(_subTableKey, subDatas);
                }
            }
            // 按子表数据一次解析每条数据该入什么科目
            int a = 1;
            for (Object subData : subDatas) {
                billData.remove(subTableKey);
                billData.put(subTableKey, subData);
                Boolean subbool = Boolean.TRUE;
                if (StringUtils.isNotEmpty(conditionExpression)) {
                    subbool = AviatorUtil.exec(conditionExpression, billData);
                }
                log.info("==========分录条件(B{})：{}，结果：{}", a, conditionExpression, subbool);
                a++;
                if (subbool) {
                    OriginVoucherEntryVO originVoucherEntryVO = new OriginVoucherEntryVO();
                    // 摘要
                    String subDescription = descriptionExpression;
                    if (StringUtils.isNotEmpty(descriptionExpression) && AccplatConsts.DescriptionFlag.BILL_COLUMN.equals(descriptionFlag)) {
                        subDescription = executeOneExpression(descriptionExpression, billData, billTypeCode, OperatorType.AND.getToken());
                    }
                    originVoucherEntryVO.setEntryDescription(subDescription);
                    // 金额
                    BigDecimal headMny = ComputeUtil.toBigDecimal((Object) AviatorUtil.exec(mnyExpression, billData));
                    if (AccplatConsts.DirectionFlag.DEBIT.equals(directionFlag)) {
                        originVoucherEntryVO.setDebitOriginalMny(headMny);
                    } else {
                        originVoucherEntryVO.setCreditOriginalMny(headMny);
                    }
                    originVoucherEntryVO.setDirectionFlag(directionFlag);
                    originVoucherEntryVO.setRateType(voucherTemplateVO.getRateType());
                    originVoucherEntryVO.setCurrencyCode(voucherTemplateVO.getCurrencyCode());

                    originVoucherEntryVO.setSubjectId(voucherTemplateDetailVO.getSubjectRuleId());
                    originVoucherEntryVO.setSubjectCode(voucherTemplateDetailVO.getSubjectRuleCode());
                    originVoucherEntryVO.setSubjectName(voucherTemplateDetailVO.getSubjectRuleName());
                    this.analysisAuxiliary(originVoucherEntryVO, auxiliarySetMap, billData);
                    entryVOList.add(originVoucherEntryVO);
                }
            }
        }
        log.info("==========解析分录【固定科目编码：" + subjectCode + "】=====end======================================");
        return entryVOList;
    }


    /**
     * @description: 根据科目分类构建分录
     *
     * @return {@link List<  OriginVoucherEntryVO >}
     * @author songlx
     * @date: 2023/12/26
     */
    public List<OriginVoucherEntryVO> analysisEntryBySubjectContrast(String billTypeCode, Map<String, Object> billData, VoucherTemplateDetailVO voucherTemplateDetailVO, VoucherTemplateVO voucherTemplateVO, Map<Long, BillAuxiliarySetVO> auxiliarySetMap) {
        String subjectCategoryCode = voucherTemplateDetailVO.getSubjectRuleCode();
        log.info("==========解析分录【科目分类编码：" + subjectCategoryCode + "】=====start====================================");
        SubjectContrastVO subjectContrast = this.getSubjectContrast(subjectCategoryCode);
        List<SubjectContrastDetailVO> subjectContrastDetailList = subjectContrast.getSubjectContrastDetailList();
        String influenceIdsStr = subjectContrast.getInfluenceIds();
        if (StringUtils.isBlank(influenceIdsStr)) {
            throw new BusinessException("该科目分类【编码：" + subjectContrast.getSubjectCategoryCode() + "】未定义影响因素！");
        }
        String[] influenceIdsArr = influenceIdsStr.split(",");
        List<Long> influenceIds = Arrays.stream(influenceIdsArr).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());
        List<BillInfluenceSetVO> billInfluenceSetList = this.getBillInfluenceSet(billTypeCode, influenceIds);
        if (influenceIdsArr.length != billInfluenceSetList.size()) {
            throw new BusinessException("单据【单据类型编码：" + billTypeCode + "】的科目分类【编码：" + subjectContrast.getSubjectCategoryCode() + "】的影响因素（" + influenceIdsArr.length + "个）和影响因素字段关联（" + billInfluenceSetList.size() + "个）不符，请检查！");
        }
        Map<Long, BillInfluenceSetVO> billInfluenceSetMap = billInfluenceSetList.stream().collect(Collectors.toMap(BillInfluenceSetVO::getInfluenceId, t -> t));

        String directionFlag = voucherTemplateDetailVO.getDirectionFlag();
        String mnyExpression = voucherTemplateDetailVO.getMnyExpression();
        String descriptionFlag = voucherTemplateDetailVO.getDescriptionFlag();
        String conditionExpression = voucherTemplateDetailVO.getConditionExpression();
        String descriptionExpression = voucherTemplateDetailVO.getDescriptionExpression();

        List<OriginVoucherEntryVO> entriesList = new LinkedList<>();
        String subTableKey = this.getSubTableKey(billTypeCode, billData, mnyExpression);
        // 主表金额、摘要解析
        if (subTableKey == null) {
            boolean conditionBool = Boolean.TRUE;
            if (StringUtils.isNotEmpty(conditionExpression)) {
                conditionBool = AviatorUtil.exec(conditionExpression, billData);
            }
            log.info("==========分录条件(H)：{}，结果：{}", conditionExpression, conditionBool);
            if (!conditionBool) return entriesList;

            OriginVoucherEntryVO originVoucherEntryVO = new OriginVoucherEntryVO();
            // 摘要
            String headDescription = descriptionExpression;
            if (StringUtils.isNotEmpty(descriptionExpression) && AccplatConsts.DescriptionFlag.BILL_COLUMN.equals(descriptionFlag)) {
                headDescription = executeOneExpression(descriptionExpression, billData, billTypeCode, OperatorType.AND.getToken());
            }
            originVoucherEntryVO.setEntryDescription(headDescription);
            // 金额
            BigDecimal headMny = ComputeUtil.toBigDecimal((Object) AviatorUtil.exec(mnyExpression, billData));
            if (AccplatConsts.DirectionFlag.DEBIT.equals(directionFlag)) {
                originVoucherEntryVO.setDebitOriginalMny(headMny);
            } else {
                originVoucherEntryVO.setCreditOriginalMny(headMny);
            }
            originVoucherEntryVO.setDirectionFlag(directionFlag);
            originVoucherEntryVO.setRateType(voucherTemplateVO.getRateType());
            originVoucherEntryVO.setCurrencyCode(voucherTemplateVO.getCurrencyCode());
            // 先放入默认科目，有符合条件再替换
            originVoucherEntryVO.setSubjectId(subjectContrast.getDefaultSubjectId());
            originVoucherEntryVO.setSubjectCode(subjectContrast.getDefaultSubjectCode());
            originVoucherEntryVO.setSubjectName(subjectContrast.getDefaultSubjectName());
            // 依据科目转换明细，看是否有符合的
            for (SubjectContrastDetailVO subjectContrastDetailVO : subjectContrastDetailList) {
                List<SubjectContrastDetailInfluenceVO> subjectContrastDetailInfluenceList = subjectContrastDetailVO.getSubjectContrastDetailInfluenceList();
                List<String> contrastExpressionList = new LinkedList<>();
                for (SubjectContrastDetailInfluenceVO subjectContrastDetailInfluenceVO : subjectContrastDetailInfluenceList) {
                    Long influenceId = subjectContrastDetailInfluenceVO.getInfluenceId();
                    String influenceValue = subjectContrastDetailInfluenceVO.getInfluenceValue();
                    BillInfluenceSetVO billInfluenceSetVO = billInfluenceSetMap.get(influenceId);
                    if (billInfluenceSetVO == null) {
                        throw new BusinessException("会计平台配置-影响因素【ID：" + subjectContrastDetailInfluenceVO.getInfluenceId() + "】未配置关联字段");
                    }
                    String columnExpression = billInfluenceSetVO.getColumnExpression();
                    String contrastExpression = columnExpression + OperatorType.EQ.getToken() + influenceValue;
                    contrastExpressionList.add(contrastExpression);
                }
                String contrastExpression = String.join(OperatorType.AND.getToken(), contrastExpressionList);
                boolean contrastBool = AviatorUtil.exec(contrastExpression, billData);
                log.info("==========分录科目转化条件(H)：{}，结果：{}", contrastExpression, contrastBool);
                if (contrastBool) {
                    originVoucherEntryVO.setSubjectId(subjectContrastDetailVO.getAccountSubjectId());
                    originVoucherEntryVO.setSubjectCode(subjectContrastDetailVO.getAccountSubjectCode());
                    originVoucherEntryVO.setSubjectName(subjectContrastDetailVO.getAccountSubjectName());
                    break;
                }
            }
            this.analysisAuxiliary(originVoucherEntryVO, auxiliarySetMap, billData);
            entriesList.add(originVoucherEntryVO);
        } else {
            String _subTableKey = "subData-" + subTableKey;
            JSONArray subDatas = new JSONArray();
            // 是否存在镜像遍历数据
            if (billData.containsKey(_subTableKey)) {
                subDatas = (JSONArray) billData.get(_subTableKey);
            } else {
                Object sub = billData.get(subTableKey);
                if (sub != null) {
                    subDatas = (JSONArray) sub;
                    // 放入备份镜像子表数据，避免拆分覆盖丢失子表数据
                    billData.put(_subTableKey, subDatas);
                }
            }
            // 按子表数据一次解析每条数据该入什么科目
            for (Object subData : subDatas) {
                billData.remove(subTableKey);
                billData.put(subTableKey, subData);
                boolean conditionBool = Boolean.TRUE;
                if (StringUtils.isNotEmpty(conditionExpression)) {
                    conditionBool = AviatorUtil.exec(conditionExpression, billData);
                }
                log.info("==========分录条件(B)：{}，结果：{}", conditionExpression, conditionBool);
                if (!conditionBool) continue;

                OriginVoucherEntryVO originVoucherEntryVO = new OriginVoucherEntryVO();
                // 摘要
                String oneSubDescription = descriptionExpression;
                if (StringUtils.isNotEmpty(descriptionExpression) && AccplatConsts.DescriptionFlag.BILL_COLUMN.equals(descriptionFlag)) {
                    oneSubDescription = executeOneExpression(descriptionExpression, billData, billTypeCode, OperatorType.AND.getToken());
                }
                originVoucherEntryVO.setEntryDescription(oneSubDescription);
                // 金额
                BigDecimal headMny = ComputeUtil.toBigDecimal((Object) AviatorUtil.exec(mnyExpression, billData));
                if (AccplatConsts.DirectionFlag.DEBIT.equals(directionFlag)) {
                    originVoucherEntryVO.setDebitOriginalMny(headMny);
                } else {
                    originVoucherEntryVO.setCreditOriginalMny(headMny);
                }
                originVoucherEntryVO.setDirectionFlag(directionFlag);
                originVoucherEntryVO.setRateType(voucherTemplateVO.getRateType());
                originVoucherEntryVO.setCurrencyCode(voucherTemplateVO.getCurrencyCode());
                // 先放入默认科目，有符合条件再替换
                originVoucherEntryVO.setSubjectId(subjectContrast.getDefaultSubjectId());
                originVoucherEntryVO.setSubjectCode(subjectContrast.getDefaultSubjectCode());
                originVoucherEntryVO.setSubjectName(subjectContrast.getDefaultSubjectName());

                // 依据科目转换明细，看是否有符合的
                for (SubjectContrastDetailVO subjectContrastDetailVO : subjectContrastDetailList) {
                    List<SubjectContrastDetailInfluenceVO> subjectContrastDetailInfluenceList = subjectContrastDetailVO.getSubjectContrastDetailInfluenceList();
                    List<String> subContrastExpressionList = new LinkedList<>();
                    // 单个子表单条数据是否符合某个条件的科目
                    for (SubjectContrastDetailInfluenceVO subjectContrastDetailInfluenceVO : subjectContrastDetailInfluenceList) {
                        Long influenceId = subjectContrastDetailInfluenceVO.getInfluenceId();
                        String influenceValue = subjectContrastDetailInfluenceVO.getInfluenceValue();
                        BillInfluenceSetVO billInfluenceSetVO = billInfluenceSetMap.get(influenceId);
                        if (billInfluenceSetVO == null) {
                            throw new BusinessException("会计平台配置-影响因素【ID：" + subjectContrastDetailInfluenceVO.getInfluenceId() + "】未配置关联字段");
                        }
                        String columnExpression = billInfluenceSetVO.getColumnExpression();
                        String eqStringExpression = columnExpression + OperatorType.EQ.getToken() + influenceValue;
                        subContrastExpressionList.add(eqStringExpression);
                    }
                    String subContrastExpression = String.join(OperatorType.AND.getToken(), subContrastExpressionList);
                    boolean subContrastBool = AviatorUtil.exec(subContrastExpression, billData);
                    log.info("==========科目转化条件(B)：{}，结果：{}", subContrastExpression, subContrastBool);
                    if (subContrastBool) {
                        originVoucherEntryVO.setSubjectId(subjectContrastDetailVO.getAccountSubjectId());
                        originVoucherEntryVO.setSubjectCode(subjectContrastDetailVO.getAccountSubjectCode());
                        originVoucherEntryVO.setSubjectName(subjectContrastDetailVO.getAccountSubjectName());
                        break;
                    }
                }
                this.analysisAuxiliary(originVoucherEntryVO, auxiliarySetMap, billData);
                entriesList.add(originVoucherEntryVO);
            }
        }
        log.info("==========解析分录【科目分类编码：" + subjectCategoryCode + "】=====end======================================");
        return entriesList;
    }


    public void analysisAuxiliary(OriginVoucherEntryVO entryVO, Map<Long, BillAuxiliarySetVO> auxiliarySetMap, Map<String, Object> billData) {
        Long subjectId = entryVO.getSubjectId();
        SubjectEntity subjectEntity = subjectService.getById(subjectId);
        String auxiliaryIds = subjectEntity.getAuxiliaryIds();
        if (StringUtils.isNotEmpty(auxiliaryIds)) {
            String[] split = auxiliaryIds.split(",");
            List<AuxiliaryEntity> subjectAuxiliaryList = (List<AuxiliaryEntity>) auxiliaryService.listByIds(Arrays.asList(split));
            if (CollectionUtils.isNotEmpty(subjectAuxiliaryList)) {
                List<OriginVoucherAuxiliaryVO> originVoucherAuxiliaryVOS = new LinkedList<>();
                List<String> auxiliaryInfoList = new LinkedList<>();
                for (AuxiliaryEntity auxiliary : subjectAuxiliaryList) {
                    Long _auxiliaryId = auxiliary.getId();
                    BillAuxiliarySetVO billAuxiliarySetVO = auxiliarySetMap.get(_auxiliaryId);
                    if (billAuxiliarySetVO == null) {
                        throw new BusinessException("会计科目【编码：" + subjectEntity.getSubjectCode() + "】未设置辅助核算项【编码：" + auxiliary.getAuxiliaryCode() + "，名称：" + auxiliary.getAuxiliaryName() + "】关联字段");
                    }
                    String columnExpression = billAuxiliarySetVO.getColumnExpression();
                    String referCode = auxiliary.getReferCode();
                    Object auxiliaryIdObj = AviatorUtil.exec(columnExpression, billData);
                    JSONObject auxiliaryValJson = getReferCodeValue(String.valueOf(auxiliaryIdObj), referCode);
                    OriginVoucherAuxiliaryVO originVoucherAuxiliaryVO = new OriginVoucherAuxiliaryVO();
                    originVoucherAuxiliaryVO.setAuxiliaryId(_auxiliaryId);
                    originVoucherAuxiliaryVO.setAuxiliaryCode(auxiliary.getAuxiliaryCode());
                    originVoucherAuxiliaryVO.setAuxiliaryName(auxiliary.getAuxiliaryName());
                    originVoucherAuxiliaryVO.setAuxiliaryValueId(auxiliaryValJson.getLong("id"));
                    String codeVal = auxiliaryValJson.getString("code");
                    String nameVal = auxiliaryValJson.getString("name");
                    originVoucherAuxiliaryVO.setAuxiliaryValueCode(codeVal);
                    originVoucherAuxiliaryVO.setAuxiliaryValueName(nameVal);
                    auxiliaryInfoList.add(codeVal + "/" + nameVal);
                    // 这里默认放入工程云辅助项编码和值编码，后续在合并分录转化财务系统档案编码和值编码
                    originVoucherAuxiliaryVO.setCompareAppId(auxiliary.getCompareAppId());
                    originVoucherAuxiliaryVO.setAuxiliaryFsCode(auxiliary.getAuxiliaryCode());
                    originVoucherAuxiliaryVO.setAuxiliaryFsValue(codeVal);
                    originVoucherAuxiliaryVOS.add(originVoucherAuxiliaryVO);
                }
                entryVO.setAuxiliaryIds(subjectEntity.getAuxiliaryIds());
                entryVO.setAuxiliaryInfo(String.join(";", auxiliaryInfoList));
                entryVO.setOriginVoucherAuxiliaryVOList(originVoucherAuxiliaryVOS);
            }
        }
    }


    public JSONObject getReferCodeValue(String value, String referCode) {
        try {
            JSONArray jArray = ReferObjectUtil.getReferEntityValue(value, referCode);
            if (jArray != null && jArray.size() > 0) {
                JSONObject json = (JSONObject) jArray.get(0);
                return json;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * @description: 依据原始凭证转化财务凭证
     *
     * @author songlx
     * @date: 2023/11/29
     */
    public Map<String, Object> transferFinanceVoucherByOriginVoucher(VoucherParams voucherParams) {
        String systemCode = voucherParams.getFinanceSystemCode();
        IFinanceVoucherService financeVoucherService = voucherTransferFactory.getFinanceVoucherService(systemCode);
        return financeVoucherService.convertToFinanceVoucherByOriginVoucher(voucherParams);
    }


    /**
     * @description: 获取账簿
     */
    public AccbookVO getAccbook(String code) {
        return accbookService.getOneByCode(code);
    }


    /**
     * @description: 根据科目表编码和科目分类编码获取科目对照
     *
     * @param subjectCategoryCode
     * @return {@link SubjectContrastVO}
     * @author songlx
     * @date: 2023/12/20
     */
    public SubjectContrastVO getSubjectContrast(String subjectCategoryCode) {
        return subjectContrastService.getOneBySubjectCategoryCode(subjectCategoryCode);
    }

    /**
     * @description: 获取固定科目
     *
     * @return {@link SubjectVO}
     * @author songlx
     * @date: 2023/12/20
     */
    public SubjectVO getSubjectById(Long subjectId) {
        SubjectEntity one = subjectService.selectById(subjectId);
        return BeanMapper.map(one, SubjectVO.class);
    }

    /**
     * @description: 获取固定科目
     *
     * @return {@link SubjectVO}
     * @author songlx
     * @date: 2023/12/20
     */
    public SubjectVO getSubject(Long subjectChartId, String subjectCode) {
        LambdaQueryWrapper<SubjectEntity> subjectEntityLambdaQueryWrapper = new LambdaQueryWrapper<>();
        subjectEntityLambdaQueryWrapper.eq(SubjectEntity::getSubjectChartId, subjectChartId);
        subjectEntityLambdaQueryWrapper.eq(SubjectEntity::getSubjectCode, subjectCode);
        subjectEntityLambdaQueryWrapper.eq(SubjectEntity::getTenantId, InvocationInfoProxy.getTenantid());
        SubjectEntity one = subjectService.getOne(subjectEntityLambdaQueryWrapper);
        if (one == null) {
            throw new BusinessException("会计科目【编码:" + subjectCode + "，科目表ID：" + subjectChartId + "】不存在！");
        }
        return BeanMapper.map(one, SubjectVO.class);
    }


    /**
     * @description: 获取辅助核算项关联
     *
     * @param billTypeCode
     * @return {@link List< BillAuxiliarySetVO>}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillAuxiliarySetVO> getBillAuxiliarySet(String billTypeCode) {
        LambdaQueryWrapper<BillAuxiliarySetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillAuxiliarySetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillAuxiliarySetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BillAuxiliarySetEntity> list = billAuxiliarySetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【辅助核算项关联】");
        }
        return BeanMapper.mapList(list, BillAuxiliarySetVO.class);
    }


    /**
     * @description: 获取影响因素关联
     *
     * @param billTypeCode
     * @return {@link List< BillInfluenceSetVO>}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillInfluenceSetVO> getBillInfluenceSet(String billTypeCode, List<Long> influenceIds) {
        LambdaQueryWrapper<BillInfluenceSetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillInfluenceSetEntity::getBillTypeCode, billTypeCode);
        wrapper.in(CollectionUtils.isNotEmpty(influenceIds), BillInfluenceSetEntity::getInfluenceId, influenceIds);
        wrapper.eq(BillInfluenceSetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BillInfluenceSetEntity> list = billInfluenceSetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未获取到【影响因素关联】");
        }
        return BeanMapper.mapList(list, BillInfluenceSetVO.class);
    }


    /**
     * @description: 获取当前组织的账簿，递归最近上级获取
     *
     * @param orgId
     * @param billTypeCode
     * @param billData
     * @return {@link BillAccbookSetVO}
     * @author songlx
     * @date: 2023/12/18
     */
    public List<BillAccbookSetVO> getBillAccbookSet(Long orgId, String billTypeCode, Map<String, Object> billData) {
        CommonResponse<OrgVO> oneById = orgApi.getOneById(orgId);
        OrgVO orgVO = oneById.getData();
        String[] parentOrgIds = orgVO.getInnerCode().split("\\|");

        LambdaQueryWrapper<BillAccbookSetEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BillAccbookSetEntity::getBillTypeCode, billTypeCode);
        wrapper.eq(BillAccbookSetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        wrapper.in(BillAccbookSetEntity::getSourceOrgId, parentOrgIds);
        List<BillAccbookSetEntity> list = billAccbookSetService.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("未设置【账簿映射设置】, orgId:" + orgId);
        }
        HashSet<BillAccbookSetVO> accbookSetVOS = new HashSet<>();
        for (BillAccbookSetEntity billAccbookSetEntity : list) {
            Boolean conditionBool = Boolean.TRUE;
            String conditionExpression = billAccbookSetEntity.getConditionExpression();
            if (StringUtils.isNotEmpty(conditionExpression)) {
                conditionBool = executeOneExpression(conditionExpression, billData, billTypeCode, OperatorType.AND.getToken());
            }
            if (conditionBool) {
                accbookSetVOS.add(BeanMapper.map(billAccbookSetEntity, BillAccbookSetVO.class));
            }
        }
        if (CollectionUtils.isEmpty(accbookSetVOS)) {
            throw new BusinessException("未获取到符合条件的【账簿映射设置】, orgId:" + orgId);
        }
        return accbookSetVOS.stream().collect(Collectors.toList());
    }


    public static void main(String[] args) {
        // 创建 Aviator 表达式
        String expressionString = "'x:' + x";

        // 编译表达式
        Expression expression = AviatorEvaluator.getInstance().compile(expressionString);

        // 定义变量
        Map<String, Object> variables = new HashMap<>();
        variables.put("x", 10);
        variables.put("y", "10");

        // 第一次执行表达式
        Object result1 = expression.execute(variables);
        System.out.println("Result 1: " + result1); // 输出 Result 1: 15

        // 修改变量值
        variables.put("x", 20);
        variables.put("y", 8);

        // 第二次执行表达式
        Object result2 = expression.execute(variables);
        System.out.println("Result 2: " + result2); // 输出 Result 2: 28

        String voucherType = "01:记账";
        String[] split = voucherType.split(":|：");
        for (String s : split) {
            System.out.println(s);
        }
    }

}
