package com.ejianc.business.voucher.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.accplat.config.bean.BillVoucherSetEntity;
import com.ejianc.business.accplat.config.service.IBillVoucherSetService;
import com.ejianc.business.accplat.config.vo.BillVoucherSetVO;
import com.ejianc.business.accplat.originvoucher.bean.OriginVoucherEntity;
import com.ejianc.business.accplat.originvoucher.service.IOriginVoucherService;
import com.ejianc.business.accplat.originvoucher.vo.OriginVoucherVO;
import com.ejianc.business.voucher.consts.VoucherBusinessConsts;
import com.ejianc.business.voucher.consts.VoucherFlag;
import com.ejianc.business.voucher.transfer.IVoucherTransferService;
import com.ejianc.business.voucher.transfer.VoucherTransferFactory;
import com.ejianc.business.voucher.utils.DataConvertUtil;
import com.ejianc.business.voucher.vo.VoucherApiQueryVO;
import com.ejianc.business.voucher.vo.VoucherInfo;
import com.ejianc.business.voucher.vo.VoucherParams;
import com.ejianc.business.voucher.vo.VoucherSimpleVO;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.googlecode.aviator.lexer.token.OperatorType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

/**
 * @description: VoucherBusinessService
 * @author songlx
 * @date 2023/11/28
 * @version 1.0
 */
@Slf4j
@Service
public class VoucherBusinessService extends BaseVoucherBusinessService {


    @Autowired
    private VoucherTransferFactory voucherTransferFactory;

    @Autowired
    private IBillVoucherSetService billVoucherSetService;

    @Autowired
    private IOriginVoucherService originVoucherService;

    /**
     * @description: 保存凭证
     * 1、构建凭证参数
     * 2、保存原始凭证，可以根据参数动态判断是否需要保存原始凭证
     * 3、推送财务凭证
     * @param voucherParams
     * @return {@link CommonResponse<VoucherInfo>}
     * @author songlx
     * @date: 2023/11/28
     */
    public CommonResponse<VoucherInfo> save(VoucherParams voucherParams) {
        BillTypeVO billTypeVO = voucherParams.getBillTypeVO();
        Map<String, Object> billData = voucherParams.getBillData();
        billData.put("billTypeCode", billTypeVO.getBillCode());
        billData.put("billTypeName", billTypeVO.getBillName());

        // 凭证模板转化规则，转化数据
        BillVoucherSetVO billVoucherSetVO = voucherParams.getBillVoucherSetVO();
        String serviceName = StringUtils.isEmpty(billVoucherSetVO.getCustomTransferServiceName()) ? VoucherBusinessConsts.COMMON_VOUCHER_TRANSFER_SERVICE : billVoucherSetVO.getCustomTransferServiceName();
        IVoucherTransferService voucherTransferService = voucherTransferFactory.getVoucherTransferService(serviceName);
        voucherTransferService.transfer(voucherParams);

        // 1.3 校验转化后的参数正确性
        checkVoucherParams(voucherParams);
        log.info(" ============================= ");
        log.info(" 业务参数(构建后)voucherParams:" + DataConvertUtil.objToString(voucherParams));
        log.info(" ============================= ");

        List<OriginVoucherVO> originVoucherVOList = voucherParams.getOriginVoucherVOList();
        boolean financeflag = true;
        Boolean isGenerateFinanceVoucherFlag = voucherParams.getIsGenerateFinanceVoucherFlag();
        CommonResponse<VoucherInfo> response = CommonResponse.success();
        // 推送财务凭证
        if (isGenerateFinanceVoucherFlag) {
            response = getFinanceVoucherService(voucherParams.getFinanceSystemCode()).save(voucherParams);
            financeflag = response.isSuccess();
            VoucherInfo voucherInfo = response.getData();
            // 财务凭证相关，在保存财务凭证后放入： 财务凭证状态financeVoucherFlag、财务凭证ID financeVoucherId、财务凭证号financeVoucherCode、生成财务凭证回参 financeVoucherInfo
            if (financeflag) {
                List<VoucherSimpleVO> vouchers = voucherInfo.getVouchers();
                // 当生成多个凭证时，是否真的理想化按顺序对应生成呢？？
                for (int i = 0; i < vouchers.size(); i++) {
                    VoucherSimpleVO voucherSimpleVO = vouchers.get(i);
                    OriginVoucherVO originVoucherVO = originVoucherVOList.get(i);
                    originVoucherVO.setFinanceVoucherFlag(voucherInfo.getVoucherFlag());
                    originVoucherVO.setFinanceVoucherId(voucherSimpleVO.getFinanceVoucherId());
                    originVoucherVO.setFinanceVoucherCode(voucherSimpleVO.getFinanceVoucherCode());
                    originVoucherVO.setFinanceVoucherInfo(DataConvertUtil.objToString(voucherInfo));
                }
            } else {
                for (OriginVoucherVO originVoucherVO : originVoucherVOList) {
                    originVoucherVO.setFinanceVoucherFlag(VoucherFlag.FAILED);
                    originVoucherVO.setFinanceVoucherInfo(response.getMsg());
                }
            }
        }
        // 保存原始凭证。标识为是并且财务凭证需要生成切生成成功
        if (voucherParams.getIsGenerateOriginVoucherFlag() && financeflag) {
            originVoucherService.saveOriginVouchers(originVoucherVOList);
        }
        return response;
    }


    /**
     * @description: 查询凭证
     *
     * @param voucherInfo
     * @return {@link CommonResponse<List<OriginVoucherVO>>}
     * @author songlx
     * @date: 2023/11/30
     */
    public CommonResponse<List<OriginVoucherVO>> get(VoucherInfo voucherInfo) {
        return getFinanceVoucherService(voucherInfo.getFinanceSystemCode()).get(voucherInfo);
    }

    /**
     * @description: 删除凭证
     *
     * @param voucherInfo
     * @return {@link CommonResponse}
     * @author songlx
     * @date: 2023/11/30
     */
    public CommonResponse del(VoucherInfo voucherInfo) {
        CommonResponse response = getFinanceVoucherService(voucherInfo.getFinanceSystemCode()).del(voucherInfo);
        boolean success = response.isSuccess();
        if (success) {
            Long billId = voucherInfo.getBillId();
            if (billId != null) {
                LambdaQueryWrapper<OriginVoucherEntity> wrapper = new LambdaQueryWrapper<>();
                wrapper.eq(OriginVoucherEntity::getSrcBillId, billId);
                originVoucherService.remove(wrapper);
            }
        }
        return response;
    }


    /**
     * @description: 根据单据类型编码和单据数据查询单据设置判断是否需要传凭证
     *
     * @param voucherParams
     * @return {@link Boolean}
     * @author songlx
     * @date: 2023/12/1
     */
    public Boolean isGenerateVoucher(VoucherParams voucherParams) {
        String billTypeCode = voucherParams.getBillTypeCode();
        Long tenantid = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<BillVoucherSetEntity> voucherSetEntityLambdaQueryWrapper = new LambdaQueryWrapper<>();
        voucherSetEntityLambdaQueryWrapper.eq(BillVoucherSetEntity::getTenantId, tenantid);
        voucherSetEntityLambdaQueryWrapper.eq(BillVoucherSetEntity::getBillTypeCode, billTypeCode);
        BillVoucherSetEntity voucherSetEntity = billVoucherSetService.getOne(voucherSetEntityLambdaQueryWrapper);
        if (voucherSetEntity == null) {
            throw new BusinessException("该单据【单据类型编码：" + billTypeCode + "】未设置传会计凭证参数！");
        }
        StringBuilder msgBuilder = new StringBuilder();
        Boolean enabled = voucherSetEntity.getEnabled().intValue() == 1;
        msgBuilder.append("`传会计凭证启用标识：" + enabled + "`");
        String conditionExpression = voucherSetEntity.getConditionExpression();
        if (StringUtils.isNotBlank(conditionExpression)) {
            Boolean condiBool = executeOneExpression(conditionExpression, voucherParams.getBillData(), billTypeCode, OperatorType.AND.getToken());
            msgBuilder.append("`条件表达式：" + conditionExpression + "，结果：" + condiBool + "`");
            enabled = enabled && condiBool;
        }
        voucherParams.setIsGenerateOriginVoucherFlag(enabled);
        voucherParams.setIsGenerateFinanceVoucherFlag(enabled);
        voucherParams.setBillVoucherSetVO(BeanMapper.map(voucherSetEntity, BillVoucherSetVO.class));
        voucherParams.setMsgInfo(msgBuilder.toString());
        return enabled;
    }


    /**
     * @description: 根据财务系统标识获取对应的财务会计服务
     * @author songlx
     * @date: 2023/11/30
     */
    private IFinanceVoucherService getFinanceVoucherService(String systemCode) {
        return voucherTransferFactory.getFinanceVoucherService(systemCode);
    }

    private void checkVoucherParams(VoucherParams voucherParams) {
        Boolean isGenerateOriginVoucherFlag = voucherParams.getIsGenerateOriginVoucherFlag();
        Boolean isGenerateFinanceVoucherFlag = voucherParams.getIsGenerateFinanceVoucherFlag();
        List<OriginVoucherVO> originVoucherVOList = voucherParams.getOriginVoucherVOList();
        Map<String, Object> financeVoucher = voucherParams.getFinanceVoucher();
        if (isGenerateOriginVoucherFlag && CollectionUtils.isEmpty(originVoucherVOList)) {
            throw new BusinessException("凭证构建参数校验错误：生成原始凭证标识为是，但缺乏原始凭证数据[originVoucherVOList]");
        }
        if (isGenerateFinanceVoucherFlag && financeVoucher == null) {
            throw new BusinessException("凭证构建参数校验错误：生成财务凭证标识为是，缺乏财务凭证参数[financeVoucher]");
        }
    }

    public CommonResponse<OriginVoucherVO> getOne(VoucherApiQueryVO voucherApiQueryVO) {
        return getFinanceVoucherService(voucherApiQueryVO.getFinanceSystemCode()).getOne(voucherApiQueryVO);

    }
}
