package com.ejianc.framework.cache;

import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.cache.redissonlock.RedissonLocker;
import com.ejianc.framework.cache.utils.LockUtil;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.redisson.config.SingleServerConfig;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.jedis.JedisClientConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springside.modules.nosql.redis.JedisTemplate;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

@Configuration
@ConfigurationProperties(prefix = "ejc.redis")
public class CacheConfiguration {

	private final static String REDIS_DIRECT = "direct://";

	private int maxTotal = 100; // 最大连接数
	private int maxIdle = 100; // 最大空闲连接数
	private int minIdle = 0; // 最小空闲连接数
	private long maxWaitMillis = 30000; // 最大建立连接等待时间
	private boolean testOnBorrow = true; // 获取连接时检查有效性
	private boolean testOnReturn = true;
	private boolean testWhileIdle = false; // 空闲时检查有效性
	private String hostName = "127.0.0.1"; // 主机名
	private int port = 6379; // 监听端口
	private String password; // 密码
	private int timeout = 10000; // 客户端连接时的超时时间（单位为秒）
	private int database = 0;

	/**
	 * 构造jedis连接池配置对象
	 */
	@Bean
	public GenericObjectPoolConfig<Jedis> getJedisPoolConfig() {
		GenericObjectPoolConfig<Jedis> config = new GenericObjectPoolConfig<>();
		if(maxTotal <= 100) {
			config.setMaxTotal(100);
			config.setMaxIdle(100);
		}else{
			config.setMaxTotal(maxTotal);
			config.setMaxIdle(maxIdle);
		}
		config.setMinIdle(minIdle);
		config.setMaxWaitMillis(maxWaitMillis);
		config.setTestOnBorrow(testOnBorrow);
		config.setTestOnReturn(testOnReturn);
		config.setTestWhileIdle(testWhileIdle);
		config.setMinEvictableIdleTimeMillis(60000);
		config.setJmxEnabled(false);
		return config;
	}

	/**
	 * 构造jedis连接池
	 */
	@Bean
	public JedisPool getJedisPool() {
		return new JedisPool(getJedisPoolConfig(), hostName, port, timeout, password, database);
	}

	/**
	 * 注入redis template
	 */
	@Bean
	@ConditionalOnMissingBean
	public RedisTemplate<String, Object> getRedisTemplate() {
		return new RedisTemplate<>(getJedisPool());
	}

	/**
	 * 初始化jedis连接池
	 */
	@Bean("jedisPool")
	public JedisPool getRedisPoolFactory() {
		JedisPool jedisPool = new JedisPool(getJedisPoolConfig(), hostName, port, 5000, password);
		return jedisPool;
	}

	/**
	 * 初始化CacheManager
	 */
	@Bean
	public CacheManager getCacheManager() {
		CacheManager cacheManager = new CacheManager();
		cacheManager.setJedisPool(getRedisPoolFactory());
		return cacheManager;
	}

	/**
	 * RedissonClient, 单机
	 * 
	 * @return
	 */
	@Bean(destroyMethod = "shutdown")
	public RedissonClient getRedissonClient() {
		Config config = new Config();
        SingleServerConfig serverConfig = config.useSingleServer()
                .setAddress("redis://"+hostName+":"+port)
                .setTimeout(timeout)
                .setPingConnectionInterval(10000)
                .setDatabase(0);
        if(StringUtils.isNotBlank(password)) {
            serverConfig.setPassword(password);
        }
        
        return Redisson.create(config);
	}
	
	@Bean
	public RedissonLocker getRedissonLocker() {
		RedissonLocker locker = new RedissonLocker(getRedissonClient());
		LockUtil.setLocker(locker);
		return locker;
	}
	
	public void setMaxTotal(int maxTotal) {
		this.maxTotal = maxTotal;
	}

	public void setMaxIdle(int maxIdle) {
		this.maxIdle = maxIdle;
	}

	public void setMinIdle(int minIdle) {
		this.minIdle = minIdle;
	}

	public void setMaxWaitMillis(long maxWaitMillis) {
		this.maxWaitMillis = maxWaitMillis;
	}

	public void setTestOnBorrow(boolean testOnBorrow) {
		this.testOnBorrow = testOnBorrow;
	}

	public void setTestWhileIdle(boolean testWhileIdle) {
		this.testWhileIdle = testWhileIdle;
	}

	public void setHostName(String hostName) {
		this.hostName = hostName;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public void setTimeout(int timeout) {
		this.timeout = timeout;
	}

	public void setDatabase(int database) {
		this.database = database;
	}

}
