package org.springframework.data.redis.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.Collection;
import java.util.concurrent.TimeUnit;

public class RedisTemplate<K, V>  {
    private static Logger logger = LoggerFactory.getLogger(RedisTemplate.class);
    private JedisPool jedisPool;

    public RedisTemplate() {
    }

    public RedisTemplate(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    public boolean delete(K key) {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            return jedis.del(key.toString()) > 0;
        }catch (Exception e) {
            logger.error("Jedis exception happen.", e.getMessage());
        }finally {
            jedisPool.returnResource(jedis);
        }
        return false;
    }

    public Long delete(Collection<K> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return 0L;
        }
        Jedis jedis = null;
        try {
            Long count = 0L;
            jedis = jedisPool.getResource();
            for(K key : keys) {
                jedis.del(key.toString());
                count += 1;
            }
            return count;
        }catch (Exception e) {
            logger.error("Jedis exception happen.", e.getMessage());
        } finally {
            jedisPool.returnResource(jedis);
        }
        return 0L;
    }

    public <HK, HV> HashOperations<K, HK, HV> opsForHash() {
        return new DefaultHashOperations<>();
    }

    public ValueOperations<K, V> opsForValue() {
        return new DefaultValueOperations<>();
    }


    public class DefaultHashOperations<H, HK, HV> implements HashOperations<H, HK, HV> {

        public void put(H key, HK hashKey, HV value) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                jedis.hset(key.toString(), hashKey.toString(), value.toString());
            }catch (Exception e) {
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
            }
        }

        public HV get(H key, Object hashKey) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                return (HV) jedis.hget(key.toString(), hashKey.toString());
            }catch (Exception e){
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
            }
            return null;
        }

        public Boolean hasKey(H key, Object hashKey) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                return jedis.hexists(key.toString(), hashKey.toString());
            }catch (Exception e){
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
            }
            return false;
        }

        public void hdel(H key, Object hashKey) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                jedis.hdel(key.toString(), hashKey.toString());
            }catch (Exception e) {
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
            }
        }

        public void hdel(H key, Collection<Object> hashKey) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                for(Object o : hashKey) {
                    jedis.hdel(key.toString(), o.toString());
                }
            }catch (Exception e) {
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
            }
        }

    }

    public class DefaultValueOperations<K, V>  implements ValueOperations<K, V> {
        public V get(Object key) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                return (V) jedis.get(key.toString());
            }catch (Exception e) {
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
            }
            return null;
        }

        public void set(K key, V value) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                jedis.set(key.toString(), value.toString());
            }catch (Exception e) {
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
            }
        }

        public void set(K key, V value, long timeout, TimeUnit unit) {
            Jedis jedis = null;
            try {
                jedis = jedisPool.getResource();
                jedis.setex(key.toString(), (int)unit.toSeconds(timeout), value.toString());
            }catch (Exception e) {
                logger.error("Jedis exception happen.", e.getMessage());
            }finally {
                jedisPool.returnResource(jedis);
           }
        }
    }

}
