package com.ejianc.business.assist.rmat.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.assist.rmat.enums.SupplierSignStatusEnum;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.foundation.share.api.IProSupplierApi;
import com.ejianc.foundation.share.api.IShareCooperateApi;
import com.ejianc.foundation.share.utils.FileUtil;
import com.ejianc.foundation.share.vo.CooperateVO;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.refer.constants.MetaDataUrlconstants;
import com.ejianc.framework.skeleton.refer.util.ContextUtil;
import com.ejianc.framework.skeleton.refer.util.ExceptionUtil;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import javax.servlet.http.HttpServletRequest;
import java.io.InputStream;
import java.util.*;
import java.util.concurrent.TimeUnit;

@Component("assistPushSupUtil")
public class PushSupUtil<T> {
    private final Logger logger = LoggerFactory.getLogger(getClass());

    private static final String REFER_CACHE_KEY = "refer_cache_key:";

    /** 无权访问 */
    private static final String noPower = "无权限访问该服务，请先联系管理员进行授权！";

    @Autowired
    private JedisPool jedisPool;

    @Autowired
    private IPushMessageApi pushMessageApi;

    @Autowired
    private IShareCooperateApi shareCooperateApi;

    @Autowired
    private IProSupplierApi proSupplierApi;

    @Autowired
    private IAttachmentApi attachmentApi;

    @Autowired
    private ISystemDataPushService systemDataPushService;

    /**
     * 单据推送到供方协同服务
     * @param entity    待推送单据
     * @param OPERATE   redis请求标识
     * @param code      单据类型主键
     * @param name      单据类型名称
     * @param url       推送URL
     * @return
     */
    public boolean pushBillToSupCenter(JSONObject entity, String OPERATE, String code, String name, String url) {
        String id = entity.getString("id");
        String supplierId = entity.getString("supplierId");
        //查询该单据是否支持协同分享，则向供方协同服务推送该单据
        CommonResponse<CooperateVO> resp = shareCooperateApi.queryCooperateBybillTypeCode(code);
        if (!resp.isSuccess()) {
            logger.error("根据单据类型-{}查询其协同配置信息失败，{}", code, resp.getMsg());
            return false;
        }
        CooperateVO cooperate = resp.getData();
        if(new Integer(0).equals(cooperate.getShareFlag())){
            logger.info("根据单据类型-{}查询，单据-{}协同配置不支持协同分享", code, name);
            return true;
        }

        boolean locked = false, syncFlag = false;
        Jedis jedis = jedisPool.getResource();
        String key = code + "::" + id;

        //设置单据当前系统信息
        CommonResponse<String> systemCodeResp = proSupplierApi.getEjcCloudSystemCode();
        if (!systemCodeResp.isSuccess()) {
            logger.error("推送单据-{}-id{}失败，获取当前系统编码失败,{}", name, id, systemCodeResp.getMsg());
            return false;
        }
        entity.put("sourceSystemId", systemCodeResp.getData());//设置当前系统ID
        //去除租户主键
        if(entity.containsKey("tenantId")){
            entity.remove("tenantId");
        }
        try {
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("transData", JSONObject.toJSONString(entity));

            //查询单据附件信息并下载
            CommonResponse<List<AttachmentVO>> fileResp = attachmentApi.queryListBySourceId(Long.valueOf(id), null, null, null);
            if (!fileResp.isSuccess()) {
                logger.error("获取单据-{}id-{}对应附件信息失败, {}", name, id, fileResp.getMsg());
                return false;
            }
            List<AttachmentVO> fileList = fileResp.getData();

            Map<String, String> sourceTypeMap = new HashMap<>();
            List<Long> fileIds = new ArrayList<>();
            //从附件信息列表获取到： 1、附件名对应附件业务类型Map,2、获取到附件Id列表
            for (AttachmentVO attach : fileList) {
                sourceTypeMap.put(attach.getFileName(), attach.getSourceType());
                fileIds.add(attach.getId());
            }
            paramMap.put("nameSourceTypeMapping", JSONObject.toJSONString(sourceTypeMap));

            //当前单据携带有附件信息
            Map<String, Map<String, InputStream>> files = new HashMap<>();
            if (CollectionUtils.isNotEmpty(fileList)) {
                Map<String, InputStream> fileMap = FileUtil.getInstance().batchDownFileFlow(fileIds, true);
                fileMap.keySet().stream().forEach(fileKey -> {
                    Map<String, InputStream> file = new HashMap<>(1);
                    file.put(fileKey, fileMap.get(fileKey));
                    files.put(fileKey, file);
                });
            }
            logger.info("向供应商-{}推送单据-{}参数-{}", supplierId, name, JSONObject.toJSONString(paramMap));
            //推送单据到指定的供方
            CommonResponse<String> syncReqResp = systemDataPushService.exchangeDataAndFilesWithEachLinkSystem(url, paramMap, supplierId, files);
            if (!syncReqResp.isSuccess()) {
                logger.error("发送请求推送单据-{}id-{}给供方id-{}失败, {}", name, id, supplierId, syncReqResp.getMsg());
                return false;
            }
            if(noPower.equals(syncReqResp.getData())){
                logger.error("发送请求URL-{}给供方id-{}失败, {}", url, supplierId, syncReqResp.getData());
                return false;
            }
            CommonResponse<String> billPushResp = JSONObject.parseObject(syncReqResp.getData(), CommonResponse.class);
            if (billPushResp.isSuccess()) {
                syncFlag = true;
            } else {
                logger.error("供方id-{}处理推送单据-{}id-{}失败, {}", supplierId, name, id, billPushResp.getMsg());
            }
        } catch (Exception e) {
            logger.error("推送单据-{}id-{}给供方id-{} 异常，", name, id, supplierId, e);
        } finally {
            if (locked) {
                //释放单据锁
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
            jedis.close();
        }
        return syncFlag;
    }

    /**
     * 同步单据供应商签字信息
     * @param request    请求头
     * @param entity    待推送单据
     * @param mainClass 待推送单据类型
     * @param OPERATE   redis请求标识
     * @param code      单据类型主键
     * @param name      单据类型名称
     * @return
     */
    public String updateBillSupSignSyncInfo(HttpServletRequest request, JSONObject entity, Class<?> mainClass, String OPERATE, String code, String name) {
        if(entity == null) {
            logger.error("单据id-{}查询失败，未查询到单据！", entity.getString("id"));
            return "未查询到单据";
        }
        String authority = request.getHeader("authority");
        String billId = request.getParameter("billId");
        String supOperatorName = request.getParameter("supOperatorName");
        String supOperatorPhone = request.getParameter("supOperatorPhone");
        String supOperatorUserCode = request.getParameter("supOperatorUserCode");
        Long time = Long.valueOf(request.getParameter("supOperateTime"));
        Date supOperateTime = time != null ? new Date(time) : null;
        String nameSourceTypeMapping = request.getParameter("nameSourceTypeMapping");
        Map<String, String> map = JSONObject.parseObject(nameSourceTypeMapping, Map.class);
        logger.info("接收到单据签章通知：id-{}, supOperatorName-{}, supOperatorPhone-{}, supOperatorUserCode-{}, supOperateTime-{}, nameSourceTypeMapping-{}",
                billId, supOperatorName, supOperatorPhone, supOperatorUserCode, supOperateTime, nameSourceTypeMapping);

        //设置供方签字信息
        entity.put("supOperateTime", supOperateTime);
        entity.put("supOperatorName", supOperatorName);
        entity.put("supOperatorPhone", supOperatorPhone);
        entity.put("supOperatorUserCode", supOperatorUserCode);

        String msg = null;
        Jedis jedis = null;
        boolean locked = false;
        String key = code + "::" + billId;
        try {
            //对单据进行加锁
            jedis = jedisPool.getResource();
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            if (!locked) {
                logger.error("单据id-{}签字信息回写加锁失败！", billId);
                return "单据签字信息回写加锁失败";
            }
            //保存单据中附件并获取到上传后附件的Id
            List<Long> attchIdsList = new ArrayList<>();
            Map<String, List<Long>> attachIdsMap = FileUtil.getInstance().handleReqFile((MultipartHttpServletRequest) request,
                    map, code, authority, billId.toString());
            for (List<Long> attachIds : attachIdsMap.values()) {
                if (CollectionUtils.isNotEmpty(attachIds)) {
                    attchIdsList.addAll(attachIds);
                }
            }
            entity.put("attachIds", attchIdsList);// 将附件关联在单据中
            entity.put("supplierSignStatus", SupplierSignStatusEnum.乙方已签字.getCode());// 将单据设置为乙方已签字状态
            //更新单据
            this.saveOrUpdate(entity, mainClass);

            //向单据制单人和经办人推送该消息
            String[] channel = new String[]{PushMsgParameter.CHANNEL_TYPE_SYS};
            String createUserCode = entity.getString("createUserCode");
            String createUserId = this.getCreateUserId(createUserCode);
            String employeeId = entity.getString("employeeId");
            String[] receivers = new String[]{createUserId, employeeId};
            String result = this.sendMsg(channel, receivers, "notice","供方已签字提醒", name + "[" + code + "]供方已签字完成");
            if (null != result) {
                logger.error("向用户-{}发送单据id-{}签字提醒失败，原因：{}", StringUtils.join(receivers), billId, result);
            }
        } catch (Exception e) {
            logger.error("单据id-{}签字信息回写异常，", billId, e);
            return "单据签字信息回写失败！";
        } finally {
            if (locked) {
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
            jedis.close();
        }
        return msg;
    }

    private void saveOrUpdate(JSONObject entity, Class<?> mainClass) {
        String className = mainClass.getSimpleName();
        className = className.substring(0,1).toLowerCase()+className.substring(1);
        String serviceName = className.substring(0, className.lastIndexOf("Entity")) + "Service";
        BaseServiceImpl service = ContextUtil.getBean(serviceName, BaseServiceImpl.class);
        service.saveOrUpdate(BeanMapper.map(entity, mainClass), false);
    }

    /**
     * 将推送至供方的单据作废
     * @param entity    待推送单据
     * @param OPERATE   redis请求标识
     * @param code      单据类型主键
     * @param name      单据类型名称
     * @return
     */
    public boolean delPushBill(JSONObject entity, String OPERATE, String code, String name, String url) {
        String id = entity.getString("id");
        String supplierId = entity.getString("supplierId");
        //查询该单据是否支持协同分享，则向供方协同服务推送该单据
        CommonResponse<CooperateVO> resp = shareCooperateApi.queryCooperateBybillTypeCode(code);
        if (!resp.isSuccess()) {
            logger.error("根据单据类型-{}查询其协同配置信息失败，{}", code, resp.getMsg());
            throw new BusinessException("审批回调失败，获取单据协同配置失败！");
        }
        CooperateVO cooperate = resp.getData();
        if(cooperate == null || new Integer(0).equals(cooperate.getShareFlag())){
            logger.info("根据单据类型-{}查询，单据-{}协同配置不支持协同分享", code, name);
            return true;
        }

        boolean locked = false, delSuc = false;
        Jedis jedis = jedisPool.getResource();
        String key = code + "::" + id;

        //设置单据当前系统信息
        CommonResponse<String> ejcCloudSystemCode = proSupplierApi.getEjcCloudSystemCode();
        if (!ejcCloudSystemCode.isSuccess()) {
            logger.error("推送单据-{}id-{}失败，获取当前系统编码失败,{}", name, id, ejcCloudSystemCode.getMsg());
        }
        String sourceSystemId = ejcCloudSystemCode.getData();
        //设置当前系统ID
        entity.put("sourceSystemId", sourceSystemId);
        try {
            //对单据进行加锁
            jedis = jedisPool.getResource();
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            if (!locked) {
                return delSuc;
            }
            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("sourceId", id);
            paramMap.put("sourceSystemId", sourceSystemId);

            //推送单据到指定的供方
            logger.info("单据-{}id-{}弃审，通知供方-{}单据作废!", name, id, supplierId);
            CommonResponse<String> syncReqResp = systemDataPushService.exchangeDataWithEachLinkSystem(url,
                    RequestMethod.POST, JSONObject.toJSONString(paramMap), supplierId);
            if (!syncReqResp.isSuccess()) {
                logger.error("发送请求通知供方-{} 单据id-{}作废失败, {}", supplierId, id, syncReqResp.getMsg());
                return delSuc;
            }
            CommonResponse<String> supHandleResp = JSONObject.parseObject(syncReqResp.getData(), CommonResponse.class);
            if (supHandleResp.isSuccess()) {
                delSuc = true;
            } else {
                logger.error("供方-{}处理作废单据-{}id-{}作废失败, {}", supplierId, name, id, supHandleResp.getMsg());
            }
        } catch (Exception e) {
            logger.error("通知供方单据id-{}作废异常，", id, e);
        } finally {
            if (locked) {
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
            jedis.close();
        }
        return delSuc;
    }

    /**
     * 通知用户消息
     * @param channel   消息发送渠道
     * @param receivers 接收人
     * @param msgType   消息类型
     * @param subject   消息主题
     * @param content   消息内容
     * @return
     */
    public String sendMsg(String[] channel, String[] receivers, String msgType, String subject, String content) {
        logger.info("发送消息开始！===========");
        // 系统管理员特殊处理
        List<String> user = new ArrayList<>(Arrays.asList(receivers));
        Collections.replaceAll(user, "1247777316689256450", "303581417601122400");
        receivers = user.toArray(new String[user.size()]);

        PushMsgParameter parameter = new PushMsgParameter();
        parameter.setReceivers(receivers);// 消息接收人
        parameter.setContent(content);// 消息内容
        parameter.setSubject(subject);// 消息主题
        parameter.setMsgType(msgType);// 消息类型
        parameter.setTenantId(InvocationInfoProxy.getTenantid().toString());
        parameter.setSaveFlag(true);// 消息保存
        parameter.setSendUserId(InvocationInfoProxy.getUserid());// 消息发送人
        parameter.setChannel(channel);// 消息发送渠道

        logger.info("发送信息{}", JSONObject.toJSONString(parameter));
        CommonResponse<String> result = pushMessageApi.pushMessage(parameter);
        if (!result.isSuccess()) {
            logger.error("消息发送失败---------------->" + result.getMsg());
            return result.getMsg();
        }
        logger.error("消息发送成功---------------->" + result.getMsg());
        return "发送成功！";
    }

    /**
     * 根据人员编码查询人员主键
     * @param userCode
     * @return
     */
    public static String getCreateUserId(String userCode) {
        try {
            MetaDataUrlconstants urlconstants = ContextUtil.getBean(MetaDataUrlconstants.class);
            if (StringUtils.isNotBlank(userCode) && StringUtils.isNotBlank(urlconstants.getBaseHost())) {
                String cacheKey = REFER_CACHE_KEY + "idm-employee:" + userCode;
                RedisTemplate<String, Object> redisTemplate = ContextUtil.getBean("redisTemplate", RedisTemplate.class);
                Object cacheValue = redisTemplate.opsForValue().get(cacheKey);
                JSONObject jsonObject = null;
                if (cacheValue != null) {
                    jsonObject = JSON.parseObject(cacheValue.toString());
                } else {
                    String referData = getReferRestUrl("idm-employee");
                    JSONObject json = JSON.parseObject(referData);
                    String referUrl = json.get("projectName").toString();
                    Map<String, Object> reflist = new HashMap<>();
                    reflist = (Map<String, Object>) json.get("refMapList");
                    reflist.put("userCode", userCode);
                    String url = urlconstants.getBaseHost() + referUrl + "/commonrefer/getAuditInfo";
                    String jsonbackstr = null;
                    try {
                        jsonbackstr = ReferHttpClientUtils.getAndHeader(url, reflist);
                    } catch (Exception e) {
                    }
                    if (StringUtils.isNotBlank(jsonbackstr)) {
                        jsonObject = JSON.parseObject(jsonbackstr);
                        redisTemplate.opsForValue().set(cacheKey, jsonbackstr, 5, TimeUnit.DAYS);
                    }
                }
                if (jsonObject != null) {
                    return jsonObject.getString("id");
                }
            }
        } catch (Exception e) {
        }
        return userCode;
    }

    /**
     * 通过元数据服务获取实体信息
     * @param referCode
     * @return
     */
    private static String getReferRestUrl(String referCode) {
        String data = null;
        MetaDataUrlconstants urlconstants = ContextUtil.getBean(MetaDataUrlconstants.class);
        String backData = null;
        try {
            backData = ReferHttpClientUtils.getAndHeader(urlconstants.getMetaDataBaseUrl() + referCode, null);
        } catch (Exception e) {
        }
        JSONObject jsonobject = JSON.parseObject(backData);
        if (jsonobject.getString("code").equals("0")) {
            if (jsonobject.getString("data") != null) {
                data = JSON.parseObject(jsonobject.getString("data")).toJSONString();
            }
        } else {
            ExceptionUtil.wrappBusinessException("参照编码" + referCode + "的实体没有发布元数据!");
        }
        return data;
    }
}
