package com.ejianc.business.assist.store.service.handler;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.assist.store.bean.FlowEntity;
import com.ejianc.business.assist.store.service.IFlowService;
import com.ejianc.business.assist.store.service.IInOutService;
import com.ejianc.business.assist.store.service.IStoreService;
import com.ejianc.business.assist.store.service.ISurplusService;
import com.ejianc.business.assist.store.util.StoreLockUtil;
import com.ejianc.business.assist.store.util.StoreManageUtil;
import com.ejianc.business.assist.store.vo.FlowVO;
import com.ejianc.business.assist.store.vo.StoreManageVO;
import com.ejianc.business.assist.store.vo.SurplusUpdateVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 退货处理器
 * @date 2022/1/25
 */
@Service
public class BackGoodsHandler implements IStoreManageHandler {

    @Autowired
    IFlowService flowService;

    @Autowired
    ISurplusService surplusService;

    @Autowired
    IInOutService inOutService;

    @Autowired
    IStoreService storeService;

    /**
     * @param storeManageVO
     * @description: 退货具体实现逻辑, 退货审批通过后调用
     * 出库  先进先出
     * 0 校验库存量
     * 1 生成负数入库流水
     * 2 更新库存
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.assist.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handle(StoreManageVO storeManageVO) {
        //设置仓库属性
        storeManageVO= storeService.setStoreAttr(storeManageVO);
        Long storeId = storeManageVO.getStoreId();
        List<FlowVO> flowVOList = storeManageVO.getFlowVOList();

        // 1 保存的负数的入库
        flowVOList.forEach(t -> {
            BigDecimal num = ComputeUtil.convertToMinusNumber(t.getNum());
            t.setNum(num);
            t.setSurplusNum(num);
            BigDecimal mny = ComputeUtil.safeMultiply(t.getNum(), t.getPrice());
            t.setMny(mny);
            t.setSurplusMny(mny);
            BigDecimal taxMny = ComputeUtil.safeMultiply(t.getNum(), t.getTaxPrice());
            t.setTaxMny(taxMny);
            t.setSurplusTaxMny(taxMny);

        });

        // 这是退货,正向操作相当于出库,属于逆向操作
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, flowVOList, false);

        Boolean lock = StoreLockUtil.getLock(storeId);

        CommonResponse<SurplusUpdateVO> res;
        try {
            //0 校验库存量
            res = surplusService.validateSurplus(surplusUpdateVO);
            if (res.isSuccess()) {
                // 2 更新库存
                surplusService.updateStoreSurplus(surplusUpdateVO, false, true, true);
                flowService.saveOrUpdateBatch(BeanMapper.mapList(flowVOList, FlowEntity.class));
                return CommonResponse.success(storeManageVO);
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }
        return CommonResponse.error(res.getMsg());
    }

    /**
     * @param storeManageVO
     * @description: 退货回滚实现逻辑
     * 1 删除退货负数流水
     * 2 更新库存增量
     * 若未使用则删除入库并逆向更新库存
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.assist.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handleRollback(StoreManageVO storeManageVO) {
        Long storeId = storeManageVO.getStoreId();
        Long sourceId = storeManageVO.getSourceId();
        QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("store_id", storeId);
        queryWrapper.eq("source_id", sourceId);
        List<FlowEntity> instoreList = flowService.list(queryWrapper);
        List<FlowVO> flowVOList = BeanMapper.mapList(instoreList, FlowVO.class);
        // 负数的入库转化成正数
        flowVOList.forEach(t -> {
            BigDecimal num = ComputeUtil.safeSub(BigDecimal.ZERO, t.getNum());
            t.setNum(num);
            t.setSurplusNum(num);
            BigDecimal mny = ComputeUtil.safeMultiply(t.getNum(), t.getPrice());
            t.setMny(mny);
            t.setSurplusMny(mny);
            BigDecimal taxMny = ComputeUtil.safeMultiply(t.getNum(), t.getTaxPrice());
            t.setTaxMny(taxMny);
            t.setSurplusTaxMny(taxMny);

        });
        //2 更新库存增量 正向增量更新库存
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, flowVOList, false);

        Boolean lock = StoreLockUtil.getLock(storeId);
        try {
            surplusService.updateStoreSurplus(surplusUpdateVO, false, true, true);
            //1 删除退货负数流水
            List<Long> delListIds = flowVOList.stream().map(FlowVO::getId).collect(Collectors.toList());
            flowService.removeByIds(delListIds);

            return CommonResponse.success(storeManageVO);

        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }
    }
}
