package com.ejianc.business.assist.rmat.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.assist.rmat.bean.*;
import com.ejianc.business.assist.rmat.enums.BillTypeEnum;
import com.ejianc.business.assist.rmat.enums.MaterialStateEnum;
import com.ejianc.business.assist.rmat.mapper.RentCalculateMapper;
import com.ejianc.business.assist.rmat.service.*;
import com.ejianc.business.assist.rmat.utils.DateUtil;
import com.ejianc.business.assist.rmat.utils.DetailListUtil;
import com.ejianc.business.assist.rmat.utils.MaterialConstant;
import com.ejianc.business.assist.rmat.utils.ValidateUtil;
import com.ejianc.business.assist.rmat.vo.*;
import com.ejianc.business.common.CommonConstant;
import com.ejianc.foundation.share.api.IShareMaterialApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 租金计算单
 * 
 * @author generator
 * 
 */
@Service("rentCalculateService")
public class RentCalculateServiceImpl extends BaseServiceImpl<RentCalculateMapper, RentCalculateEntity> implements IRentCalculateService{
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IShareMaterialApi materialApi;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IMaterialService materialService;

    @Autowired
    private IContractService contractService;

    @Autowired
    private IRecordService recordService;
    @Autowired
    private IRecordDailyRentService dailyRecordService;
    @Autowired
    private IRecordMonthRentService monthRecordService;

    private static final String BILL_CODE = "ASSIST_RMAT_RENT_CALCULATE";//此处需要根据实际修改
    private static final String BILL_NAME = BillTypeEnum.租金计算单.getName();
    private static final String RENT_TYPE_DAY = "0";
    private static final String RENT_TYPE_MONTH = "1";

    @Override
    public RentCalculateVO saveOrUpdate(RentCalculateVO saveOrUpdateVO) {
        // 同一个合同只能存在一个自由态或审批中的单据
        materialService.validateContract(saveOrUpdateVO.getContractId(), BILL_NAME, saveOrUpdateVO.getId(), MaterialConstant.保存);
        // 校验必须大于最大单据日期
        this.validateTime(saveOrUpdateVO, MaterialConstant.保存);
        RentCalculateEntity entity = BeanMapper.map(saveOrUpdateVO, RentCalculateEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE,InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        super.saveOrUpdate(entity, false);
        return BeanMapper.map(entity, RentCalculateVO.class);
    }

    @Override
    public String validateTime(RentCalculateVO rentVO, String type) {
        Map<String, Object> params = new HashMap<>();
        params.put("contractId", rentVO.getContractId());
        if(rentVO.getId() != null){
            params.put("billType", BILL_NAME);
            params.put("billId", rentVO.getId());
        }
        Date lastDate = materialService.getLastDate(params);
        Map<Date, Date> maxTimeMap = materialService.getMaxTime(params);

        if (lastDate == null) return "未获取最大单据日期！";
        if(ValidateUtil.compareDate(rentVO.getRentDate(), lastDate, maxTimeMap, rentVO.getCreateTime())){
            throw new BusinessException(DateUtil.formatDate(rentVO.getRentDate())
                    + "小于最大单据日期【" + DateUtil.formatDate(lastDate) + "】，不允许" + type + "!");
        }
        return "校验通过！";
    }

    /**
     * 租金自动计算
     *
     * @param vo 需要合同id，租金计算日期
     * @return 计算后结果
     */
    @Override
    public RentCalculateVO automaticRental(RentCalculateVO vo) {
        if (vo == null || vo.getContractId() == null) {
            throw new BusinessException("合同信息不能为空");
        }
        if (vo.getRentDate() == null) {
            throw new BusinessException("租金计算日期不能为空");
        }
        Long contractId = vo.getContractId(); // 合同id
        Date endDate = DateUtil.concatDate(vo.getRentDate(), new Date()); // 本次租金计算日期
        logger.info("开始租金自动计算，合同id：{}，租金计算日期：{}。>>>>>>>>>>>>>>>>>>>>>>>>>>", contractId, endDate);
        QueryParam param = new QueryParam();
        param.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        param.getParams().put("billState", new Parameter(QueryParam.NOT_IN, "1,3"));
        if (null != vo.getId()) {
            param.getParams().put("id", new Parameter(QueryParam.NE, vo.getId()));
        }
        List<RentCalculateEntity> checkList = super.queryList(param, false);
        if (CollectionUtils.isNotEmpty(checkList)) {
            throw new BusinessException("当前合同存在未生效的租金计算单");
        }
        // 最新租金计算单
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getOrderMap().put("rentDate", QueryParam.DESC);
        List<RentCalculateEntity> lastList = super.queryList(param, false);
        Date startDate = null;
        if(CollectionUtils.isNotEmpty(lastList)){
            RentCalculateEntity last = lastList.get(0);
            startDate = DateUtil.concatDate(DateUtil.dayAddOne(last.getRentDate()), last.getCreateTime());
        }
        /*
         * 1. 查询当前合同下物料台账信息
         */
        logger.info(">>>>查询已入场物料信息start");
        List<MaterialVO> reportList = materialService.queryCalculateList(contractId, startDate, endDate);
        if (CollectionUtils.isEmpty(reportList)) throw new BusinessException("未查询到当前合同下物料入场信息");
        logger.info("<<<<查询已入场台账信息end，查询结果：{}", JSONObject.toJSONString(reportList));
        /*
         * 2. 查询合同变更及当前合同信息
         */
        // 查询主合同信息
        logger.info(">>>>查询合同信息start");
//        ContractEntity contract = contractService.selectById(contractId);
//        ContractEntity contract = DetailListUtil.selectById(contractId, ContractServiceImpl.class);
        ContractEntity contract = DetailListUtil.selectById(contractId, contractService);
        logger.info("主合同信息：{}", JSONObject.toJSONString(contract));
        // key-日租、月租，value-{key-物料主键，value-cacheVO}     cacheVO存储开始时间和截止时间，及日租明细、月租明细
        Map<String, Map<Long, List<RentCalculateCacheVO>>> contractAllMap = this.getContractRecord(contract, endDate);
        // 3. 根据租赁方式及变更日期，对操作日期再次进行排序
        logger.info(">>>>对查询数据进行操作");
        RentCalculateEntity entity = new RentCalculateEntity();
        List<RentCalculateDailyEntity> dailyList = new ArrayList<>();
        List<RentCalculateMonthEntity> monthList = new ArrayList<>();
        List<RentCalculateCacheVO> dailyVOList = new ArrayList<>();
        List<RentCalculateCacheVO> monthVOList = new ArrayList<>();
        // 循环比对日期
        for (MaterialVO materialVO : reportList) {
            if (RENT_TYPE_DAY.equals(materialVO.getRentCalculationType())) {// 日租
                this.packParameterDetailAndContract(dailyVOList, contractAllMap, materialVO, RENT_TYPE_DAY,
                        contract.getMonthSettlementDay());
            } else if (RENT_TYPE_MONTH.equals(materialVO.getRentCalculationType())) {// 月租
                this.packParameterDetailAndContract(monthVOList, contractAllMap, materialVO, RENT_TYPE_MONTH,
                        contract.getMonthSettlementDay());
            }
        }
        // 自动带入物料挂接成本科目
        Set<Long> ids = reportList.stream().map(MaterialVO::getMaterialId).collect(Collectors.toSet());
        CommonResponse<List<com.ejianc.foundation.share.vo.MaterialVO>> resp = materialApi.queryMaterialByIds(new ArrayList<>(ids));
        Map<Long, com.ejianc.foundation.share.vo.MaterialVO> materialMap = new HashMap<>();
        if(resp.isSuccess() && CollectionUtils.isNotEmpty(resp.getData())){
            materialMap = resp.getData().stream().collect(Collectors.toMap(com.ejianc.foundation.share.vo.MaterialVO::getId, Function.identity()));
        }
        this.setRentalDayDetailValue(dailyVOList, dailyList, materialMap);
        this.setRentalMonthDetailValue(monthVOList, monthList, materialMap);
        // 4. 处理租金计算单
        entity.setDailyList(dailyList);
        entity.setMonthList(monthList);
        this.setRentalValue(entity, contract, endDate);
        logger.info(">>>>租金自动计算结束，输出结果：{}", JSONObject.toJSONString(entity));
        return BeanMapper.map(entity, RentCalculateVO.class);
    }

    /**
     * 日租明细赋值
     *
     * @param resultList        日租组装数据
     * @param dailyList 日租明细结果
     */
    private void setRentalDayDetailValue(List<RentCalculateCacheVO> resultList, List<RentCalculateDailyEntity> dailyList,
                                         Map<Long, com.ejianc.foundation.share.vo.MaterialVO> materialMap) {
        if (CollectionUtils.isEmpty(resultList)) return;
        for (RentCalculateCacheVO vo : resultList) {
            RecordDailyRentVO recordVO = vo.getDailyRecord();
            RentCalculateDailyEntity entity = new RentCalculateDailyEntity();
            entity.setSourceId(recordVO.getSourceBid()); // 来源合同明细主键
            entity.setMaterialTypeId(recordVO.getMaterialTypeId());
            entity.setMaterialTypeName(recordVO.getMaterialTypeName());
            entity.setMaterialId(recordVO.getMaterialId());
            entity.setMaterialCode(recordVO.getMaterialCode());
            entity.setMaterialName(recordVO.getMaterialName());
            entity.setSpec(recordVO.getSpec());
            entity.setUnitId(recordVO.getUnitId());
            entity.setUnitName(recordVO.getUnitName());
            entity.setUnitTaxPrice(recordVO.getUnitTaxPrice());
            entity.setUnitPrice(recordVO.getUnitPrice());
            entity.setStopUnitTaxPrice(recordVO.getStopUnitTaxPrice());
            entity.setStopUnitPrice(recordVO.getStopUnitPrice());
            entity.setTaxRate(recordVO.getTaxRate());

            entity.setStartDate(vo.getStartDate());
            entity.setEndDate(vo.getEndDate());
            entity.setNum(vo.getNum()); // 修改数量需要重新计算下面的金额
            entity.setUseStatus(String.valueOf(vo.getMaterialState() - 2));
            entity.setRentDayDate(DateUtil.getSubDay(vo.getEndDate(), vo.getStartDate())); // 租赁天数
            if (MaterialStateEnum.启用.getCode().equals(vo.getMaterialState())) {// 启用
                entity.setDailyRentMny(this.getMny(entity.getUnitPrice(), entity.getRentDayDate(), entity.getNum())); // 租赁金额(无税)
                entity.setDailyRentTaxMny(this.getMny(entity.getUnitTaxPrice(), entity.getRentDayDate(), entity.getNum())); // // 租赁金额
            } else if (MaterialStateEnum.停用.getCode().equals(vo.getMaterialState())) {// 停用
                entity.setDailyRentMny(this.getMny(entity.getStopUnitPrice(), entity.getRentDayDate(), entity.getNum())); // 租赁金额(无税)
                entity.setDailyRentTaxMny(this.getMny(entity.getStopUnitTaxPrice(), entity.getRentDayDate(), entity.getNum())); // // 租赁金额
            }
            entity.setDailyTax(ComputeUtil.safeSub(entity.getDailyRentTaxMny(), entity.getDailyRentMny())); // 租赁税额
//            if(materialMap.containsKey(entity.getMaterialId())){
//                com.ejianc.foundation.share.vo.MaterialVO material = materialMap.get(entity.getMaterialId());
//                entity.setSubjectId(material.getSubjectId());
//                entity.setSubjectName(material.getSubjectName());
//            }
            entity.setId(IdWorker.getId());
            entity.setRowState("add");
            dailyList.add(entity);
        }
        // 按照物料主键、使用状态、开始时间排序
        if (CollectionUtils.isNotEmpty(dailyList)) {
            dailyList.sort(Comparator.comparing(RentCalculateDailyEntity::getMaterialId)
                    .thenComparing(RentCalculateDailyEntity::getStartDate));
        }
    }

    /**
     * 月租明细赋值
     *
     * @param resultList          月租组装数据
     * @param monthList 月租明细结果
     */
    private void setRentalMonthDetailValue(List<RentCalculateCacheVO> resultList, List<RentCalculateMonthEntity> monthList,
                                           Map<Long, com.ejianc.foundation.share.vo.MaterialVO> materialMap) {
        if (CollectionUtils.isEmpty(resultList)) return;
        List<RentCalculateMonthEntity> rstList = new ArrayList<>();
        for (RentCalculateCacheVO vo : resultList) {
            RecordMonthRentVO recordVO = vo.getMonthRecord();
            RentCalculateMonthEntity entity = new RentCalculateMonthEntity();
            entity.setSourceId(recordVO.getSourceBid()); // 来源合同明细主键
            entity.setMaterialTypeId(recordVO.getMaterialTypeId());
            entity.setMaterialTypeName(recordVO.getMaterialTypeName());
            entity.setMaterialId(recordVO.getMaterialId());
            entity.setMaterialCode(recordVO.getMaterialCode());
            entity.setMaterialName(recordVO.getMaterialName());
            entity.setSpec(recordVO.getSpec());
            entity.setUnitId(recordVO.getUnitId());
            entity.setUnitName(recordVO.getUnitName());
            entity.setUnitTaxPrice(recordVO.getUnitTaxPrice());
            entity.setUnitPrice(recordVO.getUnitPrice());
            entity.setLackMonthDayUnitTaxPrice(recordVO.getLackMonthDayUnitTaxPrice());
            entity.setLackMonthDayUnitPrice(recordVO.getLackMonthDayUnitPrice());
            entity.setStopUnitTaxPrice(recordVO.getStopUnitTaxPrice());
            entity.setStopUnitPrice(recordVO.getStopUnitPrice());
            entity.setTaxRate(recordVO.getTaxRate());

            entity.setStartDate(vo.getStartDate());
            entity.setEndDate(vo.getEndDate());
            entity.setNum(vo.getNum()); // 修改数量需要重新计算下面的金额
            entity.setUseStatus(String.valueOf(vo.getMaterialState() - 2));
            // 1. 判断是不是满月，满月取月租单价
            // 2. 剩下日期取不足月单价
            // 3. 停租没有满月概念
            if (MaterialStateEnum.停用.getCode().equals(vo.getMaterialState())) {// 停用
                Integer days = DateUtil.getSubDay(vo.getEndDate(), vo.getStartDate());
                entity.setMonthRentMny(this.getMny(entity.getStopUnitPrice(), days, entity.getNum())); // 租赁金额(无税)
                entity.setMonthRentTaxMny(this.getMny(entity.getStopUnitTaxPrice(), days, entity.getNum())); // 租赁金额
                entity.setRentDayDate(days);
                entity.setRentMonthDate(0);
            } else {
                this.monthSum(entity, vo);
            }
            entity.setMonthTax(ComputeUtil.safeSub(entity.getMonthRentTaxMny(), entity.getMonthRentMny()));
//            if(materialMap.containsKey(entity.getMaterialId())){
//                com.ejianc.foundation.share.vo.MaterialVO material = materialMap.get(entity.getMaterialId());
//                entity.setSubjectId(material.getSubjectId());
//                entity.setSubjectName(material.getSubjectName());
//            }
            entity.setId(IdWorker.getId());
            entity.setRowState("add");
            rstList.add(entity);
        }
        rstList.sort(Comparator.comparing(RentCalculateMonthEntity::getStartDate));
        logger.info("合并前数据：{}", JSONObject.toJSONString(rstList));
        monthList.addAll(rstList);
        // 按照物料主键、使用状态、开始时间排序
        if (CollectionUtils.isNotEmpty(monthList)) {
            monthList.sort(Comparator.comparing(RentCalculateMonthEntity::getMaterialId)
                    .thenComparing(RentCalculateMonthEntity::getStartDate));
        }
    }

    /**
     * 计算月租租金
     *
     * @param entity 月租明细
     * @param vo    参数
     */
    private void monthSum(RentCalculateMonthEntity entity, RentCalculateCacheVO vo) {
        Date startDate = entity.getStartDate();
        // 包头包尾  例如  2022-05-02  到 2022-05-04   为2,3,4三天
        // 例如2022-05-04  到2022-06-03  这是一个月。
        int[] mo = DateUtil.dayCompare(startDate, entity.getEndDate());
        int month = mo[0];
        int day = mo[1];
        BigDecimal num = entity.getNum();
        // 月租无税
        BigDecimal monthMny = this.getMny(entity.getUnitPrice(), month, num);
        // 月租含税
        BigDecimal monthTaxMny = this.getMny(entity.getUnitTaxPrice(), month, num);
        // 不足月无税
        BigDecimal dayMny = this.getMny(entity.getLackMonthDayUnitPrice(), day, num);
        // 不足月含税
        BigDecimal dayTaxMny = this.getMny(entity.getLackMonthDayUnitTaxPrice(), day, num);
        entity.setMonthRentMny(ComputeUtil.safeAdd(monthMny, dayMny)); // 租赁金额(无税)
        entity.setMonthRentTaxMny(ComputeUtil.safeAdd(monthTaxMny, dayTaxMny)); // 租赁金额
        entity.setRentMonthDate(month);
        entity.setRentDayDate(day);
    }

    /**
     * 租金计算主单据赋默认值
     *
     * @param entity         租金计算单据
     * @param contract 合同信息
     * @param date           租金计算日期
     */
    private void setRentalValue(RentCalculateEntity entity, ContractEntity contract, Date date) {
        entity.setRentDate(date);
        entity.setCalculateType("1");
//        entity.setRelationFlag("0");
//        entity.setProportionFlag("0");
        entity.setSettleFlag(0);
        entity.setProjectId(contract.getProjectId());
        entity.setProjectCode(contract.getProjectCode());
        entity.setProjectName(contract.getProjectName());
        entity.setOrgId(contract.getOrgId());
        entity.setOrgCode(contract.getOrgCode());
        entity.setOrgName(contract.getOrgName());
        entity.setParentOrgId(contract.getParentOrgId());
        entity.setParentOrgCode(contract.getParentOrgCode());
        entity.setParentOrgName(contract.getParentOrgName());
        entity.setContractId(contract.getId());
        entity.setContractCode(contract.getBillCode());
        entity.setContractName(contract.getContractName());
        entity.setSupplierId(contract.getSupplierId());
        entity.setSupplierName(contract.getSupplierName());
        entity.setEmployeeId(sessionManager.getUserContext().getEmployeeId());
        entity.setEmployeeName(sessionManager.getUserContext().getEmployeeName());
        entity.setDeptId(sessionManager.getUserContext().getDeptId());
        entity.setDeptName(sessionManager.getUserContext().getDeptName());
        entity.setRentCalculationType(contract.getRentCalculationType());
        entity.setMonthSettlementDay(contract.getMonthSettlementDay());
        BigDecimal rentMny = BigDecimal.ZERO; // 租赁金额无税
        BigDecimal rentTaxMny = BigDecimal.ZERO;
        BigDecimal rentTax = BigDecimal.ZERO;
        if (CollectionUtils.isNotEmpty(entity.getDailyList())) {
            for (RentCalculateDailyEntity detail : entity.getDailyList()) {
                rentMny = ComputeUtil.safeAdd(rentMny, detail.getDailyRentMny());
                rentTaxMny = ComputeUtil.safeAdd(rentTaxMny, detail.getDailyRentTaxMny());
                rentTax = ComputeUtil.safeAdd(rentTax, detail.getDailyTax());
            }
        }
        if (CollectionUtils.isNotEmpty(entity.getMonthList())) {
            for (RentCalculateMonthEntity detail : entity.getMonthList()) {
                rentMny = ComputeUtil.safeAdd(rentMny, detail.getMonthRentMny());
                rentTaxMny = ComputeUtil.safeAdd(rentTaxMny, detail.getMonthRentTaxMny());
                rentTax = ComputeUtil.safeAdd(rentTax, detail.getMonthTax());
            }
        }
        entity.setRentMny(rentMny);
        entity.setRentTaxMny(rentTaxMny);
        entity.setRentTax(rentTax);
    }

    /**
     * 组装物料台账记录明细及合同变更明细
     *
     * @param resultList      组装后结果
     * @param contractAllMap  合同变更明细
     * @param materialVO     物料台账
     * @param rentType        租赁方式
     * @param day 月结算日
     */
    private void packParameterDetailAndContract(List<RentCalculateCacheVO> resultList,
                                                Map<String, Map<Long, List<RentCalculateCacheVO>>> contractAllMap,
                                                MaterialVO materialVO, String rentType, Integer day) {
        // 日租明细
        Map<Long, List<RentCalculateCacheVO>> map = contractAllMap.get(rentType);
        if(MapUtils.isEmpty(map)) return;
        // 取出当前物料主键变更历史
        List<RentCalculateCacheVO> cacheList = map.get(materialVO.getMaterialId());
        if(CollectionUtils.isEmpty(cacheList)) return;
        // 按照开始日期升序排序
        cacheList.sort(Comparator.comparing(RentCalculateCacheVO::getStartDate));
        // 获取物料操作记录
        List<MaterialFlowVO> flowList = materialVO.getFlowList();
        List<RentCalculateCacheVO> result = new ArrayList<>();
        // 计算期初
        List<MaterialFlowVO> beginList = materialVO.getBeginList();
        // 验收单模拟期初数据，没有期初则下一操作必为验收
        if(CollectionUtils.isEmpty(beginList) && CollectionUtils.isNotEmpty(flowList)){
            MaterialFlowVO start = BeanMapper.map(flowList.get(0), MaterialFlowVO.class);
            start.setNum(BigDecimal.ZERO);
            start.setStartedNum(BigDecimal.ZERO);
            start.setStopedNum(flowList.get(0).getNum());
            beginList = new ArrayList<>(Arrays.asList(start));
            MaterialFlowVO stop = BeanMapper.map(start, MaterialFlowVO.class);
            stop.setMaterialState(MaterialStateEnum.停用.getCode());
            stop.setUseStatus(CommonConstant.NO);
            beginList.add(stop);
        }
        // 生成期初数据
        if(CollectionUtils.isNotEmpty(beginList)){
            for (RentCalculateCacheVO cache : cacheList) {
                for (MaterialFlowVO begin : beginList) {
                    this.dealFlowList(materialVO, cache, begin, CollectionUtils.isNotEmpty(flowList) ? flowList.get(0) : null, result, null);
                }
            }
        }
        //流水拆分
        if(CollectionUtils.isNotEmpty(flowList)){
            // 按照操作日期升序排序
            flowList.sort(Comparator.comparing(MaterialFlowVO::getOperationDate));
            for (RentCalculateCacheVO cache : cacheList) {
                for (MaterialFlowVO begin : beginList) {
                    for (int i = 0; i < flowList.size(); i++) {
                        MaterialFlowVO detailVO = flowList.get(i);
                        MaterialFlowVO detailVO1 = i == flowList.size() - 1 ? null : flowList.get(i + 1);
                        this.dealFlowList(materialVO, cache, detailVO, detailVO1, result, CommonConstant.YES.equals(begin.getUseStatus()));
                    }
                }
            }
        }
        result = this.packResultCacheVo(result, day);
        resultList.addAll(result);
        logger.info("租赁方式：{}，组装结果：{}", rentType, JSONObject.toJSONString(resultList));
    }

    /**
     * 按操作日期和操作数量拆分
     * @param materialVO
     * @param cache
     * @param detailVO
     * @param detailVO1
     * @param result
     * @param beginFlag
     * 关键字段注释 A-本次操作日期，B-下次操作日期（本次操作结束日期），T1-合同拆分后开始日期（合同签订日期/合同变更日期），T2-合同拆分后结束日期（合同变更日期 - 1/计算结束日期）
     */
    private void dealFlowList(MaterialVO materialVO, RentCalculateCacheVO cache, MaterialFlowVO detailVO, MaterialFlowVO detailVO1,
                              List<RentCalculateCacheVO> result, Boolean beginFlag) {
        // 给下一vo赋值期初数量
        if(detailVO1 != null) {
            this.setBeginNum(detailVO, detailVO1);
        }
        // 根据操作日期和数量拆分
        RentCalculateCacheVO vo = this.transferCacheVO(materialVO, detailVO, cache, beginFlag);
        // 默认起始日期为本次操作日期，如果本次退场，则止租日期 + 1 作为分段起始日期
        Date startDate = MaterialStateEnum.退场.getCode().equals(detailVO.getMaterialState()) ?
                DateUtil.beginOfDate(DateUtil.dayAddOne(detailVO.getOperationDate())) : detailVO.getOperationDate();
        // 最后一条操作
        if(detailVO1 == null){
            // 操作日期大于区间开始日期 A > T1
            if(DateUtil.compareDate(detailVO.getOperationDate(), cache.getStartDate()) > 0){
                // 操作日期大于区间结束日期无效
                if(DateUtil.compareDate(detailVO.getOperationDate(), cache.getEndDate()) > 0){
                    return;
                }
                // 操作日期小于等于区间结束日期 T1 < A <= T2
                this.setValue(startDate, cache.getEndDate(), vo, result);// A-T2
            }
            // 操作日期小于等于区间开始日期 A <= T1
            else {
                this.setValue(cache.getStartDate(), cache.getEndDate(), vo, result);// T1-T2
            }
        }
        // 非最后一条其他操作
        else {
            // 退场止租日期计算当天
            Date endDate = MaterialStateEnum.退场.getCode().equals(detailVO1.getMaterialState()) ?
                    detailVO1.getOperationDate() : DateUtil.daySubOne(detailVO1.getOperationDate());
            // 操作日期大于区间开始日期 A > T1
            if(DateUtil.compareDate(detailVO.getOperationDate(), cache.getStartDate()) > 0){
                // 操作日期大于区间结束日期无效
                if(DateUtil.compareDate(detailVO.getOperationDate(), cache.getEndDate()) > 0){
                    return;
                }
                // 操作日期小于等于区间结束日期 T1 < A <= T2
                // 下一操作日期大于区间结束日期 T1 < A <= T2 < B
                if(DateUtil.compareDate(detailVO1.getOperationDate(), cache.getEndDate()) > 0){
                    this.setValue(startDate, cache.getEndDate(), vo, result);// A-T2
                }
                // 下一操作日期小于等于区间结束日期 T1 < A < B <= T2
                else {
                    this.setValue(startDate, endDate, vo, result);// A-B-1，缺失B-T2
                }
            }
            // 操作日期小于等于区间开始日期 A <= T1
            else {
                // 下一操作日期大于区间结束日期 A <= T1 < T2 < B
                if(DateUtil.compareDate(detailVO1.getOperationDate(), cache.getEndDate()) > 0){
                    this.setValue(cache.getStartDate(), cache.getEndDate(), vo, result);// T1-T2
                }
                // 下一操作日期小于等于区间结束日期 A <= T1 < B <= T2
                else {
                    this.setValue(cache.getStartDate(), endDate, vo, result);// T1-B
                }
            }
        }
    }

    /**
     * 赋值期初数量
     * @param detailVO
     * @param detailVO1
     */
    private void setBeginNum(MaterialFlowVO detailVO, MaterialFlowVO detailVO1) {
        if(MaterialStateEnum.退场.getCode().equals(detailVO1.getMaterialState())){
            // 减去退赔数量
            if(CommonConstant.YES.equals(detailVO1.getUseStatus())){
                detailVO1.setStartedNum(ComputeUtil.safeSub(detailVO.getStartedNum(), detailVO1.getNum()));
                detailVO1.setStopedNum(detailVO.getStopedNum());
            } else {
                detailVO1.setStartedNum(detailVO.getStartedNum());
                detailVO1.setStopedNum(ComputeUtil.safeSub(detailVO.getStopedNum(), detailVO1.getNum()));
            }
        } else {
            if(CommonConstant.YES.equals(detailVO1.getUseStatus())){
                detailVO1.setStartedNum(ComputeUtil.safeAdd(detailVO.getStartedNum(), detailVO1.getNum()));
                detailVO1.setStopedNum(ComputeUtil.safeSub(detailVO.getStopedNum(), detailVO1.getNum()));
            } else {
                detailVO1.setStartedNum(ComputeUtil.safeSub(detailVO.getStartedNum(), detailVO1.getNum()));
                detailVO1.setStopedNum(ComputeUtil.safeAdd(detailVO.getStopedNum(), detailVO1.getNum()));
            }
        }
    }

    /**
     * 赋值开始时间、结束时间
     * @param startDate
     * @param endDate
     * @param vo
     * @param result
     */
    private void setValue(Date startDate, Date endDate, RentCalculateCacheVO vo, List<RentCalculateCacheVO> result) {
        if(DateUtil.compareDate(startDate, endDate) > 0){
            return;
        }
        vo.setStartDate(startDate);
        vo.setEndDate(endDate);
        if(ComputeUtil.isLessOrEqual(vo.getNum(), BigDecimal.ZERO)){
            return;
        }
        result.add(vo);
    }

    /**
     * 根据租赁方式拼装不同的缓存VO
     *
     * @param materialVO    台账VO
     * @param detailVO     操作VO
     * @param cache     查询结果
     * @param flag      是否启用
     * @return
     */
    private RentCalculateCacheVO transferCacheVO(MaterialVO materialVO, MaterialFlowVO detailVO, RentCalculateCacheVO cache, Boolean flag) {
        RentCalculateCacheVO vo = new RentCalculateCacheVO();
        String rentType = materialVO.getRentCalculationType();
        vo.setParameterId(materialVO.getId());
        vo.setRentType(rentType);
        if (RENT_TYPE_DAY.equals(rentType)) {
            vo.setDailyRecord(cache.getDailyRecord());
        }
        else if (RENT_TYPE_MONTH.equals(rentType)) {
            vo.setMonthRecord(cache.getMonthRecord());
        }
        if(flag == null) {
            vo.setNum(detailVO.getNum());
            vo.setMaterialState(detailVO.getMaterialState());
        } else if(flag){
            vo.setNum(detailVO.getStartedNum());
            vo.setMaterialState(MaterialStateEnum.启用.getCode());
        } else {
            vo.setNum(detailVO.getStopedNum());
            vo.setMaterialState(MaterialStateEnum.停用.getCode());
        }
        return vo;
    }

    /**
     * 校验两个vo是否需要合并，如果需要合并，则合并入vo1中
     *
     * @param vo1 vo1
     * @param vo2 vo2
     * @return true 不需要合并，false需要合并
     */
    private boolean checkCacheVoIsMerge(RentCalculateCacheVO vo1, RentCalculateCacheVO vo2) {
        if (!vo1.getMaterialState().equals(vo2.getMaterialState())) {
            return true;
        }
        BigDecimal price1 = null;
        BigDecimal price2 = null;
        // vo1和vo2 的租赁方式一定相同
        if (RENT_TYPE_DAY.equals(vo1.getRentType())) {
            price1 = vo1.getDailyRecord().getUnitTaxPrice();
            price2 = vo2.getDailyRecord().getUnitTaxPrice();
            if (MaterialStateEnum.停用.getCode().equals(vo1.getMaterialState())) {
                price1 = vo1.getDailyRecord().getStopUnitTaxPrice();
                price2 = vo2.getDailyRecord().getStopUnitTaxPrice();
            }
        } else if (RENT_TYPE_MONTH.equals(vo1.getRentType())) {
            price1 = vo1.getMonthRecord().getUnitTaxPrice();
            price2 = vo2.getMonthRecord().getUnitTaxPrice();
            if (MaterialStateEnum.停用.getCode().equals(vo1.getMaterialState())) {
                price1 = vo1.getMonthRecord().getStopUnitTaxPrice();
                price2 = vo2.getMonthRecord().getStopUnitTaxPrice();
            }
        } else {
            return true;
        }
        if (price1.equals(price2)) {
            vo1.setEndDate(vo2.getEndDate());
            return false;
        }
        return true;
    }

    /**
     * 根据月结算日组装月租数据
     *
     * @param resultList      组装结果
     * @param day 月结算日
     */
    private List<RentCalculateCacheVO> packResultCacheVo(List<RentCalculateCacheVO> resultList, Integer day) {
        List<RentCalculateCacheVO> rtnList = new ArrayList<>();
        for (RentCalculateCacheVO vo : resultList) {
            // 如果是月租，且是启用才需要按照月结算日拆分
            if (RENT_TYPE_MONTH.equals(vo.getRentType()) && MaterialStateEnum.启用.getCode().equals(vo.getMaterialState())) {
                // 开始日期当月结算日  例如 开始日期2022-05-24  月结算日15  则日期为2022-05-15
                Date startBeginDate = DateUtil.monthDay(vo.getStartDate(), day);
                // 开始日期下月结算日  例如 开始日期2022-05-24  月结算日15  则日期为2022-06-15
                Date startEndDate = DateUtil.monthDay(DateUtil.monthAddOne(vo.getStartDate()), day);
                // 结束日期当月结算日  例如 结束日期2022-05-24  月结算日15  则日期为2022-05-15
                Date endBeginDate = DateUtil.monthDay(vo.getEndDate(), day);
                // 结束日期下月结算日  例如 结束日期2022-05-24  月结算日15  则日期为2022-06-15
                Date endEndDate = DateUtil.monthDay(DateUtil.monthAddOne(vo.getEndDate()), day);
                // 时间段对比以下几种情况中情况
                // 开始时间A，结束时间B，开始当月结算日A1，开始当月结算日A2，结束当月结算日B1，结束当月结算日B2
                // 1. A<B<A1   例如  开始2022-05-13，结束2022-05-14，月结算日2022-05-15
                // 2. 开始时间<开始月结算日<结束时间<开始下月结算日 例如  开始2022-05-13，结束2022-06-14，
                // 月结算日2022-05-15，下月结算日2022-06-15 截取时间 2022-05-13到2022-05-15,2022-05-16到2022-06-14
                // 3. 开始<开结<开下结<结束  例如 开始2022-05-13，结束2022-06-17，月结算日2022-05-15，下月结算日2022-06-15
                // 截取时间  13-15  05-16到06-15  06-16到06-17
                // 4. 开结<开始<结束<开下结
                // 开始日期、结束日期在月结算日之前
                // 月结算日15   开始10  结束13，不做处理
                if (DateUtil.compareDay(vo.getStartDate(), startBeginDate) < 0) {
                    // 1. a < b < a1 < a2
                    if (DateUtil.compareDay(vo.getEndDate(), startBeginDate) < 0) {
                        rtnList.add(vo);
                    }
                    // 2. A < A1 < B < A2
                    else if (DateUtil.compareDay(vo.getEndDate(), startBeginDate) > 0 && DateUtil.compareDay(vo.getEndDate(), startEndDate) < 0) {
                        // 开始-月结算日
                        this.setRstValue(vo.getStartDate(), startBeginDate, vo, rtnList);
                        // 月结算日-结束
                        this.setRstValue(DateUtil.dayAddOne(startBeginDate), vo.getEndDate(), vo, rtnList);
                    }
                    // 3. A < A1 < A2 < B < B1
                    else if (DateUtil.compareDay(vo.getEndDate(), startEndDate) > 0 && DateUtil.compareDay(vo.getEndDate(), endBeginDate) < 0) {
                        // 结束日期上月结算日
                        // 开始-月结算日
                        this.setRstValue(vo.getStartDate(), startBeginDate, vo, rtnList);
                        // 3.1 A < A1 < A2 < B0 < B < B1
                        Date endUpDate = DateUtil.monthDay(DateUtil.addMonths(vo.getEndDate(), -1), day);
                        Date endDate;
                        // 3.1 A < A1 < A2 < B0 < B < B1
                        if (DateUtil.compareDay(endUpDate, startEndDate) > 0) {
                            // 月结算日-月结算日
                            this.setRstValue(DateUtil.dayAddOne(startBeginDate), endUpDate, vo, rtnList);
                            endDate = endUpDate;
                        }
                        // 3.2 A < A1 < B0 <= A2 < B < B1
                        else {
                            // 月结算日-月结算日
                            this.setRstValue(DateUtil.dayAddOne(startBeginDate), startEndDate, vo, rtnList);
                            endDate = startEndDate;
                        }
                        // 月结算日 - 结束日期
                        this.setRstValue(DateUtil.dayAddOne(endDate), vo.getEndDate(), vo, rtnList);
                    }
                    // 4. A < A1 < A2 < B1 < B < B2
                    else if (DateUtil.compareDay(vo.getEndDate(), startEndDate) > 0 && DateUtil.compareDay(vo.getEndDate(), endBeginDate) > 0) {
                        // 开始-月结算日
                        this.setRstValue(vo.getStartDate(), startBeginDate, vo, rtnList);
                        if (DateUtil.compareDay(endBeginDate, startEndDate) > 0) {
                            // 月结算日-月结算日
                            this.setRstValue(DateUtil.dayAddOne(startBeginDate), endBeginDate, vo, rtnList);
                        }
                        if(DateUtil.compareDay(endBeginDate, startBeginDate) > 0){
                            // 月结算日-月结算日
                            this.setRstValue(DateUtil.dayAddOne(startBeginDate), endBeginDate, vo, rtnList);
                        }                        // 月结算日 - 结束日期
                        this.setRstValue(DateUtil.dayAddOne(endBeginDate), vo.getEndDate(), vo, rtnList);
                    }
                    else {
                        rtnList.add(vo);
                    }
                }
                // 开始日期在月结算日之后，结束日期在下月结算日之前
                // 例如月结算日15   开始日期2022-05-17，结束日期2022-06-14，则不足月，不按一个月算 中间插入 月结算日2022-05-15
                else if (DateUtil.compareDay(vo.getStartDate(), startBeginDate) > 0 ) {
                    // 5. A1 < A < B < A2
                    if (DateUtil.compareDay(vo.getEndDate(), startEndDate) < 0) {
                        rtnList.add(vo);
                    }
                    // 6. A1 < A < A2 < B < B1
                    else if (DateUtil.compareDay(vo.getEndDate(), startEndDate) > 0 && DateUtil.compareDay(vo.getEndDate(), endBeginDate) < 0) {
                        // 开始-下月结算日
                        this.setRstValue(vo.getStartDate(), startEndDate, vo, rtnList);
                        Date endUpDate = DateUtil.monthDay(DateUtil.addMonths(vo.getEndDate(), -1), day);
                        Date endDate;
                        // 6.1 A1 < A < A2 < B0 < B < B1
                        if (DateUtil.compareDay(endUpDate, startEndDate) > 0) {
                            // 月结算日-月结算日
                            this.setRstValue(DateUtil.dayAddOne(startEndDate), endUpDate, vo, rtnList);
                            endDate = endUpDate;
                        }
                        // 6.2 A1 < A < B0 <= A2 < B < B1
                        else {
                            endDate = startEndDate;
                        }
                        // 月结算日 - 结束日期
                        this.setRstValue(DateUtil.dayAddOne(endDate), vo.getEndDate(), vo, rtnList);
                    }
                    // 7. A1 < A < A2 < B1 < B < B2
                    else if (DateUtil.compareDay(vo.getEndDate(), startEndDate) > 0 && DateUtil.compareDay(vo.getEndDate(), endBeginDate) > 0) {
                        // 开始-下月结算日
                        this.setRstValue(vo.getStartDate(), startEndDate, vo, rtnList);
                        // endBeginDate >= startEndDate   只可能出现这一种情况
                        if (DateUtil.compareDay(endBeginDate, startEndDate) > 0) {
                            // 开始下月结算日 - 结束月结算日
                            this.setRstValue(DateUtil.dayAddOne(startEndDate), endBeginDate, vo, rtnList);
                        }
                        // 月结算日 - 结束日期
                        this.setRstValue(DateUtil.dayAddOne(endBeginDate), vo.getEndDate(), vo, rtnList);
                    }
                    else {
                        rtnList.add(vo);
                    }
                }
                // 开始日期等于月结算日
                else {
                    this.setRstValue(vo.getStartDate(), startBeginDate, vo, rtnList);
                    if(DateUtil.compareDay(vo.getEndDate(), startBeginDate) > 0){
                        this.setRstValue(DateUtil.dayAddOne(startBeginDate), vo.getEndDate(), vo, rtnList);
                    }
                }
            }
            else {
                rtnList.add(vo);
            }
        }
        rtnList.sort(Comparator.comparing(RentCalculateCacheVO::getStartDate));
        return rtnList;
    }

    /**
     * 给结果赋值
     *
     * @param start   开始日期
     * @param end     结束日期
     * @param vo      vo
     * @param rtnList 结果列表
     */
    private void setRstValue(Date start, Date end, RentCalculateCacheVO vo, List<RentCalculateCacheVO> rtnList) {
        RentCalculateCacheVO rstvo = new RentCalculateCacheVO();
        rstvo.setStartDate(start);
        rstvo.setEndDate(end);
        rstvo.setNum(vo.getNum());
        rstvo.setMaterialState(vo.getMaterialState());
        rstvo.setRentType(vo.getRentType());
        rstvo.setMonthRecord(vo.getMonthRecord());
        rstvo.setParameterId(vo.getParameterId());
        rstvo.setContractType(vo.getContractType());
        rtnList.add(rstvo);
    }

    /**
     * 查询合同及合同记录信息
     *
     * @param contract 合同信息
     * @param endDate        变更截止日期
     */
    private Map<String, Map<Long, List<RentCalculateCacheVO>>> getContractRecord(ContractEntity contract,  Date endDate) {
        // 查询合同记录信息,按照变更日期升序排序
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contract.getId()));
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        queryParam.getParams().put("changeDate", new Parameter(QueryParam.LE, endDate));
        queryParam.getOrderMap().put("changeDate", QueryParam.ASC);
        List<RecordEntity> recordList = recordService.queryList(queryParam, false);
        // 没有合同变更记录，则返回主合同取的信息
        Map<String, Map<Long, List<RentCalculateCacheVO>>> rtnMap = new HashMap<>();
        int version = 0;
        if (CollectionUtils.isEmpty(recordList)) {
            // 没有合同变更记录，则合同是从签订日期开始生效
            this.packContractMap(contract, rtnMap, DateUtil.concatDate(contract.getSignDate(), contract.getCreateTime()), endDate, version);
            logger.info("<<<<合同未查询到变更记录，返回信息：{}", JSONObject.toJSONString(rtnMap));
            return rtnMap;
        }
        List<Long> recordIds = recordList.stream().map(RecordEntity::getId).collect(Collectors.toList());

        QueryParam param = new QueryParam();
        param.getParams().put("recordId", new Parameter(QueryParam.IN, recordIds));
        List<RecordDailyRentEntity> dailyList = dailyRecordService.queryList(param, false);// 日租明细
        List<RecordMonthRentEntity> monthList = monthRecordService.queryList(param, false);// 月租明细
        // 明细与记录匹配
        Map<Long, List<RecordDailyRentEntity>> dayMap = dailyList.stream().collect(
                Collectors.groupingBy(RecordDailyRentEntity::getRecordId));
        Map<Long, List<RecordMonthRentEntity>> monthMap = monthList.stream().collect(
                Collectors.groupingBy(RecordMonthRentEntity::getRecordId));

        Date startDate = null;
        Date changeDate = null;
        for (int i = 0; i < recordList.size(); i++) {
            RecordEntity record = recordList.get(i);
            record.setDailyRentList(dayMap.get(record.getId()));
            record.setMonthRentList(monthMap.get(record.getId()));
            if (startDate == null) startDate = DateUtil.concatDate(record.getSignDate(), record.getCreateTime());
            changeDate = DateUtil.concatDate(record.getChangeDate(), record.getCreateTime());
            // 如果开始日期在结束日期之后，跳过此次循环
            if (DateUtil.compareDate(startDate, changeDate) > 0) continue;
            // 处理合同记录日租明细
            if (CollectionUtils.isNotEmpty(record.getDailyRentList())) {
                Map<Long, List<RentCalculateCacheVO>> detailMap = rtnMap.get(RENT_TYPE_DAY);
                if (detailMap == null) detailMap = new HashMap<>();
                for (RecordDailyRentEntity detail : record.getDailyRentList()) {
                    // 组装数据
                    RentCalculateCacheVO vo = new RentCalculateCacheVO();
                    vo.setRentType(RENT_TYPE_DAY);
                    vo.setMaterialId(detail.getMaterialId());
                    // 如果中间有发生变更，则变更日期优先
                    vo.setStartDate(startDate); // 主合同生效范围为签订日期到当前租金计算日期
                    vo.setEndDate(DateUtil.daySubOne(changeDate));// 上次结束日期为变更日期前一天
                    vo.setNum(detail.getCount());
                    vo.setContractType(1);
                    vo.setDailyRecord(BeanMapper.map(detail, RecordDailyRentVO.class));
                    vo.setVersion(version);
                    this.packCacheMap(vo, detailMap);
                }
                rtnMap.put(RENT_TYPE_DAY, detailMap);
            }
            // 处理合同记录月租明细
            if (CollectionUtils.isNotEmpty(record.getMonthRentList())) {
                Map<Long, List<RentCalculateCacheVO>> detailMap = rtnMap.get(RENT_TYPE_MONTH);
                if (detailMap == null) detailMap = new HashMap<>();
                for (RecordMonthRentEntity detail : record.getMonthRentList()) {
                    // 组装数据
                    RentCalculateCacheVO vo = new RentCalculateCacheVO();
                    vo.setRentType(RENT_TYPE_MONTH);
                    vo.setMaterialId(detail.getMaterialId());
                    // 如果中间有发生变更，则变更日期优先
                    vo.setStartDate(startDate); // 主合同生效范围为签订日期到当前租金计算日期
                    vo.setEndDate(DateUtil.daySubOne(changeDate));// 上次结束日期为变更日期前一天
                    vo.setNum(detail.getCount());
                    vo.setContractType(1);
                    vo.setMonthRecord(BeanMapper.map(detail, RecordMonthRentVO.class));
                    vo.setVersion(version);
                    this.packCacheMap(vo, detailMap);
                }
                rtnMap.put(RENT_TYPE_MONTH, detailMap);
            }
            // 下次起始日期为本次变更日期
            startDate = DateUtil.concatDate(record.getChangeDate(), record.getCreateTime());
            version++;
        }
        // 有合同变更记录时，主合同生效范围起始日期为查询出的最后一次变更日期
        this.packContractMap(contract, rtnMap, changeDate, endDate, version);
        logger.info("<<<<处理合同信息结束，处理结果：{}", JSONObject.toJSONString(rtnMap));
        return rtnMap;
    }

    /**
     * 组装数据
     *
     * @param vo 租金自动计算临时数据
     * @param map     map
     */
    private void packCacheMap(RentCalculateCacheVO vo, Map<Long, List<RentCalculateCacheVO>> map) {
        if (map == null) map = new HashMap<>();
        if (map.containsKey(vo.getMaterialId())) {
            map.get(vo.getMaterialId()).add(vo);
        } else {
            map.put(vo.getMaterialId(), new ArrayList<>(Arrays.asList(vo)));
        }
    }

    /**
     * 组装主合同数据
     *
     * @param contract 主合同信息
     * @param rtnMap         需要返回结果
     * @param startDate      生效起始日期
     * @param endDate        生效结束日期
     */
    private void packContractMap(ContractEntity contract, Map<String, Map<Long, List<RentCalculateCacheVO>>> rtnMap, Date startDate, Date endDate, int version) {
        // 如果起始日期大于结束日期，则说明不需要主合同信息
        if (DateUtil.compareDate(endDate, startDate) < 0) return;
        // 处理日租明细
        if (CollectionUtils.isNotEmpty(contract.getDailyRentList())) {
            Map<Long, List<RentCalculateCacheVO>> detailMap = rtnMap.get(RENT_TYPE_DAY);
            if (detailMap == null) detailMap = new HashMap<>();
            RecordDailyRentVO record = new RecordDailyRentVO();
            for (ContractDailyRentEntity detail : contract.getDailyRentList()) {
                // 组装数据
                RentCalculateCacheVO vo = new RentCalculateCacheVO();
                vo.setRentType(RENT_TYPE_DAY);
                vo.setMaterialId(detail.getMaterialId());
                // 如果中间有发生变更，则变更日期优先
                vo.setStartDate(startDate); // 主合同生效范围为签订日期到当前租金计算日期
                vo.setEndDate(endDate);
                vo.setNum(detail.getCount());
                vo.setContractType(0);
                record = BeanMapper.map(detail, RecordDailyRentVO.class);// 转成记录表，带入
                record.setSourceBid(detail.getId());
                vo.setDailyRecord(record);
                vo.setVersion(version);
                this.packCacheMap(vo, detailMap);
            }
            rtnMap.put(RENT_TYPE_DAY, detailMap);
        }
        // 处理月租明细
        if (CollectionUtils.isNotEmpty(contract.getMonthRentList())) {
            Map<Long, List<RentCalculateCacheVO>> detailMap = rtnMap.get(RENT_TYPE_MONTH);
            if (detailMap == null) detailMap = new HashMap<>();
            RecordMonthRentVO record = new RecordMonthRentVO();
            for (ContractMonthRentEntity detail : contract.getMonthRentList()) {
                // 组装数据
                RentCalculateCacheVO vo = new RentCalculateCacheVO();
                vo.setRentType(RENT_TYPE_MONTH);
                vo.setMaterialId(detail.getMaterialId());
                // 如果中间有发生变更，则变更日期优先
                vo.setStartDate(startDate); // 主合同生效范围为签订日期到当前租金计算日期
                vo.setEndDate(endDate);
                vo.setNum(detail.getCount());
                vo.setContractType(0);
                record = BeanMapper.map(detail, RecordMonthRentVO.class);// 转成记录表，带入
                record.setSourceBid(detail.getId());
                vo.setMonthRecord(record);
                vo.setVersion(version);
                this.packCacheMap(vo, detailMap);
            }
            rtnMap.put(RENT_TYPE_MONTH, detailMap);
        }
    }

    /**
     * 计算金额
     *
     * @param price 单价
     * @param days       天数
     * @param num       数量
     * @return 计算结果
     */
    private BigDecimal getMny(BigDecimal price, Integer days, BigDecimal num) {
        return ComputeUtil.safeMultiply(price, new BigDecimal(days), num);
    }
}
