package com.ejianc.framework.auth.session;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springside.modules.nosql.redis.JedisTemplate.JedisAction;
import org.springside.modules.nosql.redis.JedisTemplate.JedisActionNoResult;
import org.springside.modules.nosql.redis.JedisTemplate.PipelineActionNoResult;
import org.springside.modules.nosql.redis.JedisUtils;
import org.springside.modules.nosql.redis.pool.JedisPool;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.auth.security.esapi.EncryptException;
import com.ejianc.framework.auth.security.utils.TokenGenerator;
import com.ejianc.framework.auth.token.ITokenProcessor;
import com.ejianc.framework.auth.token.TokenFactory;
import com.ejianc.framework.core.context.InvocationInfoProxy;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.exceptions.JedisConnectionException;
import redis.clients.jedis.exceptions.JedisDataException;
import redis.clients.jedis.exceptions.JedisException;

/**
 * ICOP_SESSION_USER:userId:token = token
 * 
 * @author guominga
 * 
 */
public class SessionManager {

	public static final String TOKEN_SEED = "token_seed";

	public static final String SESSION_PREFIX = "ICOP_SESSION_USER:";

	public static final String DEFAULT_CHARSET = "UTF-8";

	private final Logger logger = LoggerFactory.getLogger(getClass());

	private static String localSeedValue = null;

	private JedisPool sessionJedisPool;

	public JedisPool getSessionJedisPool() {
		return sessionJedisPool;
	}

	public void setSessionJedisPool(JedisPool sessionJedisPool) {
		this.sessionJedisPool = sessionJedisPool;
	}

	private boolean sessionMutex = false;

	public boolean isSessionMutex() {
		return sessionMutex;
	}

	public void setSessionMutex(boolean sessionMutex) {
		this.sessionMutex = sessionMutex;
	}

	/**
	 * 获得当前系统的 token seed
	 */
	public String findSeed() throws EncryptException {
		if (localSeedValue != null) {
			return localSeedValue;
		} else {
			String seed = getSeedValue(TOKEN_SEED);
			if (org.apache.commons.lang.StringUtils.isBlank(seed)) {
				seed = TokenGenerator.genSeed();
				localSeedValue = seed;
				set(TOKEN_SEED, seed);
			}
			return seed;
		}
	}

	public String getSeedValue(String key) {
		return get(key);
	}

	/**
	 * 删除session缓存
	 * 
	 * @param sid
	 *            mock的sessionid
	 */
	public void removeSessionCache(String sid) {
		del(sid);
	}
	
	public void putSessionCacheAttribute(String sid, String key, String value) {
		sessionHset(sid, key, value);
    }

	/**
	 * 获取session属性
	 */
	public String getSessionCacheAttribute(String sid, String key) {
		return sessionHget(sid, key);
	}

	private int getTimeout(String sid) {
		return TokenFactory.getTokenInfo(sid).getIntegerExpr();
	}

	public void registOnlineSession(final String userid, final String userContext, final String token,
			final ITokenProcessor processor) {
		final String key = SESSION_PREFIX + userid;
		logger.debug("token processor id is {},key is {} !", processor.getId(), key);
		// 是否互斥，如果是，则踢掉所有当前用户的session，重新创建，此变量从配置文件读取
		boolean mutex = isSessionMutex();
		if (mutex) {
			deleteUserSession(userid);
		} else {
			// 清理此用户过期的session，过期的常为异常或者直接关闭浏览器，没有走正常注销的key
			clearOnlineSession(key);
		}

		// online的key,例如ICOP_ONLINE_USER:user01@126.com
		String currentTs = String.valueOf(System.currentTimeMillis());
		if (StringUtils.isNotBlank(userContext)) {
			JSONObject contextObject = JSONObject.parseObject(userContext);
			contextObject.put("currentTs", currentTs);
			sessionHset(key, token, contextObject.toJSONString());
		} else {
			JSONObject contextObject = new JSONObject();
			contextObject.put("currentTs", currentTs);
			sessionHset(key, token, contextObject.toJSONString());
		}

	}

	public Map<String, String> getUserSessionMap(Long userId) {
		return getMap(SESSION_PREFIX + userId);
	}

	public void clearOnlineSession(final String key) {
		// 获取清理的key清理，这次操作放不到pipline中，只能单独获取
		final List<String> deleteKeys = new ArrayList<String>();
		Map<String, String> userSessions = getMap(key);
		if (userSessions != null) {
			Iterator<Map.Entry<String, String>> entries = userSessions.entrySet().iterator();
			while (entries.hasNext()) {
				Map.Entry<String, String> entry = entries.next();
				String t = entry.getKey();
				String v = entry.getValue();

				JSONObject userContextObject = JSONObject.parseObject(v);
				if(userContextObject == null) {
					continue;
				}
				String lastTsStr = userContextObject.getString("currentTs");

				long exprMillis = getTimeout(t) * 1000;
				if (exprMillis > 0) {
					// 如果最后处理时间+timeout小于当前时间，则失效
					long cts = System.currentTimeMillis();
					if ((Long.parseLong(lastTsStr) + exprMillis) < cts) {
						deleteKeys.add(t);
					}
				}
			}
		}

		PipelineActionNoResult action = new PipelineActionNoResult() {
			@Override
			public void action(Pipeline pipeline) {
				// 清理
				for (int i = 0; i < deleteKeys.size(); i++) {
					pipeline.hdel(key, deleteKeys.get(i));
				}
			}
		};
		execute(action);
	}

	public boolean validateOnlineSession(final String userid, final String token) {
		boolean result = false;

		final String key = SESSION_PREFIX + userid;

		final String userContextStr = sessionHget(key, token);

		if (userContextStr != null) {
			JSONObject userContextObject = JSONObject.parseObject(userContextStr);
			String lastTsStr = userContextObject.getString("currentTs");
			final int timeOut = getTimeout(token);
			if (timeOut <= 0) {
				return true;
			}

			// 使用piplie的方式合并一次操作
			final List<Object> pipelineResult = new ArrayList<Object>();
			PipelineActionNoResult action = new PipelineActionNoResult() {
				@Override
				public void action(Pipeline pipeline) {

					long exprMillis = timeOut * 1000;
					if (exprMillis > 0) {
						long cts = System.currentTimeMillis();
						if ((Long.parseLong(lastTsStr) + exprMillis) >= cts) {

							userContextObject.put("currentTs", cts);
							pipeline.hset(key, token, userContextObject.toJSONString());

							// 重置最后时间,是否需要重新expire业务session待确认
							pipeline.expire(token, timeOut);

							pipelineResult.add(true);
						} else {
							pipeline.hdel(key, token);
						}
					}
				}
			};
			execute(action);
			result = pipelineResult.size() > 0;
		}

		return result;
	}

	// 注销用户时候需要调用
	public void delOnlineSession(final String userid, final String token) {
		final String key = SESSION_PREFIX + userid;
		PipelineActionNoResult action = new PipelineActionNoResult() {
			@Override
			public void action(Pipeline pipeline) {
				// 业务上的
				pipeline.del(token);
				// online的
				pipeline.hdel(key, token);
			}
		};
		execute(action);
	}

	// 禁用或者删除用户时候调用
	public void deleteUserSession(final String userid) {
		final String key = SESSION_PREFIX + userid;

		final Map<String, String> userSessions = getMap(key);
		if (userSessions != null) {
			PipelineActionNoResult action = new PipelineActionNoResult() {
				@Override
				public void action(Pipeline pipeline) {
					Iterator<Map.Entry<String, String>> entries = userSessions.entrySet().iterator();
					while (entries.hasNext()) {
						Map.Entry<String, String> entry = entries.next();
						String t = entry.getKey();

						// 逐个儿删除业务key
						pipeline.del(t);
					}

					// 业务上的
					pipeline.del(key);
				}
			};
			execute(action);
		}
	}

	// ------------------------------------------------------------------------------------------//
	// --------------------------------------缓存辅助相关私有方法-----------------------------------//
	// ------------------------------------------------------------------------------------------//
	private void set(final String key, final String value) {
		execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				jedis.set(key, value);
			}
		});
	}

	private String get(final String key) {
		return execute(new JedisAction<String>() {
			@Override
			public String action(Jedis jedis) {
				return jedis.get(key);
			}
		});
	}

	private void execute(JedisActionNoResult jedisAction) throws JedisException {
		Jedis jedis = null;
		boolean broken = false;
		try {
			jedis = getSessionJedisPool().getResource();
			jedisAction.action(jedis);
		} catch (JedisException e) {
			broken = handleJedisException(e);
			throw e;
		} finally {
			closeResource(jedis, broken);
		}
	}

	private <T> T execute(JedisAction<T> jedisAction) throws JedisException {
		Jedis jedis = null;
		boolean broken = false;
		try {
			jedis = getSessionJedisPool().getResource();
			return jedisAction.action(jedis);
		} catch (JedisException e) {
			broken = handleJedisException(e);
			throw e;
		} finally {
			closeResource(jedis, broken);
		}
	}

	private Boolean del(final String... keys) {
		return execute(new JedisAction<Boolean>() {

			@Override
			public Boolean action(Jedis jedis) {
				return jedis.del(keys) == keys.length ? true : false;
			}
		});
	}

	// online session
	private void sessionHset(final String key, final String fieldName, final String value) {
		execute(new JedisActionNoResult() {

			@Override
			public void action(Jedis jedis) {
				jedis.hset(key, fieldName, value);
			}
		});
	}

	// online session
	private String sessionHget(final String key, final String fieldName) {
		return execute(new JedisAction<String>() {
			@Override
			public String action(Jedis jedis) {
				return jedis.hget(key, fieldName);
			}
		});
	}

	private Map<String, String> getMap(final String key) {
		return execute(new JedisAction<Map<String, String>>() {
			@Override
			public Map<String, String> action(Jedis jedis) {
				return jedis.hgetAll(key);
			}
		});
	}

	private boolean handleJedisException(JedisException jedisException) {
		if (jedisException instanceof JedisConnectionException) {
			logger.error("Redis connection " + getSessionJedisPool().getAddress() + " lost.", jedisException);
		} else if (jedisException instanceof JedisDataException) {
			if ((jedisException.getMessage() != null) && (jedisException.getMessage().indexOf("READONLY") != -1)) {
				logger.error("Redis connection " + getSessionJedisPool().getAddress() + " are read-only slave.",
						jedisException);
			} else {
				return false;
			}
		} else {
			logger.error("Jedis exception happen.", jedisException);
		}
		return true;
	}

	private void closeResource(Jedis jedis, boolean conectionBroken) {
		try {
			if (conectionBroken) {
				getSessionJedisPool().returnBrokenResource(jedis);
			} else {
				getSessionJedisPool().returnResource(jedis);
			}
		} catch (Exception e) {
			logger.error("return back jedis failed, will fore close the jedis.", e);
			JedisUtils.destroyJedis(jedis);
		}
	}

	private void execute(PipelineActionNoResult pipelineAction) throws JedisException {
		Jedis jedis = null;
		boolean broken = false;
		try {
			jedis = getSessionJedisPool().getResource();
			Pipeline pipeline = jedis.pipelined();
			pipelineAction.action(pipeline);
			pipeline.sync();
		} catch (JedisException e) {
			broken = handleJedisException(e);
			throw e;
		} finally {
			closeResource(jedis, broken);
		}
	}
	
	public UserContext getUserContext() {
		String sid = SESSION_PREFIX + InvocationInfoProxy.getUserid();
		String userContextStr = (String) getSessionCacheAttribute(sid, InvocationInfoProxy.getToken());
		UserContext userContext = JSONObject.parseObject(userContextStr, UserContext.class);
		return userContext;
	}

}
