package com.ejianc.framework.auth.shiro;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.ServletContext;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.subject.Subject;
import org.apache.shiro.web.filter.AccessControlFilter;
import org.apache.shiro.web.util.WebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.auth.EjcAuthConfiguration;
import com.ejianc.framework.auth.log.constants.LogConstants;
import com.ejianc.framework.auth.log.utils.ThreadCallerIdGenerator;
import com.ejianc.framework.auth.token.ITokenProcessor;
import com.ejianc.framework.auth.token.TokenFactory;
import com.ejianc.framework.auth.token.TokenParameter;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.util.CookieUtil;


public class StatelessAuthcFilter extends AccessControlFilter {

	private static final Logger log = LoggerFactory.getLogger(StatelessAuthcFilter.class);

	public static final int HTTP_STATUS_AUTH = 306;
	private String sysid;
	@Autowired
	private TokenFactory tokenFactory;

	private String[] esc = new String[] { "/login", "/formLogin", ".jpg", ".png", ".gif", ".css", ".js", ".html", ".jpeg", "/oauth_login", "/oauth_approval" };

	private List<String> excludCongtextKeys = Arrays.asList("u_sysid", "tenantid", "u_callid", "u_usercode", "token",
			"u_logints", "u_locale", "u_theme", "u_timezone", "current_user_name", "call_thread_id",
			"current_tenant_id");

	public void setSysid(String sysid) {
		this.sysid = sysid;
	}

	public void setTokenFactory(TokenFactory tokenFactory) {
		this.tokenFactory = tokenFactory;
	}

	public void setEsc(String[] esc) {
		this.esc = esc;
	}

	public void setExcludCongtextKeys(List<String> excludCongtextKeys) {
		this.excludCongtextKeys = excludCongtextKeys;
	}

	@Override
	protected boolean isAccessAllowed(ServletRequest request, ServletResponse response, Object mappedValue)
			throws Exception {
		return false;
	}

	@Override
	protected boolean onAccessDenied(ServletRequest request, ServletResponse response) throws Exception {
		boolean isAjax = isAjax(request);

		// 1、客户端发送来的摘要
		HttpServletRequest hReq = (HttpServletRequest) request;
		HttpServletRequest httpRequest = hReq;
		Cookie[] cookies = httpRequest.getCookies();

		// 从cookie中获取
		String cookieUserId = null;
		String cookieUserCode = null;
		String theme = null;
		String locale = null;
		String timeZone = null;
		String logints = null;
		String callerThreadId = null;
		String orgId = null;
		String userType = null;
		String tokenStr = null;
		String employeeId = null;
		if (cookies != null && cookies.length > 0) { //先从cookie中获取
			cookieUserId = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_USERID);
			cookieUserCode = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_USERCODE);
			theme = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_THEME);
			locale = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_LOCALE);
			timeZone = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_TIMEZONE);
			logints = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_LOGINTS);
			callerThreadId = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_CALLID);
			tokenStr = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_TOKEN);
			employeeId = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_EMPLOYEEID);
		}

		String authority = httpRequest.getHeader("authority");
		// 如果header中包含，则以header为主，否则，以cookie为主
		if (StringUtils.isNotBlank(authority)) {
			Set<Cookie> cookieSet = new HashSet<Cookie>();
			String[] ac = authority.split(";");
			for (String s : ac) {
				String[] cookieArr = s.split("=");
				String key = StringUtils.trim(cookieArr[0]);
				String value = StringUtils.trim(cookieArr[1]);
				Cookie cookie = new Cookie(key, value);
				cookieSet.add(cookie);
			}
			cookies = cookieSet.toArray(new Cookie[] {});
			tokenStr = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_TOKEN);
		} else {
			if(StringUtils.isBlank(tokenStr)) {
				log.error("no-token-in-cookie&header");
			}
		}
		// 从authority中获取
		if (StringUtils.isBlank(cookieUserId)) cookieUserId = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_USERID);
		if (StringUtils.isBlank(cookieUserCode)) cookieUserCode = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_USERCODE);
		if (StringUtils.isNoneBlank(theme)) theme = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_THEME);
		if (StringUtils.isBlank(locale)) locale = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_LOCALE);
		if (StringUtils.isBlank(timeZone)) timeZone = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_TIMEZONE);
		if (StringUtils.isBlank(logints)) logints = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_LOGINTS);
		if (StringUtils.isBlank(callerThreadId)) callerThreadId = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_CALLID);
		if(StringUtils.isBlank(employeeId)) employeeId = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_EMPLOYEEID);
		orgId = CookieUtil.findCookieValue(cookies, AuthConstants.PARAM_ORGID);
		userType = CookieUtil.findCookieValue(cookies, AuthConstants.USER_TYPE);
		if (StringUtils.isEmpty(sysid)) throw new Exception("sysid is empty! add  sysid  parameter to 'StatelessAuthcFilter' bean in application-shiro.xml");
		
		
		// 2、客户端传入的用户身份
		String username = request.getParameter(AuthConstants.PARAM_USERID);
		if (username == null && StringUtils.isNotBlank(cookieUserId)) {
			username = cookieUserId;
		}

		boolean needCheck = !include(hReq);
		if (needCheck) {
			if (tokenStr == null || username == null) {
				if (isAjax) {
					onAjaxAuthFail(request, response);
				} else {
					onLoginFail(request, response);
				}
				return false;
			}
			try {
				// 3、客户端请求的参数列表
				Map<String, String[]> params = new HashMap<String, String[]>(request.getParameterMap());
				
				ITokenProcessor tokenProcessor = tokenFactory.getTokenProcessor(tokenStr);
				TokenParameter tp = tokenProcessor.getTokenParameterFromCookie(cookies);
				// 4、生成无状态Token
				StatelessToken token = new StatelessToken(username, tokenProcessor, tp, params, new String(tokenStr));
				
				InvocationInfoProxy.setSysid(sysid);
				InvocationInfoProxy.setTheme(theme);
				InvocationInfoProxy.setLocale(locale);
				if (!StringUtils.isEmpty(timeZone)) {
					InvocationInfoProxy.setTimeZone(timeZone);				
				}
				
				// 5、委托给Realm进行登录
				getSubject(request, response).login(token); // 这个地方应该验证上下文信息中的正确性

				// 设置上下文变量
				InvocationInfoProxy.setUserid(Long.parseLong(username));
				InvocationInfoProxy.setUsercode(cookieUserCode);
				InvocationInfoProxy.setLogints(logints);
				InvocationInfoProxy.setTenantid(Long.parseLong(tp.getExt().get(AuthConstants.PARAM_TENANTID)));
				InvocationInfoProxy.setToken(tokenStr);
				InvocationInfoProxy.setCallid(callerThreadId);
				if(StringUtils.isNotBlank(orgId)) {
					InvocationInfoProxy.setOrgId(Long.parseLong(orgId));
				}
				if(StringUtils.isNotBlank(userType)) {
					InvocationInfoProxy.setUserType(userType);
				}
				if(StringUtils.isNotBlank(employeeId)) {
					InvocationInfoProxy.setEmployeeId(Long.parseLong(employeeId));
				}
				// 设置上下文携带的额外属性
				initExtendParams(cookies);

				initMDC();
				afterValidate(hReq);
			} catch (Exception e) {
				log.info("login failed");
				log.error(e.getMessage(), e);
				if (isAjax && e instanceof AuthenticationException) {
					onAjaxAuthFail(request, response); // 6、验证失败，返回ajax调用方信息
					return false;
				} else {
					onLoginFail(request, response); // 6、登录失败，跳转到登录页
					return false;
				}
			}
			return true;
		} else {
			return true;
		}

	}

	private boolean isAjax(ServletRequest request) {
		boolean isAjax = false;
		if (request instanceof HttpServletRequest) {
			HttpServletRequest rq = (HttpServletRequest) request;
			String requestType = rq.getHeader("X-Requested-With");
			if (requestType != null && "XMLHttpRequest".equals(requestType)) {
				isAjax = true;
			}
		}
		return isAjax;
	}

	protected void onAjaxAuthFail(ServletRequest request, ServletResponse resp) throws IOException {
		HttpServletResponse response = (HttpServletResponse) resp;
		JSONObject json = new JSONObject();
		json.put("msg", "auth check error!");
		response.setStatus(HttpServletResponse.SC_FORBIDDEN);
		response.getWriter().write(json.toString());
	}

	// 登录失败时默认返回306状态码
	protected void onLoginFail(ServletRequest request, ServletResponse response) throws IOException {
		HttpServletResponse httpResponse = (HttpServletResponse) response;
		httpResponse.setStatus(306);

		redirectToLogin(request, httpResponse);
	}

	@Override
	protected void redirectToLogin(ServletRequest request, ServletResponse response) throws IOException {
		HttpServletRequest hReq = (HttpServletRequest) request;
		String rURL = hReq.getRequestURI();
		String qryString = hReq.getQueryString();

		if (qryString != null && !qryString.isEmpty()) {
			qryString = qryString + "?" + hReq.getQueryString();
		}

		rURL = Base64.encodeBase64URLSafeString(rURL.getBytes());
		// 加入登录前地址
		String loginUrl = getLoginUrl() + "?r=" + rURL;
		WebUtils.issueRedirect(request, response, loginUrl);
	}

	public boolean include(HttpServletRequest request) {
		String u = request.getRequestURI();
		for (String e : esc) {
			if (u.endsWith(e)) {
				return true;
			}
		}
		
		ServletContext context = request.getServletContext();
		ApplicationContext ctx = WebApplicationContextUtils.getWebApplicationContext(context);
		EjcAuthConfiguration icopAuthConfig = ctx.getBean("ejcAuthConfiguration",EjcAuthConfiguration.class);
		String exeludeStr = icopAuthConfig.getFilterExcludes();
		if (StringUtils.isNotBlank(exeludeStr)) {
			String[] customExcludes = exeludeStr.split(",");
			for (String e : customExcludes) {
				if (u.endsWith(e)) {
					return true;
				}
			}
		}

		return false;
	}

	@Override
	public void afterCompletion(ServletRequest request, ServletResponse response, Exception exception)
			throws Exception {
		super.afterCompletion(request, response, exception);
		InvocationInfoProxy.reset();
		clearMDC();
	}

	// 设置上下文中的扩展参数，rest传递上下文时生效，Authority
	// header中排除固定key的其它信息都设置到InvocationInfoProxy的parameters
	private void initExtendParams(Cookie[] cookies) {
		for (Cookie cookie : cookies) {
			String cname = cookie.getName();
			String cvalue = cookie.getValue();
			if (!excludCongtextKeys.contains(cname)) {
				InvocationInfoProxy.setParameter(cname, cvalue);
			}
		}
	}

	private void initMDC() {
		String username = "";
		Subject subject = SecurityUtils.getSubject();
		if (subject != null && subject.getPrincipal() != null) {
			username = (String) SecurityUtils.getSubject().getPrincipal();
		}

		// MDC中记录用户信息
		MDC.put(LogConstants.CURRENT_USERNAME, username);
		String call_thread_id = InvocationInfoProxy.getCallid();
		if (StringUtils.isBlank(call_thread_id)) {
			call_thread_id = ThreadCallerIdGenerator.genCallerThreadId();
			InvocationInfoProxy.setCallid(call_thread_id);
		} else {
			MDC.put(LogConstants.THREAD_CALLID, call_thread_id);
		}

		MDC.put(LogConstants.CURRENT_TENANTID, String.valueOf(InvocationInfoProxy.getTenantid()));

		initCustomMDC();
	}

	protected void initCustomMDC() {
	}

	protected void afterValidate(HttpServletRequest hReq) {
	}

	protected void clearMDC() {
		// MDC中记录用户信息
		MDC.remove(LogConstants.CURRENT_USERNAME);
		MDC.remove(LogConstants.THREAD_CALLID);
		MDC.remove(LogConstants.CURRENT_TENANTID);

		clearCustomMDC();
	}

	protected void clearCustomMDC() {
	}

}