package com.ejianc.business.bid.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.bid.bean.AreaEntity;
import com.ejianc.business.bid.bean.BusOpportunityEntity;
import com.ejianc.business.bid.service.IAreaService;
import com.ejianc.business.bid.vo.AreaVO;
import com.ejianc.business.bid.vo.BusOpportunityVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import com.ejianc.framework.core.util.ResultAsTree;
import com.ejianc.framework.skeleton.template.BaseVO;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 行政地区控制器
 *
 * @author CJ
 * @Description:
 * @date 2021/10/12 10:46
 */
@RestController
@RequestMapping("/area/")
public class AreaController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());
    @Autowired
    private IAreaService areaService;

    @Autowired
    private CacheManager cacheManager;

    // 所有地区缓存
    private final static String EJC_ADMINISTRATIVE_AREA = "EJC_ADMINISTRATIVE_AREA";
    /**
     * 行政地区懒加载
     *
     * @param pid
     * @return
     */
    @GetMapping(value = "queryList")
    public CommonResponse<List<AreaVO>> queryList(@RequestParam(value = "pid", required = false) Long pid,
                                                  @RequestParam(value = "billCode", required = false) String billCode,
                                                  @RequestParam(value = "areaName", required = false) String areaName,
                                                  @RequestParam(value = "status", required = false) Integer status) {
        List<AreaVO> listMap = areaService.queryList(pid, billCode, areaName, status);
        if(CollectionUtils.isNotEmpty(listMap)) {
            listMap.stream().forEach(item -> {if(item.isLeaf()) {item.setChildren(null);}});
        }
        return CommonResponse.success("查询成功！", listMap);
    }

    /**
     * 行政地区信息保存
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<AreaVO> saveOrUpdate(@RequestBody AreaVO vo) {
        //编号重复校验
        QueryWrapper<AreaEntity> query = new QueryWrapper();
        query.eq("bill_code", vo.getBillCode());
        if(null != vo.getId()) {
            query.ne("id", vo.getId());
        }
        List<AreaEntity> billCodeEntityList = areaService.list(query);
        if(CollectionUtils.isNotEmpty(billCodeEntityList)) {
            return CommonResponse.error("保存失败，编码重复！");
        }

        AreaEntity saveEntity = null;
        if(null == vo.getId()) {
            saveEntity = BeanMapper.map(vo, AreaEntity.class);
            saveEntity.setId(IdWorker.getId());
            saveEntity.setStatus(AreaVO.STATUS_VALID);
        } else {
            saveEntity = areaService.selectById(vo.getId());

            saveEntity.setAreaName(vo.getAreaName());
            saveEntity.setBillCode(vo.getBillCode());

            saveEntity.setDescription(vo.getDescription());
        }
        if(null != vo.getPid()) {
            AreaEntity parent = areaService.selectById(vo.getPid());
            saveEntity.setPid(parent.getId());
            saveEntity.setpName(parent.getAreaName());
            saveEntity.setInnerCode(parent.getInnerCode() + "|" + saveEntity.getId());
        } else {
            saveEntity.setInnerCode(saveEntity.getId().toString());
        }
        saveEntity.setSequence(vo.getSequence());
        areaService.saveOrUpdate(saveEntity);
//        refreshCache();
        return CommonResponse.success("保存成功！", BeanMapper.map(saveEntity, AreaVO.class));
    }

    /**
     * 查询行政地区详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<AreaVO> queryDetail(@RequestParam(value = "id") Long id) {
        AreaVO resp = null;

        AreaEntity dbEntity = areaService.selectById(id);
        if(null != dbEntity) {
            resp = BeanMapper.map(dbEntity, AreaVO.class);
        }
        return CommonResponse.success("查询成功！", resp);
    }

    /**
     * 行政地区删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<AreaVO> vos) {
        List<Long> ids = vos.stream().map(AreaVO::getId).collect(Collectors.toList());
        //查询是否删除节点是否有子节点，如有则不允许删除
        QueryWrapper<AreaEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
        queryWrapper.in("pid", ids);
        List<AreaEntity> childList = areaService.list(queryWrapper);
        if(CollectionUtils.isNotEmpty(childList)) {
            return CommonResponse.error("删除数据存在子级节点，不允许删除！");
        }
        areaService.batchDelByIds(ids);
        refreshCache();
        return CommonResponse.success("删除成功！");
    }

    /**
     * 行政地区起停用状态改变
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "changeStatus")
    public CommonResponse<String> changeStatus(@RequestBody AreaVO vo) {
        AreaEntity dbEntity = areaService.selectById(vo.getId());
        if(null != dbEntity) {
            dbEntity.setStatus(vo.getStatus());
            areaService.saveOrUpdate(dbEntity, false);
        }
        refreshCache();
        return CommonResponse.success("操作成功！");
    }

    /**
     * 按照下了级联数据格式查询所有的行政地区
     *
     * @param pid
     * @return
     */
    @GetMapping(value = "loadAreaCascader")
    public List<Map<String, Object>> loadAreaCascader(@RequestParam(value = "pid", required = false) Long pid) {
        List<Map> listMap = areaService.queryAll(pid);
        List<Map<String, Object>> resp = ResultAsTree.createTreeData(listMap);

        return resp;
    }

    @GetMapping(value = "refAreaList")
    public CommonResponse<IPage<AreaVO>> refAreaList(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                    String condition,
                                                    String searchObject,
                                                    String searchText) {
        QueryParam param = new QueryParam();
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);

        param.getParams().put("status", new Parameter(QueryParam.EQ, AreaVO.STATUS_VALID));

        if(StringUtils.isNotBlank(condition)) {
            JSONObject _json = JSONObject.parseObject(condition);
            if(_json.containsKey("level")) {
                param.getParams().put("level", new Parameter(QueryParam.EQ, _json.get("level")));
            }
            if(_json.containsKey("pid")) {
                param.getParams().put("pid", new Parameter(QueryParam.EQ, _json.get("pid")));
            }
        }
        param.getOrderMap().put("sequence", "asc");

        IPage<AreaEntity> page = areaService.queryPage(param,false);
        IPage<AreaVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), AreaVO.class));
        pageData.getRecords().forEach(item -> item.setChildren(null));
        return CommonResponse.success("查询参照数据成功！",pageData);
    }

    @PostMapping(value = "areaExcelImport")
    public CommonResponse<JSONObject> areaExcelImport(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        List<JSONObject> successList = new ArrayList<>();
        List<JSONObject> errorList = new ArrayList<>();
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }
        if(isFailed) {
            return CommonResponse.error("文件格式不合法");
        }else{
            List<List<String>> result = ExcelReader.readExcel(mf);
            if(result != null && result.size() > 0) {
                if(result.size()>10000){
                    return CommonResponse.error("地区数据超过10000条，请分批上传！");
                }
                Map<String, JSONObject> areaBeanMap = new HashMap<>();
                JSONObject parent = null;
                AreaEntity parentEntity = null;
                for(int i = 0;i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    JSONObject area = new JSONObject();
                    area.put("id", IdWorker.getId());
                    area.put("billCode", datas.get(0));
                    area.put("areaName", datas.get(1));
                    area.put("pName", datas.get(2));
                    area.put("pCode", datas.get(3));
                    area.put("description", datas.get(4));

                    if(StringUtils.isBlank(datas.get(0))){
                        area.put("err", "行政地区编码为空！");
                        errorList.add(area);
                        continue;
                    }
                    if(StringUtils.isBlank(datas.get(1))){
                        area.put("err", "行政地区名称为空！");
                        errorList.add(area);
                        continue;
                    }

                    if(datas.get(0).length() > 50) {
                        area.put("err", "行政地区编码长度限制为50字以内！");
                        errorList.add(area);
                        continue;
                    }
                    if(datas.get(1).length() > 50) {
                        area.put("err", "行政地区名称长度限制为50字以内！");
                        errorList.add(area);
                        continue;
                    }
                    if(StringUtils.isNotBlank(datas.get(4)) && datas.get(4).length() > 200) {
                        area.put("err", "行政地区描述长度限制为200字以内！");
                        errorList.add(area);
                        continue;
                    }

                    if(areaBeanMap.containsKey(area.get("billCode").toString())) {
                        area.put("err", "行政地区编码重复！");
                        errorList.add(area);
                        continue;
                    } else {
                        QueryParam queryParam = new QueryParam();
                        queryParam.getParams().put("billCode", new Parameter(QueryParam.EQ, area.get("billCode")));
                        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
                        List<AreaEntity> list = areaService.queryList(queryParam, false);
                        if(CollectionUtils.isNotEmpty(list)) {
                            area.put("err", "行政地区编码重复！");
                            errorList.add(area);
                            continue;
                        }
                    }


                    if(null != area.get("pName") && StringUtils.isNotBlank(area.get("pName").toString())) {
                        //检测上级行政地区是否存在
                        parent = areaBeanMap.get(area.get("pCode"));
                        if(null != parent) {
                            area.put("pid", parent.get("id"));
                            area.put("pName", parent.get("areaName"));
                            area.put("level", Integer.valueOf(parent.get("level").toString()) + 1);
                        } else if(null == area.get("pCode") || StringUtils.isBlank(area.get("pCode").toString())) {
                            area.put("err", "上级行政地区编码为空！");
                            errorList.add(area);
                            continue;
                        } else  {
                            QueryParam parentQueryParam = new QueryParam();
                            parentQueryParam.getParams().put("area_name", new Parameter(QueryParam.EQ, area.get("pName")));
                            parentQueryParam.getParams().put("bill_code", new Parameter(QueryParam.EQ, area.get("pCode")));
                            parentQueryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
                            List<AreaEntity> parentList = areaService.queryList(parentQueryParam, false);
                            if(CollectionUtils.isEmpty(parentList)) {
                                area.put("err", "上级行政地区在系统中不存在！");
                                errorList.add(area);
                                continue;
                            } else {
                                parentEntity = parentList.get(0);
                                area.put("pid", parentEntity.getId());
                                area.put("pName", parentEntity.getAreaName());
                                area.put("level", parentEntity.getLevel() + 1);
                                area.put("id", IdWorker.getId());
                                area.put("innerCode", parentEntity.getInnerCode() + "|" + area.get("id"));
                            }
                        }
                    } else {
                        area.put("level", 1);
                        area.put("id", IdWorker.getId());
                        area.put("innerCode", area.get("id"));
                    }

                    //设置为启用状态
                    area.put("status", AreaVO.STATUS_VALID);
                    successList.add(area);
                    areaBeanMap.put(area.get("billCode").toString(), area);
                }
            }
        }
        JSONObject json = new JSONObject();
        json.put("successList", successList);
        json.put("errorList", errorList);
        return CommonResponse.success(json);
    }

    @PostMapping(value = "downloadImportTemplate")
    public void downloadImportTemplate(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "administrativeArea-import.xlsx", "行政地区导入模板");
    }

    @PostMapping(value = "saveAreaExcelImport")
    public CommonResponse<String> saveAreaExcelImport(@RequestBody List<JSONObject> importData) {
        if(ListUtil.isEmpty(importData)){
            return CommonResponse.error("导入的数据为空！");
        }
        List<AreaEntity> saveData = BeanMapper.mapList(importData, AreaEntity.class);
        areaService.saveOrUpdateBatch(saveData, saveData.size(), false);
        refreshCache();
        return CommonResponse.success("导入成功！");
    }

    @GetMapping("refreshCache")
    public void refreshCache(){
        Thread t = new Thread(() -> {
            logger.info("开始刷新地址缓存.......");
            List<JSONObject> data = queryArea();
            if(ListUtil.isNotEmpty(data)){
                cacheManager.set(EJC_ADMINISTRATIVE_AREA,JSONObject.toJSONString(data));
            }else {
                cacheManager.set(EJC_ADMINISTRATIVE_AREA,"[]");
            }
            logger.info("刷新地址缓存完成.......");
        });
        t.start();
    }

    @GetMapping(value = "queryAreaCascader")
    public JSONArray queryAreaCascader() {
        JSONArray data = JSONObject.parseArray(cacheManager.get(EJC_ADMINISTRATIVE_AREA).toString());
        logger.info(data.toJSONString());
        return data;
    }

    public List<JSONObject> queryArea() {
        QueryParam param = new QueryParam();
        param.getParams().put("status",new Parameter(QueryParam.EQ,1));
        param.getParams().put("pid",new Parameter(QueryParam.EQ,null));
        List<AreaEntity> listMap = areaService.queryList(param);
        List<JSONObject> resp = createTreeData(listMap);
        return resp;
    }


    public List<JSONObject> createTreeData(List<AreaEntity> list) {
        List<JSONObject> resp = new ArrayList<>();

        for(AreaEntity item:list) {
            JSONObject object = JSONObject.parseObject(JSONObject.toJSONString(item));
            object.put("value",item.getId()+"");
            object.put("title",item.getAreaName());
            object.put("label",item.getAreaName());
            resp.add(object);
        }
        for (JSONObject parent : resp) {
            setChildren(parent);
        }
        return resp;
    }

    private void setChildren(JSONObject parent){
        List<JSONObject> children = new ArrayList<>();

        QueryParam param = new QueryParam();
        param.getParams().put("status",new Parameter(QueryParam.EQ,1));
        param.getParams().put("pid",new Parameter(QueryParam.EQ,parent.getString("id")));
        List<AreaEntity> list = areaService.queryList(param);
        if(ListUtil.isNotEmpty(list)){
            for(AreaEntity item:list) {
                JSONObject child = JSONObject.parseObject(JSONObject.toJSONString(item));
                child.put("value",item.getId()+"");
                child.put("title",item.getAreaName());
                child.put("label",item.getAreaName());
                children.add(child);
            }
        }
        if(ListUtil.isNotEmpty(children)){
            parent.put("children",children);
            for (JSONObject child : children) {
                setChildren(child);
            }
        }
    }

}
