package com.ejianc.business.car.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.car.bean.*;
import com.ejianc.business.car.service.*;
import com.ejianc.business.car.vo.CarVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 车辆信息
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("car")
public class CarController implements Serializable {
    private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());


    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private ICarService service;

    @Autowired
    private SessionManager sessionManager;

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<CarVO> saveOrUpdate(@RequestBody CarVO saveOrUpdateVO) {
        Long id = saveOrUpdateVO.getId();
        LambdaQueryWrapper<CarEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(CarEntity::getTenantId, InvocationInfoProxy.getTenantid());
        queryWrapper.eq(CarEntity::getCode, saveOrUpdateVO.getCode());
        queryWrapper.ne(id != null, CarEntity::getId, id);
        int count = service.count(queryWrapper);
        if (count > 0) {
            throw new BusinessException("保存失败，车牌号已存在！");
        }
        CarEntity entity = BeanMapper.map(saveOrUpdateVO, CarEntity.class);
        service.saveOrUpdate(entity, false);
        CarVO vo = BeanMapper.map(entity, CarVO.class);
        return CommonResponse.success("保存或修改单据成功！", vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<CarVO> queryDetail(Long id) {
        CarEntity entity = service.selectById(id);
        CarVO vo = BeanMapper.map(entity, CarVO.class);
        return CommonResponse.success("查询详情数据成功！", vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<CarVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            for (CarVO vo : vos) {
                // 参数是单据类型编码字符串 根据需求是否打开下面代码
                /* CommonResponse<String> resp = billTypeApi.checkQuote("billTypeCode", vo.getId());
                if(!resp.isSuccess()){
                    return CommonResponse.error("删除失败！"+resp.getMsg());
                }*/
            }
        }
        service.removeByIds(vos.stream().map(CarVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<CarVO>> queryList(@RequestBody QueryParam param) {

        /** 车牌号、车辆名称、所属公司、车辆所属区域、车辆所属项目
         * */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("code", "carName", "orgName", "carAreaName", "carProjectName"));
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        /** 数据隔离 本下 */
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse = iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                    collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }

        IPage<CarEntity> page = service.queryPage(param, false);
        IPage<CarVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), CarVO.class));

        return CommonResponse.success("查询列表数据成功！", pageData);
    }


    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("code", "carName", "orgName", "carAreaName", "carProjectName"));
        param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<CarEntity> list = service.queryList(param);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", BeanMapper.mapList(list, CarVO.class));
        ExcelExport.getInstance().exportWithTrans("Car-export.xlsx", beans, response);
    }

    /**
     * @Description 参照
     * @Return void
     */
    @RequestMapping(value = "/refCarData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<CarVO>> refCarData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                   String condition,
                                                   String searchObject,
                                                   String searchText) {
        QueryParam param = new QueryParam();
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("code", "carName", "orgName", "carAreaName", "carProjectName"));
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        if (StringUtils.isNotEmpty(condition)) {
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
            // 如果带range条件则查询所有车辆
            if (null != _con.get("range")) {
                param.getParams().remove("orgId");
            }
        }
        /** 已提交的数据 或者 审批通过的数据 */
        param.getParams().put("billState", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));

        IPage<CarEntity> page = service.queryPage(param, false);
        IPage<CarVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), CarVO.class));

        return CommonResponse.success("查询参照数据成功！", pageData);
    }


    /**
     * @description: 车辆统计表
     *
     * @author songlx
     * @date: 2023/10/9
     */
    @RequestMapping(value = "/carCount", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> carCount() {
        return CommonResponse.success(service.carCount());
    }


    /**
     * @description: 车辆统计表穿透详情、 按类型和区域
     *
     * @author songlx
     * @date: 2023/10/9
     */
    @RequestMapping(value = "/carCountDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<CarVO>> carCountDetail(@RequestParam Long carTypeId, @RequestParam(required = false) Long carAreaId) {
        LambdaQueryWrapper<CarEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(CarEntity::getCarTypeId, carTypeId);
        queryWrapper.eq(carAreaId != null, CarEntity::getCarAreaId, carAreaId);
        queryWrapper.in(CarEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        queryWrapper.in(CarEntity::getOrgId, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        queryWrapper.orderByAsc(CarEntity::getCarModelId);
        List<CarEntity> list = service.list(queryWrapper);
        return CommonResponse.success(BeanMapper.mapList(list, CarVO.class));
    }


    /**
     * @description: 车辆统计表-- 导出
     *
     * @author songlx
     * @date: 2023/10/9
     */
    @RequestMapping(value = "exportCarCount", method = RequestMethod.POST)
    @ResponseBody
    public void exportCarCount(HttpServletResponse response) throws IOException {
        JSONObject dataObj = service.carCount();
        List<JSONObject> data = (List<JSONObject>) dataObj.get("data");
        List<JSONObject> head = (List<JSONObject>) dataObj.get("head");
        if (ListUtil.isEmpty(data)) {
            throw new BusinessException("未查询到数据！");
        }

        int lastColIndex = head.size();
        ServletOutputStream outputStream = response.getOutputStream();
        XSSFWorkbook workbook = new XSSFWorkbook();
        XSSFSheet sheet = workbook.createSheet();
        sheet.createFreezePane(0, 2, 0, 2);
        sheet.addMergedRegion(new CellRangeAddress(0, 0, 0, lastColIndex - 1));

        //首行标题
        XSSFRow row = sheet.createRow(0);
        CellStyle style = workbook.createCellStyle();
        style.setFillForegroundColor(IndexedColors.PALE_BLUE.getIndex());
        style.setFillPattern(CellStyle.SOLID_FOREGROUND);
        style.setAlignment(CellStyle.ALIGN_CENTER);
        XSSFCell cell = row.createCell(0);
        cell.setCellValue("车辆统计表");
        cell.setCellStyle(style);
        //第二行列名
        row = sheet.createRow(1);
        for (int a = 0; a < head.size(); a++) {
            cell = row.createCell(a);
            cell.setCellValue(head.get(a).get("name").toString());
            cell.setCellStyle(style);
        }
        for (int i = 0; i < lastColIndex; i++) {
            sheet.autoSizeColumn(i);
            sheet.setColumnWidth(i, sheet.getColumnWidth(i) * 35 / 10);
        }

        CellStyle dataStyle = workbook.createCellStyle();
        dataStyle.setAlignment(CellStyle.ALIGN_RIGHT);
        for (int i = 0; i < data.size(); i++) {
            JSONObject vo = data.get(i);
            row = sheet.createRow(i + 2);
            for (int j = 0; j < lastColIndex; j++) {
                String key = head.get(j).get("key").toString();
                cell = row.createCell(j);
                String val = vo.get(key) == null ? "" : vo.get(key).toString();
                cell.setCellValue(val);
                if (!"carType".equals(key)) {
                    cell.setCellStyle(dataStyle);
                }
            }
        }

        workbook.write(outputStream);
    }


    @Autowired
    private IMaintainService maintainService;

    /**
     * @description: 查询维修记录
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryMaintainInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryMaintainInfo(@RequestParam Long id) {
        LambdaQueryWrapper<MaintainEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(MaintainEntity::getCarId, id);
        wrapper.in(MaintainEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(MaintainEntity::getCreateTime);
        List<MaintainEntity> list = maintainService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            BigDecimal maintainTaxMny = list.stream().filter(t -> t.getMaintainTaxMny() != null).map(MaintainEntity::getMaintainTaxMny).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            resObj.put("maintainTaxMny", maintainTaxMny);
            resObj.put("maintainCount", list.size());
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }


    @Autowired
    private IDispatchService dispatchService;

    /**
     * @description: 查询调度记录
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryDispatch", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryDispatch(@RequestParam Long id) {
        LambdaQueryWrapper<DispatchEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(DispatchEntity::getCarId, id);
        wrapper.in(DispatchEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(DispatchEntity::getCreateTime);
        List<DispatchEntity> list = dispatchService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            resObj.put("dispatchCount", list.size());
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }



    @Autowired
    private IOilRecordService oilRecordService;

    /**
     * @description: 查询用油记录
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryOilRecord", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryOilRecord(@RequestParam Long id) {
        LambdaQueryWrapper<OilRecordEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(OilRecordEntity::getCarId, id);
        wrapper.in(OilRecordEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(OilRecordEntity::getCreateTime);
        List<OilRecordEntity> list = oilRecordService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            BigDecimal totalOilConsumption = list.stream().filter(t -> t.getTotalOilConsumption() != null).map(OilRecordEntity::getTotalOilConsumption).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            resObj.put("totalOilConsumption", totalOilConsumption);
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }


    @Autowired
    private IOilFeeService oilFeeService;

    /**
     * @description: 查询油卡充值
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryOilFee", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryOilFee(@RequestParam Long id) {
        LambdaQueryWrapper<OilFeeEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(OilFeeEntity::getCarId, id);
        wrapper.in(OilFeeEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(OilFeeEntity::getCreateTime);
        List<OilFeeEntity> list = oilFeeService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            BigDecimal useAmount = list.stream().filter(t -> t.getUseAmount() != null).map(OilFeeEntity::getUseAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            BigDecimal totalRechargeAmount = list.stream().filter(t -> t.getTotalRechargeAmount() != null).map(OilFeeEntity::getTotalRechargeAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            resObj.put("useAmount", useAmount);
            resObj.put("totalRechargeAmount", totalRechargeAmount);
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }
}
