package com.ejianc.business.car.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.car.bean.*;
import com.ejianc.business.car.service.*;
import com.ejianc.business.car.vo.CarVO;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IDefdocApi;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 车辆信息
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("car")
public class CarController implements Serializable {
    private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());


    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private ICarService service;

    @Autowired
    private SessionManager sessionManager;

    private final String CAR_TYPE = "car-type";
    private final String CAR_MODEL = "car-model";
    private final String CAR_AREA = "car-area";

    @Autowired
    private IDefdocApi defdocApi;

    @Autowired
    private IProjectApi projectApi;

    SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");


    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<CarVO> saveOrUpdate(@RequestBody CarVO saveOrUpdateVO) {
        Long id = saveOrUpdateVO.getId();
        LambdaQueryWrapper<CarEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(CarEntity::getTenantId, InvocationInfoProxy.getTenantid());
        queryWrapper.eq(CarEntity::getCode, saveOrUpdateVO.getCode());
        queryWrapper.ne(id != null, CarEntity::getId, id);
        int count = service.count(queryWrapper);
        if (count > 0) {
            throw new BusinessException("保存失败，车牌号已存在！");
        }
        CarEntity entity = BeanMapper.map(saveOrUpdateVO, CarEntity.class);
        service.saveOrUpdate(entity, false);
        CarVO vo = BeanMapper.map(entity, CarVO.class);
        return CommonResponse.success("保存或修改单据成功！", vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<CarVO> queryDetail(Long id) {
        CarEntity entity = service.selectById(id);
        CarVO vo = BeanMapper.map(entity, CarVO.class);
        return CommonResponse.success("查询详情数据成功！", vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<CarVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            for (CarVO vo : vos) {
                // 参数是单据类型编码字符串 根据需求是否打开下面代码
                /* CommonResponse<String> resp = billTypeApi.checkQuote("billTypeCode", vo.getId());
                if(!resp.isSuccess()){
                    return CommonResponse.error("删除失败！"+resp.getMsg());
                }*/
            }
        }
        service.removeByIds(vos.stream().map(CarVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<CarVO>> queryList(@RequestBody QueryParam param) {

        /** 车牌号、车辆名称、所属公司、车辆所属区域、车辆所属项目
         * */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("code", "carName", "orgName", "carAreaName", "carProjectName"));
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        /** 数据隔离 本下 */
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse = iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                    collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }

        IPage<CarEntity> page = service.queryPage(param, false);
        IPage<CarVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), CarVO.class));

        return CommonResponse.success("查询列表数据成功！", pageData);
    }


    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("code", "carName", "orgName", "carAreaName", "carProjectName"));
        param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<CarEntity> list = service.queryList(param);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", BeanMapper.mapList(list, CarVO.class));
        ExcelExport.getInstance().exportWithTrans("Car-export.xlsx", beans, response);
    }

    /**
     * @Description 参照
     * @Return void
     */
    @RequestMapping(value = "/refCarData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<CarVO>> refCarData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                   String condition,
                                                   String searchObject,
                                                   String searchText) {
        QueryParam param = new QueryParam();
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("code", "carName", "orgName", "carAreaName", "carProjectName"));
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        if (StringUtils.isNotEmpty(condition)) {
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
            // 如果带range条件则查询所有车辆
            if (null != _con.get("range")) {
                param.getParams().remove("orgId");
            }
        }
        /** 已提交的数据 或者 审批通过的数据 */
        param.getParams().put("billState", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));

        IPage<CarEntity> page = service.queryPage(param, false);
        IPage<CarVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), CarVO.class));

        return CommonResponse.success("查询参照数据成功！", pageData);
    }


    /**
     * @description: 车辆统计表
     *
     * @author songlx
     * @date: 2023/10/9
     */
    @RequestMapping(value = "/carCount", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> carCount() {
        return CommonResponse.success(service.carCount());
    }


    /**
     * @description: 车辆统计表穿透详情、 按类型和区域
     *
     * @author songlx
     * @date: 2023/10/9
     */
    @RequestMapping(value = "/carCountDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<CarVO>> carCountDetail(@RequestParam Long carTypeId, @RequestParam(required = false) Long carAreaId) {
        LambdaQueryWrapper<CarEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(CarEntity::getCarTypeId, carTypeId);
        queryWrapper.eq(carAreaId != null, CarEntity::getCarAreaId, carAreaId);
        queryWrapper.in(CarEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        queryWrapper.in(CarEntity::getOrgId, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        queryWrapper.orderByAsc(CarEntity::getCarModelId);
        List<CarEntity> list = service.list(queryWrapper);
        return CommonResponse.success(BeanMapper.mapList(list, CarVO.class));
    }


    /**
     * @description: 车辆统计表-- 导出
     *
     * @author songlx
     * @date: 2023/10/9
     */
    @RequestMapping(value = "exportCarCount", method = RequestMethod.POST)
    @ResponseBody
    public void exportCarCount(HttpServletResponse response) throws IOException {
        JSONObject dataObj = service.carCount();
        List<JSONObject> data = (List<JSONObject>) dataObj.get("data");
        List<JSONObject> head = (List<JSONObject>) dataObj.get("head");
        if (ListUtil.isEmpty(data)) {
            throw new BusinessException("未查询到数据！");
        }

        int lastColIndex = head.size();
        ServletOutputStream outputStream = response.getOutputStream();
        XSSFWorkbook workbook = new XSSFWorkbook();
        XSSFSheet sheet = workbook.createSheet();
        sheet.createFreezePane(0, 2, 0, 2);
        sheet.addMergedRegion(new CellRangeAddress(0, 0, 0, lastColIndex - 1));

        //首行标题
        XSSFRow row = sheet.createRow(0);
        CellStyle style = workbook.createCellStyle();
        style.setFillForegroundColor(IndexedColors.PALE_BLUE.getIndex());
        style.setFillPattern(CellStyle.SOLID_FOREGROUND);
        style.setAlignment(CellStyle.ALIGN_CENTER);
        XSSFCell cell = row.createCell(0);
        cell.setCellValue("车辆统计表");
        cell.setCellStyle(style);
        //第二行列名
        row = sheet.createRow(1);
        for (int a = 0; a < head.size(); a++) {
            cell = row.createCell(a);
            cell.setCellValue(head.get(a).get("name").toString());
            cell.setCellStyle(style);
        }
        for (int i = 0; i < lastColIndex; i++) {
            sheet.autoSizeColumn(i);
            sheet.setColumnWidth(i, sheet.getColumnWidth(i) * 35 / 10);
        }

        CellStyle dataStyle = workbook.createCellStyle();
        dataStyle.setAlignment(CellStyle.ALIGN_RIGHT);
        for (int i = 0; i < data.size(); i++) {
            JSONObject vo = data.get(i);
            row = sheet.createRow(i + 2);
            for (int j = 0; j < lastColIndex; j++) {
                String key = head.get(j).get("key").toString();
                cell = row.createCell(j);
                String val = vo.get(key) == null ? "" : vo.get(key).toString();
                cell.setCellValue(val);
                if (!"carType".equals(key)) {
                    cell.setCellStyle(dataStyle);
                }
            }
        }

        workbook.write(outputStream);
    }


    @Autowired
    private IMaintainService maintainService;

    /**
     * @description: 查询维修记录
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryMaintainInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryMaintainInfo(@RequestParam Long id) {
        LambdaQueryWrapper<MaintainEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(MaintainEntity::getCarId, id);
        wrapper.in(MaintainEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(MaintainEntity::getCreateTime);
        List<MaintainEntity> list = maintainService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            BigDecimal maintainTaxMny = list.stream().filter(t -> t.getMaintainTaxMny() != null).map(MaintainEntity::getMaintainTaxMny).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            resObj.put("maintainTaxMny", maintainTaxMny);
            resObj.put("maintainCount", list.size());
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }


    @Autowired
    private IDispatchService dispatchService;

    /**
     * @description: 查询调度记录
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryDispatch", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryDispatch(@RequestParam Long id) {
        LambdaQueryWrapper<DispatchEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(DispatchEntity::getCarId, id);
        wrapper.in(DispatchEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(DispatchEntity::getCreateTime);
        List<DispatchEntity> list = dispatchService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            resObj.put("dispatchCount", list.size());
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }


    @Autowired
    private IOilRecordService oilRecordService;

    /**
     * @description: 查询用油记录
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryOilRecord", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryOilRecord(@RequestParam Long id) {
        LambdaQueryWrapper<OilRecordEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(OilRecordEntity::getCarId, id);
        wrapper.in(OilRecordEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(OilRecordEntity::getCreateTime);
        List<OilRecordEntity> list = oilRecordService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            BigDecimal totalOilConsumption = list.stream().filter(t -> t.getTotalOilConsumption() != null).map(OilRecordEntity::getTotalOilConsumption).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            resObj.put("totalOilConsumption", totalOilConsumption);
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }


    @Autowired
    private IOilFeeService oilFeeService;

    /**
     * @description: 查询油卡充值
     *
     * @param id 车辆id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2023/10/10
     */
    @RequestMapping(value = "/queryOilFee", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryOilFee(@RequestParam Long id) {
        LambdaQueryWrapper<OilFeeEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(OilFeeEntity::getCarId, id);
        wrapper.in(OilFeeEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.orderByDesc(OilFeeEntity::getCreateTime);
        List<OilFeeEntity> list = oilFeeService.list(wrapper);

        JSONObject resObj = new JSONObject();
        if (CollectionUtils.isNotEmpty(list)) {
            BigDecimal useAmount = list.stream().filter(t -> t.getUseAmount() != null).map(OilFeeEntity::getUseAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            BigDecimal totalRechargeAmount = list.stream().filter(t -> t.getTotalRechargeAmount() != null).map(OilFeeEntity::getTotalRechargeAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
            resObj.put("useAmount", useAmount);
            resObj.put("totalRechargeAmount", totalRechargeAmount);
        }
        resObj.put("list", list);
        return CommonResponse.success(resObj);
    }

    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "Car-import.xlsx", "车辆信息导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse response) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }

        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            List<CarVO> successList = new ArrayList<>();
            List<CarVO> errorList = new ArrayList<>();
            if (result != null && result.size() > 0) {
                CommonResponse<List<DefdocDetailVO>> carTypeRes = defdocApi.getDefDocByDefCode(CAR_TYPE);
                Map<String, Long> carTypeMap = carTypeRes.getData().stream().collect(Collectors.toMap(DefdocDetailVO::getName, DefdocDetailVO::getId));
                CommonResponse<List<DefdocDetailVO>> carAreaRes = defdocApi.getDefDocByDefCode(CAR_AREA);
                Map<String, Long> carAreaMap = carAreaRes.getData().stream().collect(Collectors.toMap(DefdocDetailVO::getName, DefdocDetailVO::getId));
                CommonResponse<List<DefdocDetailVO>> carModelRes = defdocApi.getDefDocByDefCode(CAR_MODEL);
                Map<String, Long> carModelMap = carModelRes.getData().stream().collect(Collectors.toMap(DefdocDetailVO::getName, DefdocDetailVO::getId));

                CommonResponse<List<DefdocDetailVO>> carStatusRes = defdocApi.getDefDocByDefCode("car_car_status");
                Map<String, Long> carStatusMap = carStatusRes.getData().stream().collect(Collectors.toMap(DefdocDetailVO::getName, DefdocDetailVO::getId));

                LambdaQueryWrapper<CarEntity> queryWrapper = new LambdaQueryWrapper<>();
                queryWrapper.eq(CarEntity::getTenantId, InvocationInfoProxy.getTenantid());
                List<CarEntity> list = service.list(queryWrapper);
                Set<String> codeList = list.stream().map(CarEntity::getCode).collect(Collectors.toSet());
                CommonResponse<List<OrgVO>> orgByTenantId = iOrgApi.findOrgByType(InvocationInfoProxy.getTenantid(), Arrays.asList(1, 2, 3), null);

                Map<String, Long> orgMap = orgByTenantId.getData().stream().
                        collect(Collectors.toMap(
                                OrgVO::getName,
                                OrgVO::getId,
                                (val1, val2) -> val2)
                        );

                CommonResponse<List<ProjectRegisterVO>> commonResponse = projectApi.queryChildrenProjectByOrgId(InvocationInfoProxy.getOrgId());
                Map<String, Long> projectMap = commonResponse.getData().stream().collect(Collectors.toMap(ProjectRegisterVO::getName, ProjectRegisterVO::getId, (k1, k2) -> k2));

                for (int i = 0; i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    CarVO carVO = new CarVO();
                    StringBuilder errMsg = new StringBuilder();
                    //车辆类型
                    if (StringUtils.isBlank(datas.get(0))) {
                        errMsg.append("`车辆类型为必填项`");
                    } else {
                        carVO.setCarTypeName(datas.get(0).trim());
                        Long carTypeId = carTypeMap.get(carVO.getCarTypeName());
                        if (carTypeId == null) {
                            errMsg.append("`车辆类型不存在`");
                        }
                        carVO.setCarTypeId(carTypeId);
                    }
                    //型号
                    if (StringUtils.isBlank(datas.get(1))) {
                        errMsg.append("`型号为必填项`");
                    } else {
                        carVO.setCarModelName(datas.get(1).trim());
                        Long carModelId = carModelMap.get(carVO.getCarModelName());
                        if (carModelId == null) {
                            errMsg.append("`型号不存在`");
                        }
                        carVO.setCarModelId(carModelId);
                    }
                    // 名称
                    if (StringUtils.isBlank(datas.get(2))) {
                        errMsg.append("`车辆名称为必填项`");
                    } else {
                        carVO.setCarName(datas.get(2).trim());
                    }
                    // 车牌号
                    if (StringUtils.isBlank(datas.get(3))) {
                        errMsg.append("`车牌号为必填项`");
                    } else {
                        carVO.setCode(datas.get(3).trim());
                        if (codeList.contains(carVO.getCode())) {
                            errMsg.append("`车牌号已在库或重复`");
                        } else {
                            codeList.add(carVO.getCode());
                        }
                    }
                    // 所属公司
                    if (StringUtils.isBlank(datas.get(4).trim())) {
                        errMsg.append("`所属公司为必填项`");
                    } else {
                        carVO.setOrgName(datas.get(4).trim());
                        Long orgId = orgMap.get(carVO.getOrgName());
                        if (orgId == null) {
                            errMsg.append("`所属公司不存在`");
                        }
                        carVO.setOrgId(orgId);
                    }

                    // 所属项目
                    if (StringUtils.isNotBlank(datas.get(5))) {
                        carVO.setCarProjectName(datas.get(5).trim());
                        Long projectId = projectMap.get(carVO.getCarProjectName());
                        if (projectId == null) {
                            errMsg.append("`所属项目不存在`");
                        }
                        carVO.setCarProjectId(projectId);
                    }


                    // 车辆状态
                    if (StringUtils.isNotBlank(datas.get(6).trim())) {
                        carVO.setCarStatusName(datas.get(6).trim());
                        Long carStatusId = carStatusMap.get(carVO.getCarStatusName());
                        if (carStatusId == null) {
                            errMsg.append("`车辆状态不存在`");
                        }
                        carVO.setCarStatusId(carStatusId);
                    }
                    // 品牌
                    if (StringUtils.isNotBlank(datas.get(7).trim())) {
                        carVO.setBrand(datas.get(7).trim());
                    }
                    // 车架号
                    if (StringUtils.isNotBlank(datas.get(8).trim())) {
                        carVO.setVinNo(datas.get(8).trim());
                    }
                    // 发动机号
                    if (StringUtils.isNotBlank(datas.get(9).trim())) {
                        carVO.setEngineNo(datas.get(9).trim());
                    }
                    // 注册日期
                    if (StringUtils.isNotBlank(datas.get(10).trim())) {
                        try {
                            Date parse = sdf.parse(datas.get(10).trim());
                            carVO.setRegistDate(parse);
                        } catch (ParseException e) {
                            errMsg.append("`注册日期格式不正确`");
                        }
                    }
                    // 发证日期
                    if (StringUtils.isNotBlank(datas.get(11).trim())) {
                        try {
                            Date parse = sdf.parse(datas.get(11).trim());
                            carVO.setCertDate(parse);
                        } catch (ParseException e) {
                            errMsg.append("`发证日期格式不正确`");
                        }
                    }
                    // 总质量
                    if (StringUtils.isNotBlank(datas.get(12).trim())) {
                        try {
                            carVO.setTotalWeight(new BigDecimal(datas.get(12).trim()));
                        } catch (Exception e) {
                            errMsg.append("`总质量必须为数字`");
                        }
                    }
                    // 整备质量
                    if (StringUtils.isNotBlank(datas.get(13).trim())) {
                        try {
                            carVO.setAllWeight(new BigDecimal(datas.get(13).trim()));
                        } catch (Exception e) {
                            errMsg.append("`整备质量必须为数字`");
                        }
                    }
                    // 核定载质量
                    if (StringUtils.isNotBlank(datas.get(14).trim())) {
                        try {
                            carVO.setCheckWeight(new BigDecimal(datas.get(14).trim()));
                        } catch (Exception e) {
                            errMsg.append("`整备质量必须为数字`");
                        }
                    }
                    // 购车价
                    if (StringUtils.isNotBlank(datas.get(15).trim())) {
                        try {
                            carVO.setBuyPrice(new BigDecimal(datas.get(15).trim()));
                        } catch (Exception e) {
                            errMsg.append("`整备质量必须为数字`");
                        }
                    }
                    // 报废时间
                    if (StringUtils.isNotBlank(datas.get(16).trim())) {
                        try {
                            Date parse = sdf.parse(datas.get(16).trim());
                            carVO.setScrapDate(parse);
                        } catch (ParseException e) {
                            errMsg.append("`报废时间格式不正确`");
                        }
                    }
                    // 外廓尺寸
                    if (StringUtils.isNotBlank(datas.get(17).trim())) {
                        carVO.setCarSize(datas.get(17).trim());
                    }
                    // 年检时间
                    if (StringUtils.isNotBlank(datas.get(18).trim())) {
                        try {
                            Date parse = sdf.parse(datas.get(18).trim());
                            carVO.setYearCheckDate(parse);
                        } catch (ParseException e) {
                            errMsg.append("`年检时间格式不正确`");
                        }
                    } else {
                        errMsg.append("`年检时间为必填项`");
                    }
                    // 保险公司
                    if (StringUtils.isNotBlank(datas.get(19).trim())) {
                        carVO.setInsureCompany(datas.get(19).trim());
                    }
                    // 保险时间(强制险)
                    if (StringUtils.isNotBlank(datas.get(20).trim())) {
                        try {
                            Date parse = sdf.parse(datas.get(20).trim());
                            carVO.setInsureForceDate(parse);
                        } catch (ParseException e) {
                            errMsg.append("`保险时间(强制险)格式不正确`");
                        }
                    }
                    // 保险时间(商业险)
                    if (StringUtils.isNotBlank(datas.get(21).trim())) {
                        try {
                            Date parse = sdf.parse(datas.get(21).trim());
                            carVO.setInsureBusinessDate(parse);
                        } catch (ParseException e) {
                            errMsg.append("`保险时间(商业险)格式不正确`");
                        }
                    }
                    // 油卡号
                    if (StringUtils.isNotBlank(datas.get(22).trim())) {
                        carVO.setOilCard(datas.get(22).trim());
                    }
                    // 车辆所在区域
                    if (StringUtils.isBlank(datas.get(23))) {
                        errMsg.append("`车辆所在区域为必填项`");
                    } else {
                        carVO.setCarAreaName(datas.get(23).trim());
                        Long carAreaId = carAreaMap.get(carVO.getCarAreaName());
                        if (carAreaId == null) {
                            errMsg.append("`车辆所在区域不存在`");
                        }
                        carVO.setCarAreaId(carAreaId);
                    }
                    // 备注
                    if (StringUtils.isNotBlank(datas.get(24).trim())) {
                        carVO.setMemo(datas.get(24).trim());
                    }
                    if (StringUtils.isNotBlank(errMsg.toString())) {
                        carVO.setId(Long.valueOf(i));
                        carVO.setErrorMessage(errMsg.toString());
                        errorList.add(carVO);
                    } else {
                        UserContext userContext = sessionManager.getUserContext();
                        carVO.setOrgId(userContext.getOrgId());
                        carVO.setOrgName(userContext.getOrgName());
                        carVO.setEmployeeId(userContext.getEmployeeId());
                        carVO.setEmployeeName(userContext.getEmployeeName());
                        carVO.setDepartmentName(userContext.getDeptName());
                        successList.add(carVO);
                    }
                }
            }
            JSONObject json = new JSONObject();
            json.put("successList", successList);
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        }
    }

    @RequestMapping(value = "/saveImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> saveImport(@RequestBody List<CarVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            service.saveBatch(BeanMapper.mapList(vos, CarEntity.class));
        }
        return CommonResponse.success("导入保存成功！");
    }
}
