package com.ejianc.business.contractbase.controller;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.contractbase.entity.TemplateCategoryEntity;
import com.ejianc.business.contractbase.entity.TemplateEntity;
import com.ejianc.business.contractbase.entity.TemplateVersionEntity;
import com.ejianc.business.contractbase.enums.PartyEnum;
import com.ejianc.business.contractbase.enums.QRCodePagePositonEnum;
import com.ejianc.business.contractbase.enums.QRCodeShowTypeEnum;
import com.ejianc.business.contractbase.service.*;
import com.ejianc.business.contractbase.util.ReferTool;
import com.ejianc.business.contractbase.vo.PartyVO;
import com.ejianc.business.contractbase.vo.TemplateVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author:刘家乐
 * @version: 1
 * @document:
 */


@RestController
@RequestMapping("templateController")
public class TemplateController {
    @Autowired
    private IBillCodeApi billCodeApi;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /**此处需要根据实际修改*/
    private static final String BILL_CODE = "LJL_TEMPLATE";
    @Autowired
    private ITemplateService service;
    @Autowired
    private ITemplateCategoryService categoryService;

    @Autowired
    private ITemplateVersionService versionService;

    @Autowired
    private ITemplateSignatoryRectService templateSignatoryRectService;

    @Value("${common.env.base-host}")
    private String BaseHost;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private ITemplDetailExportSettingService templDetailExportSettingService;

    /**
     *
     * @param saveorUpdateVO 请求实体
     * @return 返回 合同模板
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TemplateVO> saveOrUpdate(@RequestBody TemplateVO saveorUpdateVO) {
        TemplateEntity entity = BeanMapper.map(saveorUpdateVO, TemplateEntity.class);
        if (entity.getId()==null){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
            if (!billCode.isSuccess()){
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
            entity.setId(IdWorker.getId());
            entity.setTenantId(InvocationInfoProxy.getTenantid());
            entity.setCode(billCode.getData());
            entity.setQrCodeFlag(false); //默认不开启二维码
        }
        //设置停启用 如果当前分类下 有启用的合同模板 就设置为停用
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("category_id",new Parameter(QueryParam.EQ,entity.getCategoryId()));
        List<TemplateEntity> templateEntityList = service.queryList(queryParam);
        //如果当前分类下无启用的 当前新增的就会被启用
        if (templateEntityList.isEmpty()){
            entity.setEnableStatus(TemplateVO.STATUS_ENABLE);
        }else {
            entity.setEnableStatus(TemplateVO.STATUS_DISABLE);
        }
        TemplateCategoryEntity templateCategoryEntity = categoryService.getById(entity.getCategoryId());
        if (null != templateCategoryEntity){
            entity.setBillTypeCode(templateCategoryEntity.getBillTypeCode());
        }else {
            //这里可能是招标类模板
            entity.setBillTypeCode(saveorUpdateVO.getBillTypeCode());
        }

        if(null != saveorUpdateVO.getBelongOrgId()) {
            CommonResponse<OrgVO> orgResp = orgApi.getOneById(saveorUpdateVO.getBelongOrgId());
            if(!orgResp.isSuccess()) {
                throw new BusinessException("保存失败，获取组织信息失败！");
            }
            entity.setBelongOrgInnerCode(orgResp.getData().getInnerCode());
        }

        //创建合同模板 版本信息
        TemplateVersionEntity version = versionService.createVersion(entity);
        entity.setCurVersionId(version.getId());
        entity.setCurVersion(1L);
        service.saveOrUpdate(entity,false);
        return service.returnCommon(entity.getId());
    }

    /**
     * 根据 合同分类 判断 合同模板的停启用状态
     * @param updateVo 需要修改模板
     * @return 返回当前被修改的合同模板
     */
    @RequestMapping(value = "/updateEnable", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TemplateVO> updateEnable(@RequestBody TemplateVO updateVo){
        TemplateEntity entity = service.selectById(updateVo.getId());
        entity.setEnableStatus(updateVo.getEnableStatus());
            UpdateWrapper<TemplateEntity> updateWrapper = new UpdateWrapper<TemplateEntity>()
                    .set("enable_status",TemplateVO.STATUS_DISABLE)
                    .eq("category_id",entity.getCategoryId());
            service.update(updateWrapper);
        service.saveOrUpdate(entity,false);
        return service.returnCommon(entity.getId());
    }

    /**
     * 合同模板复制
     * @param copyVo 复制信息
     * @return
     */
    @RequestMapping(value = "/copyTemplate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TemplateVO> copyTemplate(@RequestBody TemplateVO copyVo){
        //名称重复验证
        QueryWrapper<TemplateEntity> nameRepeatCheck = new QueryWrapper<>();
        nameRepeatCheck.eq("tenant_id", InvocationInfoProxy.getTenantid());
        nameRepeatCheck.eq("template_name", copyVo.getTemplateName());
        List<TemplateEntity> list = service.list(nameRepeatCheck);
        if(CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("保存失败，模板名称重复！");
        }

        return CommonResponse.success(service.saveCopyTempl(copyVo));
    }

    /**
     * 根据 合同分类 判断 合同模板的停启用状态
     * @param updateVo 需要修改模板
     * @return 返回当前被修改的合同模板
     */
    @RequestMapping(value = "/updateQeCodeFlag", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TemplateVO> updateQeCodeFlag(@RequestBody TemplateVO updateVo){
        TemplateEntity entity = service.selectById(updateVo.getId());

        //TODO 临时方案，后期优化项，增加对二维码生成及展示的优化实现
        if(updateVo.getQrCodeFlag()) {
            entity.setQrCodeShowType(QRCodeShowTypeEnum.首页.getTypeCode());
            entity.setQrCodePagePosition(QRCodePagePositonEnum.右上角.getPositionCode());
        } else {
            entity.setQrCodeShowType(null);
            entity.setQrCodePagePosition(null);
        }
        entity.setQrCodeFlag(updateVo.getQrCodeFlag());
        service.saveOrUpdate(entity,false);
        return service.returnCommon(entity.getId());
    }

    /**
     * 查询接口
     * @param queryParam 查询参数
     * @return 返回分页数据
     */
    @RequestMapping(value = "/queryPage", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<TemplateVO>> queryPage(@RequestBody QueryParam queryParam){
        //如果传入的键值对 含有 categoryId
        List<String> idList = new ArrayList();
        if (queryParam.getParams().containsKey("categoryId")){
            Parameter parameter = queryParam.getParams().get("categoryId");
            String value = (String) parameter.getValue();
            QueryParam query = new QueryParam();
            query.getParams().put("pid",new Parameter(QueryParam.EQ,value));
            List<TemplateCategoryEntity> categoryEntityList = categoryService.queryList(query, false);
            if (!categoryEntityList.isEmpty()){
                for (TemplateCategoryEntity t: categoryEntityList) {
                    idList.add(String.valueOf(t.getId()));
                }
            }
        }
        if (!idList.isEmpty()){
            queryParam.getParams().put("categoryId",new Parameter(QueryParam.IN,idList));
        }
        queryParam.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        IPage<TemplateEntity> page = service.queryPage(queryParam,false);
        IPage<TemplateVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), TemplateVO.class));

        //      map<templateId, exportMode>
        if(CollectionUtils.isNotEmpty(pageData.getRecords())){
            Map<String, String> modeMap = templDetailExportSettingService.getTemplExportMode(pageData.getRecords().stream().map(TemplateVO::getId).collect(Collectors.toList()));

            // 默认放置合同方对照关系
            pageData.getRecords().forEach(record -> {
                if(null != record.getQrCodeShowType()) {
                    record.setQrCodeShowTypeName(QRCodeShowTypeEnum.getByCode(record.getQrCodeShowType()).getTypeName());
                }
                if(StringUtils.isNotBlank(record.getQrCodePagePosition())) {
                    StringBuilder positionNames = new StringBuilder();
                    for (String pCode : record.getQrCodePagePosition().split(",")) {
                        positionNames.append(",").append(QRCodePagePositonEnum.getByCode(pCode).getPositionName());
                    }
                    record.setQrCodePagePositionName(positionNames.substring(1));
                }
                record.setDetailExportMode(modeMap.get(record.getId().toString()));
                record.setPartyList(this.getPartyList());
            });
        }


        return CommonResponse.success("查询列表数据成功！", pageData);
    }

    /**
     * 合同方对照关系
     *
     * @return 返回合同方对照关系
     */
    private List<PartyVO> getPartyList() {
        List<PartyVO> partyList = new ArrayList<>();
        for (PartyEnum party : PartyEnum.values()) {
            PartyVO partyVO = new PartyVO();
            partyVO.setPartyName(party.getPartyName());
            partyVO.setPartyCode(party.getPartyCode());
            partyList.add(partyVO);
        }
        return partyList;
    }

    /**
     * 删除 对应模板
     * @param vos 这里 vo中只有id
     * @return 返回删除信息
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<TemplateVO> vos) {
        Collection<TemplateEntity> templateEntities = service.listByIds(vos.stream().map(TemplateVO::getId).collect(Collectors.toList()));
        versionService.removeByIds(templateEntities.stream().map(TemplateEntity::getCurVersionId).collect(Collectors.toList()),true);
        service.removeByIds(vos.stream().map(TemplateVO::getId).collect(Collectors.toList()),true);

        // 合同模版删除，对应签章定位也删除
        List<Long> templateIds = vos.stream().map(TemplateVO::getId).collect(Collectors.toList());
        templateSignatoryRectService.batchRemove(templateIds);

        return CommonResponse.success("删除成功");
    }

    @PostMapping(value = "/saveTempleMetaData")
    @ResponseBody
    public CommonResponse<String> saveTempleMetaData(@RequestBody TemplateVO tmplVo) {
        TemplateEntity tmpl = service.selectById(tmplVo.getId());
        if(null == tmpl) {
            return CommonResponse.error("未获取到匹配的模板信息！");
        }
        tmpl.setMetadata(tmplVo.getMetadata());
        service.saveOrUpdate(tmpl,false);
        return CommonResponse.success("保存成功！");
    }

    @GetMapping(value = "/detail")
    @ResponseBody
    public CommonResponse<TemplateVO> detailById(@RequestParam Long id) {
        return CommonResponse.success(BeanMapper.map(service.selectById(id), TemplateVO.class));
    }

    /**
     * 根据分类查询分类下的模板列表
     */
    @ResponseBody
    @RequestMapping(value = "/pageListRef", method = RequestMethod.GET)
    public CommonResponse<IPage<TemplateVO>> getCategoryParent(@RequestParam("pageNumber") Integer pageNumber,
                                                               @RequestParam("pageSize")  Integer pageSize,
                                                               @RequestParam(value = "searchText", required = false) String searchText,
                                                               @RequestParam(value = "relyCondition") String relyCondition){
        QueryParam param = new QueryParam();
        param.setPageIndex(pageNumber);
        param.setPageSize(pageSize);
        param.getFuzzyFields().add("templateName");
        param.getFuzzyFields().add("code");
        if(StringUtils.isNotBlank(searchText)) {
            param.setSearchText(searchText);
        }

        //数据租户隔离
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        Map<String, Parameter> parameterMap = ReferTool.parseCondition(relyCondition);
        parameterMap.keySet().stream().forEach(key -> param.getParams().put(key, parameterMap.get(key)));

        IPage<TemplateEntity> page = service.queryPage(param);

        IPage<TemplateVO> pageDate = new Page<>();
        pageDate.setCurrent(page.getCurrent());
        pageDate.setRecords(BeanMapper.mapList(page.getRecords(), TemplateVO.class));
        pageDate.setSize(page.getSize());
        pageDate.setTotal(page.getTotal());
        pageDate.setPages(page.getPages());

        return CommonResponse.success(pageDate);
    }

    @GetMapping(value = "/preview")
    public void templatePreview(@RequestParam(value = "id") Long id, HttpServletResponse resp) throws IOException {
        TemplateEntity templ = service.selectById(id);
        if(null == templ.getFileId()) {
            throw new BusinessException("模板文件为空，无法预览！");
        }
        resp.sendRedirect(BaseHost+"ejc-file-web/onlinePreview?fileId="+templ.getFileId());
    }

}
