package com.ejianc.business.contractbase.report.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.contractbase.pool.settlepool.service.ISettlePoolService;
import com.ejianc.business.contractbase.util.DateUtil;
import com.ejianc.business.contractbase.vo.enums.ContractFeeTypeEnum;
import com.ejianc.business.contractbase.vo.report.ContractSettleReportVO;
import com.ejianc.business.procost.api.ICostSettingApi;
import com.ejianc.business.procost.vo.SettingVO;
import com.ejianc.business.profinance.api.IReceiptAndPaymentRegisterApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 合同结算报表
 *
 * @author CJ
 * @Description:
 * @date 2023/4/11 15:20
 */
@RestController
@RequestMapping(value = "/conSettleReport/")
public class ContractSettleReportController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IReceiptAndPaymentRegisterApi receiptAndPaymentRegisterApi;

    @Autowired
    private ISettlePoolService settlePoolService;

    @Autowired
    private ICostSettingApi costSettingApi;

    @PostMapping(value = "pageList")
    public CommonResponse<JSONObject> pageList(@RequestBody QueryParam queryParam) throws ParseException {
        JSONObject resp = new JSONObject();

        Map<String, Parameter> parameterMap = queryParam.getParams();
        Map<String, Object> paramMap = new HashMap<>();

        if(null == parameterMap.get("projectId") || null == parameterMap.get("startDate") || null == parameterMap.get("endDate") || null == parameterMap.get("contractType")) {
            return CommonResponse.error("请选择查询范围参数！");
        }
        String[] contractType = parameterMap.get("contractType").getValue().toString().split(",");
        paramMap.put("contractType", contractType);
        paramMap.put("conTypes", contractType.length > 1 ? "multiConType" : "oneConType");
        if(null != parameterMap.get("supplierName")) {
            paramMap.put("supplierName", parameterMap.get("supplierName").getValue());
        }
        if(null != parameterMap.get("contractFeeTypeName")) {
            paramMap.put("contractFeeTypeName", parameterMap.get("contractFeeTypeName").getValue());
        }

        //查询成本期间设置信息
        CommonResponse<SettingVO> settingResp = costSettingApi.getSettingInfo(InvocationInfoProxy.getOrgId());
        if(!settingResp.isSuccess()) {
            return CommonResponse.error("查询失败，获取成本期间设置信息失败！");
        }
        SettingVO setting = settingResp.getData();
        if(null == setting) {
            return CommonResponse.error("查询失败，当前组织未设置进行成本其间设置！");
        }

        String startDate = parameterMap.get("startDate").getValue().toString();
        String endDate = parameterMap.get("endDate").getValue().toString();

        //根据项目过滤
        paramMap.put("projectId", parameterMap.get("projectId").getValue());

        //根据成本期间设置信息拼接查询日期范围
        if("0".equals(setting.getPeriodRule())) {
            int maxDaysOfEndDate = DateUtil.getDaysOfMonth(DateFormatUtil.parseDate("yyyy-MM", endDate));
            int maxDaysOfStartDate = DateUtil.getDaysOfMonth(DateFormatUtil.parseDate("yyyy-MM", startDate));
            endDate += "-" + (maxDaysOfEndDate > Integer.valueOf(setting.getEndDay()) ? setting.getEndDay() : maxDaysOfEndDate);
            startDate += "-" + (maxDaysOfStartDate > Integer.valueOf(setting.getEndDay()) ? setting.getEndDay() : maxDaysOfStartDate);
            startDate = DateUtil.addDay(DateFormatUtil.parseDate("yyyy-MM-dd", DateUtil.getLastMonth(DateFormatUtil.parseDate("yyyy-MM-dd", startDate), "yyyy-MM-dd")),1);
            //固定日期
            paramMap.put("curStartDate", startDate);
            paramMap.put("curEndDate", endDate);

            paramMap.put("curYear", endDate.substring(0,4));
            //计算去年结算开始日期，取去年第一个月的结束日期的前一个月再加一天的日期
            paramMap.put("lastYearStartDate", DateUtil.addDay(
                    DateFormatUtil.parseDate("yyyy-MM-dd",
                            DateUtil.getLastMonth(
                                    DateFormatUtil.parseDate("yyyy-MM-dd",
                                            Integer.valueOf(endDate.substring(0,4)) - 1 + "-01-" + setting.getEndDay()),
                                    "yyyy-MM-dd")
                    ),1));
            //计算去年结算结束日期
            paramMap.put("lastYearEndDate", Integer.valueOf(endDate.substring(0,4)) - 1 + "-12-" + setting.getEndDay());
        } else {
            //自然月
            startDate += "-01";
            endDate += "-" + DateUtil.getDaysOfMonth(DateFormatUtil.parseDate("yyyy-MM", endDate));
            paramMap.put("curStartDate", startDate);
            paramMap.put("curEndDate", endDate);

            parameterMap.remove("startDate");
            parameterMap.remove("endDate");

            paramMap.put("curYear", endDate.substring(0,4));
            //计算去年结算开始日期
            paramMap.put("lastYearStartDate", Integer.valueOf(endDate.substring(0,4)) - 1 + "-01-01");
            //计算去年结算结束日期
            paramMap.put("lastYearEndDate", Integer.valueOf(endDate.substring(0,4)) - 1 + "-12-31");
        }

        if(null != parameterMap.get("contractCode")) {
            paramMap.put("contractCode", parameterMap.get("contractCode").getValue());
        }
        if(null != parameterMap.get("contractName")) {
            paramMap.put("contractName", parameterMap.get("contractName").getValue());
        }

        Map<String, Object> totalInfo = settlePoolService.count(paramMap);
        if(Integer.valueOf(totalInfo.get("total").toString()) == 0) {
            resp.put("total", 0);
            resp.put("records", new ArrayList<>());
            resp.put("current", queryParam.getPageIndex());
            resp.put("size", queryParam.getPageSize());
            resp.put("pages", 0);
            resp.put("curTotalSettleMny", 0);
            resp.put("curTotalSettleTaxMny", 0);
            resp.put("totalSettleMnyThisYear", 0);
            resp.put("totalSettleTaxMnyThisYear", 0);
            resp.put("totalStartSettleMny", 0);
            resp.put("totalStartSettleTaxMny", 0);
            return CommonResponse.success(resp);
        }

        IPage<ContractSettleReportVO> page = new Page<>();
        page.setTotal(Long.valueOf(totalInfo.get("total").toString()));
        page.setSize(queryParam.getPageSize());
        page.setCurrent(queryParam.getPageIndex());
        resp = BeanMapper.map(page, JSONObject.class);

        paramMap.put("pageSize", queryParam.getPageSize());
        paramMap.put("startLine", queryParam.getPageIndex() - 1 < 0 ? 0 : (queryParam.getPageIndex() - 1) * queryParam.getPageSize());

        logger.info("合同结算查询参数：{}", JSONObject.toJSONString(paramMap));
        List<ContractSettleReportVO> rsList = settlePoolService.pageList(paramMap);
        Map<Long, ContractSettleReportVO> conIdMap = new HashMap<>();
        Map<String, ContractSettleReportVO> conTypeMap = new HashMap<>();

        if(contractType.length > 1) {
            List<JSONObject> conTypeMapList = new ArrayList<>();
            List<Long> condIds = new ArrayList<>();
            JSONObject tmp = null;
            for(ContractSettleReportVO item : rsList) {
                //设置费用名称
                item.setContractFeeTypeName(ContractFeeTypeEnum.getByCode(item.getContractType()).getContractFeeTypeName());
                if (null != item.getContractTaxMny()){
                    item.setContractTaxMnyInW(ComputeUtil.safeDiv(item.getContractTaxMny(), BigDecimal.valueOf(10000)).setScale(2, RoundingMode.HALF_UP));
                }
                if (null != item.getIncomeContractTaxMny()){
                    item.setCurConTypeRatioInIncomeCon(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(item.getContractTaxMny(), item.getIncomeContractTaxMny()), new BigDecimal("100")));
                }else {
                    item.setCurConTypeRatioInIncomeCon(BigDecimal.ZERO);
                }
                conTypeMap.put(item.getContractType(), item);
                for(String conId : item.getContractIds().split(",")) {
                    tmp = new JSONObject();
                    tmp.put("contractId", conId);
                    tmp.put("contractType", item.getContractType());
                    condIds.add(Long.valueOf(conId));
                    conTypeMapList.add(tmp);
                }
            }
            conTypeMap.putAll(rsList.stream().collect(Collectors.toMap(item -> item.getContractType(), item -> item)));
            paramMap.put("contractIds", new ArrayList<>());
            paramMap.put("contractTypeMap", conTypeMapList);
        } else {
            conIdMap.putAll(rsList.stream().collect(Collectors.toMap(item -> item.getContractId(), item -> item)));
            paramMap.put("contractIds", conIdMap.keySet());
        }

        //查询合同付款登记信息
        logger.info("合同付款登记查询参数：{}", JSONObject.toJSONString(paramMap));
        MultiValueMap<String, String> param = new LinkedMultiValueMap<>();
        param.add("paramJson", JSONObject.toJSONString(paramMap));
        CommonResponse<List<Map<String, Object>>> payInfoResp = receiptAndPaymentRegisterApi.queryContractPayInfo(param);
        if(!payInfoResp.isSuccess()) {
            logger.error("合同付款登记信息查询失败：", JSONObject.toJSONString(payInfoResp));
            return CommonResponse.error("获取合同付款登记信息失败！");
        }

        if(null != payInfoResp.getData() && CollectionUtils.isNotEmpty(payInfoResp.getData())) {
            payInfoResp.getData().parallelStream().forEach(payInfo -> {
                ContractSettleReportVO tmp = contractType.length == 1 ? conIdMap.get(Long.valueOf(payInfo.get("contractId").toString())) : conTypeMap.get(payInfo.get("conType"));
                tmp.setCurPaidMny(null != payInfo.get("curPaidMny") ? new BigDecimal(payInfo.get("curPaidMny").toString()) : BigDecimal.ZERO);
                tmp.setTotalPaidMnyLastYear(null != payInfo.get("totalPaidMnyLastYear") ? new BigDecimal(payInfo.get("totalPaidMnyLastYear").toString()) : BigDecimal.ZERO);
                tmp.setTotalStartPaidMny(null != payInfo.get("totalStartPaidMny") ? new BigDecimal(payInfo.get("totalStartPaidMny").toString()) : BigDecimal.ZERO);
                //计算
                //合同付款比例  开工累计付款金额/合同金额*100%
                tmp.setPayScale(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(tmp.getTotalStartPaidMny(), tmp.getContractTaxMny()), new BigDecimal("100")));
                //合同欠付金额  合同金额-开工累计付款金额
                tmp.setOverPaidMny(ComputeUtil.safeSub(tmp.getContractTaxMny(), tmp.getTotalStartPaidMny()));
                //上年结算累计欠付款：取期间结束年，该合同的查询年份的上一年结算金额-付款金额
                tmp.setSettleOverPaidMnyLastYear(ComputeUtil.safeSub(tmp.getTotalSettleTaxMnyLastYear(), tmp.getTotalPaidMnyLastYear()));
            });
        }


        resp.put("records", rsList);
        resp.put("curTotalSettleMny", new BigDecimal(totalInfo.get("curTotalSettleMny").toString()));
        resp.put("curTotalSettleTaxMny", new BigDecimal(totalInfo.get("curTotalSettleTaxMny").toString()));
        resp.put("totalSettleMnyThisYear", new BigDecimal(totalInfo.get("totalSettleMnyThisYear").toString()));
        resp.put("totalSettleTaxMnyThisYear", new BigDecimal(totalInfo.get("totalSettleTaxMnyThisYear").toString()));
        resp.put("totalStartSettleMny", new BigDecimal(totalInfo.get("totalStartSettleMny").toString()));
        resp.put("totalStartSettleTaxMny", new BigDecimal(totalInfo.get("totalStartSettleTaxMny").toString()));
        if (null != rsList.get(0).getIncomeContractTaxMny()){
            resp.put("incomeContractTaxMnyInW", ComputeUtil.safeDiv(rsList.get(0).getIncomeContractTaxMny(), BigDecimal.valueOf(10000)).setScale(2, RoundingMode.HALF_UP));
        }

        return CommonResponse.success(resp);
    }

    @PostMapping(value = "export")
    public void pageList(@RequestBody QueryParam queryParam, HttpServletResponse response) throws ParseException {
        Map<String, Parameter> parameterMap = queryParam.getParams();
        Map<String, Object> paramMap = new HashMap<>();

        if(null == parameterMap.get("projectId") || null == parameterMap.get("startDate") || null == parameterMap.get("endDate") || null == parameterMap.get("contractType")) {
            throw new BusinessException("请选择查询范围参数！");
        }
        String[] contractType = parameterMap.get("contractType").getValue().toString().split(",");
        paramMap.put("contractType", contractType);
        paramMap.put("conTypes", contractType.length > 1 ? "multiConType" : "oneConType");
        if(null != parameterMap.get("supplierName")) {
            paramMap.put("supplierName", parameterMap.get("supplierName").getValue());
        }
        if(null != parameterMap.get("contractFeeTypeName")) {
            paramMap.put("contractFeeTypeName", parameterMap.get("contractFeeTypeName").getValue());
        }

        //查询成本期间设置信息
        CommonResponse<SettingVO> settingResp = costSettingApi.getSettingInfo(InvocationInfoProxy.getOrgId());
        if(!settingResp.isSuccess()) {
            throw new BusinessException("查询失败，获取成本期间设置信息失败！");
        }
        SettingVO setting = settingResp.getData();

        String startDate = parameterMap.get("startDate").getValue().toString();
        String endDate = parameterMap.get("endDate").getValue().toString();

        //根据项目过滤
        paramMap.put("projectId", parameterMap.get("projectId").getValue());

        //根据成本期间设置信息拼接查询日期范围
        if("0".equals(setting.getPeriodRule())) {
            int maxDaysOfEndDate = DateUtil.getDaysOfMonth(DateFormatUtil.parseDate("yyyy-MM", endDate));
            int maxDaysOfStartDate = DateUtil.getDaysOfMonth(DateFormatUtil.parseDate("yyyy-MM", startDate));
            endDate += "-" + (maxDaysOfEndDate > Integer.valueOf(setting.getEndDay()) ? setting.getEndDay() : maxDaysOfEndDate);
            startDate += "-" + (maxDaysOfStartDate > Integer.valueOf(setting.getEndDay()) ? setting.getEndDay() : maxDaysOfStartDate);
            startDate = DateUtil.addDay(DateFormatUtil.parseDate("yyyy-MM-dd", DateUtil.getLastMonth(DateFormatUtil.parseDate("yyyy-MM-dd", startDate), "yyyy-MM-dd")),1);
            //固定日期
            paramMap.put("curStartDate", startDate);
            paramMap.put("curEndDate", endDate);

            paramMap.put("curYear", endDate.substring(0,4));
            //计算去年结算开始日期，取去年第一个月的结束日期的前一个月再加一天的日期
            paramMap.put("lastYearStartDate", DateUtil.addDay(
                    DateFormatUtil.parseDate("yyyy-MM-dd",
                            DateUtil.getLastMonth(
                                    DateFormatUtil.parseDate("yyyy-MM-dd",
                                            Integer.valueOf(endDate.substring(0,4)) - 1 + "-01-" + setting.getEndDay()),
                                    "yyyy-MM-dd")
                    ),1));
            //计算去年结算结束日期
            paramMap.put("lastYearEndDate", Integer.valueOf(endDate.substring(0,4)) - 1 + "-12-" + setting.getEndDay());
        } else {
            //自然月
            startDate += "-01";
            endDate += "-" + DateUtil.getDaysOfMonth(DateFormatUtil.parseDate("yyyy-MM", endDate));
            paramMap.put("curStartDate", startDate);
            paramMap.put("curEndDate", endDate);

            parameterMap.remove("startDate");
            parameterMap.remove("endDate");

            paramMap.put("curYear", endDate.substring(0,4));
            //计算去年结算开始日期
            paramMap.put("lastYearStartDate", Integer.valueOf(endDate.substring(0,4)) - 1 + "-01-01");
            //计算去年结算结束日期
            paramMap.put("lastYearEndDate", Integer.valueOf(endDate.substring(0,4)) - 1 + "-12-31");
        }

        if(null != parameterMap.get("contractCode")) {
            paramMap.put("contractCode", parameterMap.get("contractCode").getValue());
        }
        if(null != parameterMap.get("contractName")) {
            paramMap.put("contractName", parameterMap.get("contractName").getValue());
        }

        String exportTitle = parameterMap.get("exportTitle").getValue().toString();

        List<ContractSettleReportVO> rsList = settlePoolService.pageList(paramMap);
        Map<Long, ContractSettleReportVO> conIdMap = new HashMap<>();
        Map<String, ContractSettleReportVO> conTypeMap = new HashMap<>();
        if(contractType.length > 1) {
            List<JSONObject> conTypeMapList = new ArrayList<>();
            List<Long> condIds = new ArrayList<>();
            JSONObject tmp = null;
            for(ContractSettleReportVO item : rsList) {
                //设置费用名称
                item.setContractFeeTypeName(ContractFeeTypeEnum.getByCode(item.getContractType()).getContractFeeTypeName());
                if (null != item.getContractTaxMny()){
                    item.setContractTaxMnyInW(ComputeUtil.safeDiv(item.getContractTaxMny(), BigDecimal.valueOf(10000)).setScale(2, RoundingMode.HALF_UP));
                }
                if (null != item.getIncomeContractTaxMny()){
                    item.setCurConTypeRatioInIncomeCon(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(item.getContractTaxMny(), item.getIncomeContractTaxMny()), new BigDecimal("100")));
                }else {
                    item.setCurConTypeRatioInIncomeCon(BigDecimal.ZERO);
                }
                conTypeMap.put(item.getContractType(), item);
                for(String conId : item.getContractIds().split(",")) {
                    tmp = new JSONObject();
                    tmp.put("contractId", conId);
                    tmp.put("contractType", item.getContractType());
                    condIds.add(Long.valueOf(conId));
                    conTypeMapList.add(tmp);
                }
            }
            conTypeMap.putAll(rsList.stream().collect(Collectors.toMap(item -> item.getContractType(), item -> item)));
            paramMap.put("contractIds", new ArrayList<>());
            paramMap.put("contractTypeMap", conTypeMapList);
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            rsList.parallelStream().forEach(item -> {
                item.setCurSettleDateStr(null != item.getCurSettleDate() ? sdf.format(item.getCurSettleDate()) : null);
                conIdMap.put(item.getContractId(), item);
            });
            //查询合同付款登记信息
            paramMap.put("contractIds", conIdMap.keySet());
        }

        MultiValueMap<String, String> param = new LinkedMultiValueMap<>();
        param.add("paramJson", JSONObject.toJSONString(paramMap));
        CommonResponse<List<Map<String, Object>>> payInfoResp = receiptAndPaymentRegisterApi.queryContractPayInfo(param);
        if(!payInfoResp.isSuccess()) {
            throw new BusinessException("获取合同付款登记信息失败！");
        }


        if(null != payInfoResp.getData() && CollectionUtils.isNotEmpty(payInfoResp.getData())) {
            payInfoResp.getData().parallelStream().forEach(payInfo -> {
                ContractSettleReportVO tmp = contractType.length == 1 ? conIdMap.get(Long.valueOf(payInfo.get("contractId").toString())) : conTypeMap.get(payInfo.get("conType"));
                tmp.setCurPaidMny(null != payInfo.get("curPaidMny") ? new BigDecimal(payInfo.get("curPaidMny").toString()) : BigDecimal.ZERO);
                tmp.setTotalPaidMnyLastYear(null != payInfo.get("totalPaidMnyLastYear") ? new BigDecimal(payInfo.get("totalPaidMnyLastYear").toString()) : BigDecimal.ZERO);
                tmp.setTotalStartPaidMny(null != payInfo.get("totalStartPaidMny") ? new BigDecimal(payInfo.get("totalStartPaidMny").toString()) : BigDecimal.ZERO);
                //计算
                //合同付款比例  开工累计付款金额/合同金额*100%
                tmp.setPayScale(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(tmp.getTotalStartPaidMny(), tmp.getContractTaxMny()), new BigDecimal("100")));
                //合同欠付金额  合同金额-开工累计付款金额
                tmp.setOverPaidMny(ComputeUtil.safeSub(tmp.getContractTaxMny(), tmp.getTotalStartPaidMny()));
                //上年结算累计欠付款：取期间结束年，该合同的查询年份的上一年结算金额-付款金额
                tmp.setSettleOverPaidMnyLastYear(ComputeUtil.safeSub(tmp.getTotalSettleTaxMnyLastYear(), tmp.getTotalPaidMnyLastYear()));
            });
        }

        Map<String, Object> records = new HashMap<>();
        records.put("records", rsList);
        ExcelExport.getInstance().exportWithTrans(contractType.length > 1 ? "ProjectSettleReport-export.xlsx" : "ContractSettleReport-export.xlsx", records, response, exportTitle);
    }

}
