package com.ejianc.business.contractbase.pool.contractpool.controller;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.contractbase.entity.TemplateCategoryEntity;
import com.ejianc.business.contractbase.mapper.TemplateCategoryMapper;
import com.ejianc.business.contractbase.pool.contractpool.bean.ContractPoolEntity;
import com.ejianc.business.contractbase.pool.contractpool.bean.SubLeaderEntity;
import com.ejianc.business.contractbase.pool.contractpool.bean.SupplierEntity;
import com.ejianc.business.contractbase.pool.contractpool.service.IContractPoolService;
import com.ejianc.business.contractbase.pool.contractpool.service.ISubLeaderService;
import com.ejianc.business.contractbase.pool.contractpool.service.ISupplierService;
import com.ejianc.business.contractbase.pool.contractpool.vo.ContractPoolNameVO;
import com.ejianc.business.contractbase.pool.contractpool.vo.ContractPoolVO;
import com.ejianc.business.contractbase.pool.contractpool.vo.SupplierVO;
import com.ejianc.business.contractbase.pool.enums.ContractPerformanceStateEnum;
import com.ejianc.business.contractbase.pool.enums.ContractPropertyEnum;
import com.ejianc.business.contractbase.pool.enums.ContractTypeEnum;
import com.ejianc.business.contractbase.util.ReferTool;
import com.ejianc.business.contractbase.vo.report.PayContractVO;
import com.ejianc.business.contractbase.vo.report.SubUnitTopVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IProSupplierApi;
import com.ejianc.foundation.share.api.IProjectPoolApi;
import com.ejianc.foundation.share.vo.dto.SubleadersDTO;
import com.ejianc.foundation.share.vo.dto.SupplierDTO;
import com.ejianc.foundation.supplier.api.IShareSubleadersApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.foundation.support.vo.ParamRegisterSetVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.ComplexParam;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ImportTemplate;
import com.ejianc.framework.skeleton.template.BaseVO;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 合同池主实体
 *
 * @author generator
 */
@RestController
@RequestMapping("contractPool")
public class ContractPoolController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private IContractPoolService service;

    @Autowired
    private TemplateCategoryMapper templateCategoryMapper;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IProSupplierApi proSupplierApi;

    @Autowired
    private ISupplierService supplierService;

    @Autowired
    private IProjectPoolApi projectPoolApi;

    @Autowired
    private IBillTypeApi billTypeApi;

    // 单据编码规则
    private static final String BILL_CODE = "CONTRACT_POOL";

    // 归档参数控制
    private final String FILLING_TYPE_STATE = "P-eg7rBO0134";

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private IShareSubleadersApi shareSubleadersApi;

    @Autowired
    private ISubLeaderService subLeaderService;

    /**
     * @param id
     * @Description queryDetail 查询详情
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    public CommonResponse<ContractPoolVO> queryDetail(Long id) {
        ContractPoolEntity entity = service.selectById(id);
        ContractPoolVO vo = BeanMapper.map(entity, ContractPoolVO.class);
        return CommonResponse.success("查询详情数据成功！", vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    public CommonResponse<String> delete(@RequestBody List<ContractPoolVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            for (ContractPoolVO vo : vos) {
                // 参数是单据类型编码字符串 根据需求是否打开下面代码
                /* CommonResponse<String> resp = billTypeApi.checkQuote("billTypeCode", vo.getId());
                if(!resp.isSuccess()){
                    return CommonResponse.error("删除失败！"+resp.getMsg());
                }*/
                //合同是否被其他单据引用
                CommonResponse<String> res = billTypeApi.checkQuote("EJCBT202204000016", vo.getId());
                logger.info("平台返回查询被引用情况" + res.isSuccess() + "----" + res.getMsg());
                if(!res.isSuccess()) {
                    return CommonResponse.error("当前合同已被下游业务引用，不能删除！");
                }
            }
        }
        service.removeByIds(vos.stream().map(ContractPoolVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    public CommonResponse<ContractPoolVO> saveOrUpdate(@RequestBody ContractPoolVO saveOrUpdateVO) {

        ContractPoolEntity entity = BeanMapper.map(saveOrUpdateVO, ContractPoolEntity.class);
        if(null == entity.getId()) {
            if(StringUtils.isBlank(entity.getContractCode())) {
                // 生成单据编码并关联实体属性
                BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
                CommonResponse<String> billCodeRes = billCodeApi.generateBillCode(billCodeParam);
                if(billCodeRes.isSuccess()) {
                    entity.setContractCode(billCodeRes.getData());
                }else{
                    throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
                }
            }
        }
        entity.setContractTaxMny(entity.getBaseTaxMny());
        entity.setContractMny(entity.getBaseTaxMny());
        entity.setContractTax(entity.getBaseTax());

        service.saveOrUpdate(entity, false);
        return CommonResponse.success("保存或修改单据成功！", BeanMapper.map(entity, ContractPoolVO.class));
    }

    /**
     * @param param
     * @Description queryList 查询列表
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
        JSONObject resp = new JSONObject();
        param.getFuzzyFields().addAll(Arrays.asList(new String[]{"contractCode","contractName","projectName"}));
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        //查询本下范围内日的合同
        Long orgId = null; String orgType= null;
        if(null == param.getParams().get("projectId")) {
            if(null != param.getParams().get("orgId") && null != param.getParams().get("orgType")) {
                orgId = Long.valueOf(param.getParams().get("orgId").getValue().toString());
                orgType = param.getParams().get("orgType").getValue().toString();

                param.getParams().remove("orgId");
                param.getParams().remove("orgType");
            } else {
                orgType = InvocationInfoProxy.getOrgType();
                orgId = InvocationInfoProxy.getOrgId();
            }
            //若当前上下文为项目部，则根据项目部Id来进行查询
            if(OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(orgType))) {
                param.getParams().put("orgId", new Parameter(QueryParam.EQ, orgId));
            } else {
                CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(orgId);
                if(!orgResp.isSuccess()) {
                    logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                    return CommonResponse.error("查询失败，获取组织信息失败！");
                }
                param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
                        orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
            }
        }
        String projectRange = null != param.getParams().get("projectRange") ? param.getParams().get("projectRange").getValue().toString() : null;
        String dateIn = null != param.getParams().get("dateIn") ? param.getParams().get("dateIn").getValue().toString() : null;
        param.getParams().remove("dateIn");
        param.getParams().remove("projectRange");
        if(StringUtils.isNotBlank(projectRange) || StringUtils.isNotBlank(dateIn)) {
            CommonResponse<List<Long>> projectIdsResp = projectPoolApi.queryProjectIdsByorgId(orgId,
                    projectRange, dateIn);

            if(!projectIdsResp.isSuccess()) {
                logger.error("根据组织Id-{}查询其下在建项目列表失败, {}", JSONObject.toJSONString(projectIdsResp));
                return CommonResponse.error("查询失败，获取组织下项目信息失败！");
            }

            if(CollectionUtils.isNotEmpty(projectIdsResp.getData())) {
                param.getParams().put("projectId", new Parameter(QueryParam.IN, projectIdsResp.getData()));
            } else {
                logger.info("根据组织Id-{}查询其下在建项目列表为空, 则返回空的数据列表！");
                resp.put("current", param.getPageIndex());
                resp.put("size", param.getPageSize());
                resp.put("pages", 0);
                resp.put("totalMny", 0);
                resp.put("total", 0);
                resp.put("records", new ArrayList<>());
                return CommonResponse.success("查询列表数据成功！", resp);
            }
        }

        param.getOrderMap().put("createTime", QueryParam.DESC);
        IPage<ContractPoolEntity> page = service.queryPage(param, false);

        resp.put("current", page.getCurrent());
        resp.put("size", page.getSize());
        resp.put("pages", page.getPages());
        resp.put("total", page.getTotal());
        resp.put("totalMny", service.calcTotalMny(param));
        resp.put("totalSupplierNumThisYear", service.queryContractionSupplierNum(param));
        resp.put("records", BeanMapper.mapList(page.getRecords(), ContractPoolVO.class));

        return CommonResponse.success("查询列表数据成功！", resp);
    }

    // 郑州一建主数据同步接口
    @RequestMapping(value = "/queryListJson", method = RequestMethod.POST)
    public CommonResponse<IPage<ContractPoolNameVO>> queryListJson(HttpServletRequest request,@RequestBody QueryParam param) {

        String syncMode = request.getParameter("syncMode");
        if("increment".equals(syncMode)) {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
            Calendar calendar = Calendar.getInstance(); //创建Calendar 的实例
            calendar.add(Calendar.DAY_OF_MONTH, -1);
            String syncTime = dateFormat.format(calendar.getTime()) + " 00:00:00";
            param.getParams().put("syncTime", new Parameter(QueryParam.EQ, syncTime));
        }

        List<ContractPoolEntity> page = service.queryMdmPage(param);
        IPage<ContractPoolNameVO> pageData = new Page<>(param.getPageIndex(), param.getPageSize(), param.getPageSize());
        if(page!=null){
            List<ContractPoolVO> records = BeanMapper.mapList(page, ContractPoolVO.class);
            pageData.setRecords(convertPoolVo(records));
        }
        logger.info("主数据同步：参数：{},结果：{}", JSONObject.toJSONString(param),JSONObject.toJSONString(page));
        return CommonResponse.success("查询合同同步数据成功！", pageData);
    }

    /**
     * 获取RPC数据
     * resp 返回值
     * isMustSuc 是否必须成功
     * errMsg 失败提示
     */
    private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
        if (isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }


    /**
     * @param param
     * @Description 导出
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        // 模糊查询字段配置
//        param.setFuzzyFields(Arrays.asList("contractCode","contractName","projectName"));
        // 租户隔离
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        //查询本下范围内日的合同
        Long orgId = InvocationInfoProxy.getOrgId();
        //若当前上下文为项目部，则根据项目部Id来进行查询
        if(OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
            param.getParams().put("orgId", new Parameter(QueryParam.EQ, orgId));
        } else {
            CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(orgId);
            if(!orgResp.isSuccess()) {
                logger.error("查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                throw new BusinessException("查询失败，获取组织信息失败！");
            }
            param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
                    orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        List<ContractPoolEntity> list = service.queryList(param);
        //todo:字段翻译等等
        List<ContractPoolVO> voList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(list)) {
            voList = BeanMapper.mapList(list, ContractPoolVO.class);
            for (ContractPoolVO vo : voList) {
                if (vo == null) {
                    continue;
                }
                // 合同属性：1-收入，0支出
                Integer contractProperty = vo.getContractProperty();
                if (contractProperty != null) {
                    if (1 == contractProperty) {
                        vo.setContractPropertyName("收入");
                    } else {
                        vo.setContractPropertyName("支出");
                    }
                }
                // 合同履约状态
                String performanceStatus = vo.getPerformanceStatus();
                if (StringUtils.isNotBlank(performanceStatus)) {
                    ContractPerformanceStateEnum performanceStateEnum = ContractPerformanceStateEnum.getByStateCode(performanceStatus);
                    vo.setPerformanceStatusName(performanceStateEnum != null ? performanceStateEnum.getStateName() : "");
                }
                // 变更状态(1-未变更 2-变更中 3-已变更)
                Integer changeStatus = vo.getChangeStatus();
                if (changeStatus != null) {
                    switch (changeStatus) {
                        case 1:
                            vo.setChangeStatusName("未变更");
                            break;
                        case 2:
                            vo.setChangeStatusName("变更中");
                            break;
                        case 3:
                            vo.setChangeStatusName("已变更");
                            break;
                        default:
                            break;
                    }
                }
                // 是否补充协议(1:是，0：否)
                Integer supplementFlag = vo.getSupplementFlag();
                if (supplementFlag != null) {
                    if (1 == supplementFlag) {
                        vo.setSupplementFlagName("是");
                    } else {
                        vo.setSupplementFlagName("否");
                    }
                }

                vo.setHandleTypeStr(Integer.valueOf("1").equals(vo.getHandleType()) ? "手动补录" : "业务推送");
            }
        }
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", voList);
        ExcelExport.getInstance().export("ContractPool-export.xlsx", beans, response);
    }

    /**
     * 合同池列表参照
     *
     * @param pageNumber
     * @param pageSize
     * @param relyCondition //     * @param searchObject
     * @param searchText
     * @return
     */
    @RequestMapping(value = "/refContractPoolDataList", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<ContractPoolVO>> getCategoryParent(@RequestParam("pageNumber") Integer pageNumber,
                                                                   @RequestParam("pageSize") Integer pageSize,
                                                                   String searchText,
                                                                   String searchObject,
                                                                   String condition,
                                                                   String relyCondition) {
        QueryParam param = new QueryParam();
        param.setPageIndex(pageNumber);
        param.setPageSize(pageSize);
        param.setSearchObject(searchObject);
        param.setSearchText(searchText);
        param.setFuzzyFields(Arrays.asList("contractCode", "contractName", "projectName", "partyaName", "partybName"));
        //数据租户隔离
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        Map<String, Parameter> parameterMap = ReferTool.parseCondition(relyCondition);
        parameterMap.keySet().forEach(key -> param.getParams().put(key, parameterMap.get(key)));

        // 根据分类ID查询所有子分类的分类ID
        if(null != param.getParams().get("categoryId")) {
            String categoryId = param.getParams().get("categoryId").getValue().toString();
            QueryWrapper<TemplateCategoryEntity> wapper = new QueryWrapper<>();
            wapper.likeRight("inner_code", categoryId);
            List<TemplateCategoryEntity> categoryEntityList = templateCategoryMapper.selectList(wapper);
            if (CollectionUtils.isNotEmpty(categoryEntityList)) {
                List<Long> categoryIdList = categoryEntityList.stream().map(TemplateCategoryEntity::getId).collect(Collectors.toList());
                param.getParams().put("categoryId", new Parameter(QueryParam.IN, categoryIdList));
            }
        }

        //归档参数设置为2时，过滤所有未归档的合同
        String filingTypeConfig = "0";
        CommonResponse<ParamRegisterSetVO> response = paramConfigApi.getByCode(FILLING_TYPE_STATE);
        if (!response.isSuccess() || response.getData() == null) {
            logger.error("根据归档参数编码-{}，查询系统参数（合同归档控制）配置信息，失败原因：{}", FILLING_TYPE_STATE, response.getMsg());
        }else {
            if (null != response.getData().getValueData()){
                filingTypeConfig = response.getData().getValueData();
            }
        }
        if (!"0".equals(filingTypeConfig)){ //过滤掉未归档合同数据
            ComplexParam c1 = new ComplexParam();
            c1.setLogic(ComplexParam.AND);

            ComplexParam c2 = new ComplexParam();
            c2.setLogic(ComplexParam.OR);
            c2.getParams().put("filingStatus", new Parameter(QueryParam.EQ, 1));

            //收入类合同无归档状态控制
            ComplexParam c3 = new ComplexParam();
            c3.setLogic(ComplexParam.OR);
            c3.getParams().put("contractProperty", new Parameter(QueryParam.EQ, 1));
            c1.getComplexParams().add(c2);
            c1.getComplexParams().add(c3);

            param.getComplexParams().add(c1);
        }

        // 合同池查询时添加根据履约状态查询的逻辑，如前端未传此参数，则默认查询履约中（枚举值为'2'）的合同
        List<String> performanceStatusList = new ArrayList<>();
        if (StringUtils.isNotBlank(condition)) {
            JSONObject conditionJson = JSONObject.parseObject(condition);
            // 根据合同类型查询
            String sourceType = conditionJson.getString("sourceType");
            if (StringUtils.isNotBlank(sourceType)) {
                param.getParams().put("sourceType", new Parameter(QueryParam.EQ, sourceType));
            }
            String sourceTypes = conditionJson.getString("sourceTypes");
            if (StringUtils.isNotBlank(sourceTypes)) {
                param.getParams().put("sourceType", new Parameter(QueryParam.IN, Arrays.asList(sourceTypes.split(","))));
            }
            //合同类别
            if(StringUtils.isNotBlank(conditionJson.getString("contractProperty"))) {
                param.getParams().put("contractProperty", new Parameter(QueryParam.EQ, conditionJson.getInteger("contractProperty")));
            }
            // 根据合同履约状态查询
            String performanceStatusParam = conditionJson.getString("performanceStatus");
            if (StringUtils.isNotBlank(performanceStatusParam)) {
                performanceStatusList = Lists.newArrayList(performanceStatusParam.split(","));
            }
            // 根据项目ID查询
            String projectId = conditionJson.getString("projectId");
            String orgId = conditionJson.getString("orgId");
            String orgType = conditionJson.getString("orgType");
            if(StringUtils.isBlank(projectId) && StringUtils.isBlank(orgId) && StringUtils.isBlank(orgType)) {
                return CommonResponse.error("查询失败，缺少查询参数projectId或组织过滤参数-orgId,orgType！");
            }
            if (StringUtils.isNotBlank(projectId)){
                logger.info("condition中projectId：{}", projectId);
                param.getParams().put("projectId", new Parameter(QueryParam.EQ, Long.valueOf(projectId)));
            } else {
                if(StringUtils.isBlank(orgId)) {
                    return CommonResponse.error("查询失败，缺少组织过滤参数-orgId！");
                }
                if(StringUtils.isBlank(orgType)) {
                    return CommonResponse.error("查询失败，缺少组织过滤参数-orgType！");
                }
                //查询本下范围内日的合同
                if(OrgVO.ORG_TYPE_DEPARTMENT.toString().equals(orgType)) {
                    //按项目部过滤
                    param.getParams().put("orgId", new Parameter(QueryParam.EQ, Long.valueOf(orgId)));
                } else {
                    //按组织过滤
                    CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(Long.valueOf(orgId));
                    if(!orgResp.isSuccess()) {
                        logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                        return CommonResponse.error("查询失败，获取组织信息失败！");
                    }
                    param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
                            orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
                }
            }

            //增加排序
            if(null != conditionJson.get("orderType")) {
                if(QueryParam.ASC.equals(conditionJson.getString("orderType"))) {
                    param.getOrderMap().put("createTime", QueryParam.ASC);
                } else {
                    param.getOrderMap().put("createTime", QueryParam.DESC);
                }
            } else {
                param.getOrderMap().put("createTime", QueryParam.DESC);
            }

        } else {
            return CommonResponse.error("查询失败，缺少查询参数projectId或组织过滤参数-orgId,orgType！");
        }
        if (CollectionUtils.isEmpty(performanceStatusList)) {
            // 默认查询履约中（枚举值为'2'）的合同
            performanceStatusList.add("2");
        }
        param.getParams().put("performanceStatus", new Parameter(QueryParam.IN, performanceStatusList));

        IPage<ContractPoolEntity> page = service.queryPage(param);
        IPage<ContractPoolVO> pageDate = new Page<>();
        pageDate.setCurrent(page.getCurrent());
        pageDate.setRecords(BeanMapper.mapList(page.getRecords(), ContractPoolVO.class));
        pageDate.setSize(page.getSize());
        pageDate.setTotal(page.getTotal());
        pageDate.setPages(page.getPages());

        return CommonResponse.success(pageDate);
    }

    /**
     * 将contractPoolVO中各种枚举类型数据转换为其对应的name
     *
     * @param voList ContractPoolNameVO类型，是contractPoolVO的子类，子类中定义了父类中各个枚举的name字段
     * @return ContractPoolNameVO类型
     */
    private List<ContractPoolNameVO> convertPoolVo(List<ContractPoolVO> voList) {
        if (CollectionUtils.isEmpty(voList)) {
            return new ArrayList<>();
        }
        List<ContractPoolNameVO> res = new ArrayList<>();

        voList.forEach(v -> {
            ContractPoolNameVO vo = BeanMapper.map(v,ContractPoolNameVO.class);
            vo.setOrgId(v.getOrgId()!=null?v.getOrgId().toString():null);
            vo.setOrgCode(v.getOrgName());
            vo.setCustomerId(v.getPartyaId());
            vo.setCustomerName(v.getPartyaName());
            vo.setSupplierId(v.getPartybId());
            vo.setSupplierCode(v.getPartybName());
            vo.setConstructionPartner(v.getJoinPartyName());
            vo.setContractPartyc(v.getPartycName());
            if(v.getSignDate()!=null){
                vo.setSignDate(DateFormatUtil.formatDate("yyyy-MM-dd",v.getSignDate()));
            }
            if(v.getContractStartDate()!=null){
                vo.setStartDate(DateFormatUtil.formatDate("yyyy-MM-dd",v.getContractStartDate()));
            }
            if(v.getContractFinishDate()!=null){
                vo.setEndDate(DateFormatUtil.formatDate("yyyy-MM-dd",v.getContractFinishDate()));
            }
            vo.setSchedule(v.getContractDaysLimit());
            String performanceStatus = v.getPerformanceStatus();
            if(performanceStatus != null) {
                switch (performanceStatus) {
                    case "1":
                        vo.setContractStatus("未签订");
                        break;
                    case "2":
                        vo.setContractStatus("履约中");
                        break;
                    case "3":
                        vo.setContractStatus("已封账");
                        break;
                    case "4":
                        vo.setContractStatus("已解除");
                        break;
                    default:
                        break;
                }
            }
            Integer isEstimation = v.getIsEstimation();
            if(isEstimation != null) {
                switch (isEstimation) {
                    case 0:
                        vo.setIsEstimation("正式合同");
                        break;
                    case 1:
                        vo.setIsEstimation("暂估合同");
                        break;

                    default:
                        break;
                }
            }
            vo.setCostChangeTotalMny(v.getTotalChangeMny());
            vo.setCostChangeTotalTaxMny(v.getTotalChangeTaxMny());
            vo.setEmployeeId(v.getEmployeeName());
            vo.setDeptId(v.getDepartmentName());
            res.add(vo);
        });
        return res;
    }

    /**
     * 各支出合同新签比例
     * @param yearFlag 0-年累，1-开累
     * @return
     */
    @GetMapping(value = "/payContractNewSignScale")
    @ResponseBody
    public CommonResponse<List<PayContractVO>> payContractNewSignScale(@RequestParam("yearFlag") Integer yearFlag
            , @RequestParam(value = "orgId", required = false) Long orgId) {
        QueryParam param = new QueryParam();
        // 数据租户隔离
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        if (orgId == null) {
            orgId = InvocationInfoProxy.getOrgId();
        }
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        if (yearFlag == 0) {
            // 本年 
            String beginYearDate = DateUtil.formatDate(DateUtil.beginOfYear(new Date()).toJdkDate());
            String endYearDate = DateUtil.formatDate(DateUtil.endOfYear(new Date()).toJdkDate());
            param.getParams().put("sign_date", new Parameter(QueryParam.BETWEEN, beginYearDate + "," + endYearDate));
        }
        // 查询支出合同
        param.getParams().put("contract_property", new Parameter(QueryParam.EQ, ContractPropertyEnum.支出合同.getPropertyCode()));
        
        List<ContractPoolEntity> list = service.queryList(param);
        
        // 总数量、金额
        BigDecimal totalContractTaxMny = list.stream().map(ContractPoolEntity::getContractTaxMny).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);

        Map<String, List<ContractPoolEntity>> sourceTypeMap = list.stream().filter(e -> e.getSourceType() != null).collect(Collectors.groupingBy(ContractPoolEntity::getSourceType));
        
        List<PayContractVO> resultList = new ArrayList<>();
        for (Map.Entry<String, List<ContractPoolEntity>> entry : sourceTypeMap.entrySet()) {
            String sourceType = entry.getKey();
            if (StringUtils.isNotBlank(sourceType)) {
                List<ContractPoolEntity> entityList = entry.getValue();

                PayContractVO vo = new PayContractVO();
                BigDecimal contractTaxMny = entityList.stream().map(ContractPoolEntity::getContractTaxMny).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);

                vo.setContractCategoryName(null != ContractTypeEnum.getByCode(sourceType) ? ContractTypeEnum.getByCode(sourceType).getTypeName() : null);
                vo.setContractCategoryCode(null != ContractTypeEnum.getByCode(sourceType) ? ContractTypeEnum.getByCode(sourceType).getTypeCode() : null);
                vo.setNum(new BigDecimal(String.valueOf(entityList.size())));
                vo.setAmount(contractTaxMny.setScale(2, RoundingMode.HALF_DOWN));
                vo.setWeight(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(contractTaxMny, totalContractTaxMny), new BigDecimal("100")).setScale(2 , RoundingMode.HALF_DOWN));
                vo.setWeightNum(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getNum(), new BigDecimal(String.valueOf(list.size()))), new BigDecimal("100")).setScale(2 , RoundingMode.HALF_DOWN));
                resultList.add(vo);
            }
        }
        
        // 
        resultList = resultList.stream().sorted(Comparator.comparing(PayContractVO::getAmount).reversed()).collect(Collectors.toList());

        return CommonResponse.success("查询成功！", resultList);
    }

    
    /**
     * 子分单位排名
     * @param topN 取前几条数据，默认前五条
     * @param mnyFlag 0-新签合同额，1-产值统计
     * @param yearFlag 0-年度，1-季度，2-月度
     * @return
     */
    @GetMapping(value = "/querySubUnitTopN")
    @ResponseBody
    public CommonResponse<List<SubUnitTopVO>> querySubUnitTopN(@RequestParam(value = "topN", defaultValue = "5") Integer topN
            , @RequestParam("mnyFlag") Integer mnyFlag, @RequestParam("yearFlag") Integer yearFlag
            , @RequestParam(value = "orgId", required = false) Long orgId) {
        
        return CommonResponse.success("查询成功！", service.querySubUnitTopN(topN, mnyFlag, yearFlag, orgId));
    }

    // 导入模板下载
    @RequestMapping(value = "/downloadExcelModel")
    @ResponseBody
    public void downloadExcelModel(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "ContractPool-import-template.xlsx", "合同池-导入模板");
    }

    // excel数据导入
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse response) {
        CommonResponse<JSONObject> commonResponse = service.excelImport(request,response);
        return commonResponse;
    }

    @ResponseBody
    @RequestMapping(value = "/querySumContract",method = RequestMethod.POST)
    public CommonResponse<IPage<SupplierVO>> querySumContract(@RequestBody QueryParam param){
        CommonResponse<IPage<SupplierVO>> iPageCommonResponse = service.querySumContract(param);
        if(!iPageCommonResponse.isSuccess()){
            return CommonResponse.error("查询失败！");
        }
        return iPageCommonResponse;
    }

    @ResponseBody
    @RequestMapping(value = "/excelExportSumContract",method = RequestMethod.POST)
    public void excelExportSumContract(@RequestBody QueryParam param, HttpServletResponse response){
        param.setPageIndex(1);
        param.setPageSize(10000);
        CommonResponse<IPage<SupplierVO>> iPageCommonResponse = service.querySumContract(param);
        List<SupplierVO> records = iPageCommonResponse.getData().getRecords();
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", records);
        ExcelExport.getInstance().export("SupplierList-export.xlsx", beans, response);
    }

    @ResponseBody
    @RequestMapping(value = "/querySumContractDetail",method = RequestMethod.GET)
    public CommonResponse<SupplierVO> querySumContractDetail(@RequestParam("suplyId") Long suplyId){
        QueryParam param = new QueryParam();
        Parameter parameter = new Parameter();
        parameter.setValue(suplyId);
        param.getParams().put("suplyId",parameter);
        CommonResponse<IPage<SupplierVO>> iPageCommonResponse = service.querySumContract(param);
        if(!iPageCommonResponse.isSuccess() || iPageCommonResponse.getData() == null || iPageCommonResponse.getData().getRecords() == null || iPageCommonResponse.getData().getRecords().size() == 0){
            return CommonResponse.error("查询失败！");
        }
        SupplierVO supplierVO = iPageCommonResponse.getData().getRecords().get(0);
        if(null == supplierVO){
            return CommonResponse.error("查询失败！");
        }
        //
        String supplyTypeName = supplierVO.getSupplyTypeName();
        // 去除首尾逗号
        if(!StringUtils.isEmpty(supplyTypeName)){
            String str1 = supplyTypeName.substring(0, 1);
            String str2 = supplyTypeName.substring(supplyTypeName.length()-1, supplyTypeName.length());
            if(str1.equals(",")){
                supplyTypeName = supplyTypeName.substring(1, supplyTypeName.length());
            }
            if(str2.equals(",")){
                supplyTypeName = supplyTypeName.substring(0, supplyTypeName.length()-1);
            }
        }
        supplierVO.setSupplyTypeName(supplyTypeName);
        return CommonResponse.success(supplierVO);
    }

    /**
     * 同步更新供应商数据
     * 即时同步接口，默认一天
     * @param
     * @return
     */
    @GetMapping(value = "/upData")
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<String> upData() {
        logger.info("开始执行数据同步");
        // 默认1天
        Integer day = 1;
        Date now = new Date();
        String upDate = com.ejianc.business.contractbase.util.DateUtil.addDay(now, -day);
        CommonResponse<List<SupplierDTO>> listCommonResponse = proSupplierApi.selectByUpDate(upDate);
        logger.info("proSupplierApi接口调用结束！");
        if(!listCommonResponse.isSuccess()){
            logger.info("查询数据失败!");
            return CommonResponse.error("查询供应商数据失败！");
        }
        if(listCommonResponse.getData() == null || listCommonResponse.getData().size() == 0){
            logger.info("数据无更新!");
            return CommonResponse.success("供应商基础数据无更新！");
        }
        List<SupplierDTO> newData = listCommonResponse.getData();
        logger.info("获取到的数据集长度为:"+newData.size());
        List<SupplierEntity> supplierEntities = BeanMapper.mapList(newData, SupplierEntity.class);
        Stream<SupplierEntity> delEntityStream = supplierEntities.stream().filter(item -> item.getDr() == 1);
        List<Long> resIds = new ArrayList<>();
        List<SupplierEntity> delEntitys = delEntityStream.collect(Collectors.toList());
        delEntitys.forEach(item->{
            resIds.add(item.getId());
        });
        logger.info("执行删除的数据条数："+resIds.size());
        if(resIds.size() > 0){
            boolean b = supplierService.removeByIds(resIds);
            logger.info("执行删除结果："+b);
        }

        Stream<SupplierEntity> entityStream = supplierEntities.stream().filter(item -> item.getDr() == 0);
        supplierEntities = entityStream.collect(Collectors.toList());
        // 解决更新数据时version不同的问题
        List<Long> suplyIds = supplierEntities.stream().map(SupplierEntity::getId).collect(Collectors.toList());
        QueryWrapper<SupplierEntity> wrapper = new QueryWrapper<>();
        wrapper.in("id",suplyIds);
        List<SupplierEntity> list = supplierService.list(wrapper);
        Map<Long,Integer> versionMap = new HashMap<>();
        list.forEach(item->{
            versionMap.put(item.getId(),item.getVersion());
        });
        supplierEntities.forEach(item->{
            item.setVersion(null == versionMap.get(item.getId()) ? item.getVersion() : versionMap.get(item.getId()));
        });
        logger.info("执行更新的数据条数："+supplierEntities.size());
        if(supplierEntities.size() > 0){
            boolean b1 = supplierService.saveOrUpdateBatch(supplierEntities,supplierEntities.size(),false);
            logger.info("执行更新结果："+b1);
        }
        logger.info("处理成功！成功更新供应商数据"+(supplierEntities.size()+delEntitys.size())+"条数据！");

        //开始同步更新分包负责人信息
        CommonResponse<List<SubleadersDTO>> subLeadersResp = shareSubleadersApi.selectByUpDateSubLeaders(upDate);
        logger.info("开始处理待同步更新分包负责人数据 ------->");
        if(!listCommonResponse.isSuccess()){
            logger.info("查询待同步更新分包负责人数据失败, {}", JSONObject.toJSONString(subLeadersResp));
            return CommonResponse.error("查询待同步更新分包负责人数据失败！");
        }
        List<SubleadersDTO> subLeaderList = subLeadersResp.getData();
        if(CollectionUtils.isEmpty(subLeaderList)){
            logger.info("分包负责人数据无更新!");
            return CommonResponse.success("分包负责人基础数据无更新！");
        }
        List<Long> delSubLeaderIds = new ArrayList<>();
        List<SubLeaderEntity> saveOrUpdateList = new ArrayList<>();
        subLeaderList.forEach(subLeader -> {
            if(BaseVO.DR_DELETED.equals(subLeader.getDr())) {
                delSubLeaderIds.add(subLeader.getId());
            } else {
                saveOrUpdateList.add(BeanMapper.map(subLeader, SubLeaderEntity.class));
            }
        });
        logger.info("本次待删除分包负责人Id-{}", JSONObject.toJSONString(delSubLeaderIds));
        if(CollectionUtils.isNotEmpty(delSubLeaderIds)){
            boolean b = subLeaderService.removeByIds(delSubLeaderIds);
            logger.info("分包负责人执行删除结果：{}", b);
        }
        logger.info("本次执行更新分包负责人数量：{}", saveOrUpdateList.size());
        if(CollectionUtils.isNotEmpty(saveOrUpdateList)) {
            List<Long> updateIds = saveOrUpdateList.stream().map(SubLeaderEntity::getId).collect(Collectors.toList());
            QueryWrapper<SubLeaderEntity> subWrapper = new QueryWrapper<>();
            subWrapper.in("id", updateIds);
            List<SubLeaderEntity> dbSubList = subLeaderService.list(subWrapper);
            Map<String, SubLeaderEntity> dbListMap = dbSubList.stream().collect(Collectors.toMap(item -> item.getId().toString(), item -> item));
            saveOrUpdateList.forEach(item -> {
                if(dbListMap.containsKey(item.getId().toString())) {
                    item.setVersion(dbListMap.get(item.getId().toString()).getVersion());
                }
            });
            boolean subUpdateRs = subLeaderService.saveOrUpdateBatch(saveOrUpdateList, saveOrUpdateList.size(), false);
            logger.info("本次分包负责人更新结果：{}", subUpdateRs);
        }
        logger.info("分包负责人同步更新成功！");

        return CommonResponse.success("处理成功！成功更新"+(supplierEntities.size()+delEntitys.size())+"条数据！");
    }

    /**
     * 同步更新供应商数据
     * @param request
     * @return
     */
    @PostMapping(value = "/updateSuplyData")
    public CommonResponse<String> updateSuplyData(HttpServletRequest request) {
        logger.info("开始执行数据同步");
        //获取定时任务配置的参数
        String upDate = getUpDate(request);
        CommonResponse<List<SupplierDTO>> listCommonResponse = proSupplierApi.selectByUpDate(upDate);
        logger.info("proSupplierApi接口调用结束！");
        if(!listCommonResponse.isSuccess()){
            logger.info("查询数据失败!");
            return CommonResponse.error("查询数据失败！");
        }
        if(listCommonResponse.getData() == null || listCommonResponse.getData().size() == 0){
            logger.info("数据无更新!");
            return CommonResponse.success("数据无更新！");
        }
        List<SupplierDTO> newData = listCommonResponse.getData();
        logger.info("获取到的数据集长度为:"+newData.size());
        List<SupplierEntity> supplierEntities = BeanMapper.mapList(newData, SupplierEntity.class);
        Stream<SupplierEntity> delEntityStream = supplierEntities.stream().filter(item -> item.getDr() == 1);
        List<Long> resIds = new ArrayList<>();
        List<SupplierEntity> delEntitys = delEntityStream.collect(Collectors.toList());
        delEntitys.forEach(item->{
            resIds.add(item.getId());
        });
        logger.info("执行删除的数据条数："+resIds.size());
        if(resIds.size() > 0){
            boolean b = supplierService.removeByIds(resIds);
            logger.info("执行删除结果："+b);
        }

        Stream<SupplierEntity> entityStream = supplierEntities.stream().filter(item -> item.getDr() == 0);
        supplierEntities = entityStream.collect(Collectors.toList());
        // 解决更新数据时version不同的问题
        List<Long> suplyIds = supplierEntities.stream().map(SupplierEntity::getId).collect(Collectors.toList());
        QueryWrapper<SupplierEntity> wrapper = new QueryWrapper<>();
        wrapper.in("id",suplyIds);
        List<SupplierEntity> list = supplierService.list(wrapper);
        Map<Long,Integer> versionMap = new HashMap<>();
        list.forEach(item->{
            versionMap.put(item.getId(),item.getVersion());
        });
        supplierEntities.forEach(item->{
            item.setVersion(null == versionMap.get(item.getId()) ? item.getVersion() : versionMap.get(item.getId()));
        });
        logger.info("执行更新的数据条数："+supplierEntities.size());
        if(supplierEntities.size() > 0){
            boolean b1 = supplierService.saveOrUpdateBatch(supplierEntities,supplierEntities.size(),false);
            logger.info("执行更新结果："+b1);
        }
        logger.info("处理成功！成功更新"+(supplierEntities.size()+delEntitys.size())+"条数据！");
        return CommonResponse.success("处理成功！成功更新"+(supplierEntities.size()+delEntitys.size())+"条数据！");
    }

    /**
     * 获取开始读取数据的时间
     * @param request
     * @return
     */
    private String getUpDate(HttpServletRequest request) {
        // 默认7天
        Integer day = 7;
        String param = request.getParameter("param");
        JSONArray json = JSONArray.parseArray(param);
        logger.info("获取到的参数JSON是：" + param + "----");
        JSONArray parameters = json.getJSONObject(0).getJSONArray("parameters");
        for (int i = 0; i < parameters.size(); i++) {
            JSONObject obj = parameters.getJSONObject(i);
            String property = (String) obj.get("property");
            if ("day".equalsIgnoreCase(property)) {
                day = obj.getInteger("value");
            }
        }
        Date now = new Date();
        String date = com.ejianc.business.contractbase.util.DateUtil.addDay(now, -day);
        return date;
    }
}
