package com.ejianc.framework.core.http;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.TimeUnit;

import okhttp3.*;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpClientUtils {
	private static final Logger logger = LoggerFactory.getLogger(HttpClientUtils.class);
	private OkHttpClient client;// okhttp对象
	private Map<String, Call> requestHandleMap = null;// 以URL为KEY存储的请求
	private CacheControl cacheControl = null;// 缓存控制器
	private HttpConfig httpConfig = new HttpConfig();// 配置信息

	private HttpClientUtils() {
		client = new OkHttpClient.Builder().connectTimeout(httpConfig.getConnectTimeout(), TimeUnit.SECONDS)
				.writeTimeout(httpConfig.getWriteTimeout(), TimeUnit.SECONDS)
				.readTimeout(httpConfig.getReadTimeout(), TimeUnit.SECONDS).build();

		requestHandleMap = Collections.synchronizedMap(new WeakHashMap<String, Call>());
		cacheControl = new CacheControl.Builder().noStore().noCache().build();// 不使用缓存
	}
	
	private static class HttpClientUtilsInstance {
		private static final HttpClientUtils instance = new HttpClientUtils();
	}
	public static HttpClientUtils getInstance() {
		return HttpClientUtilsInstance.instance;
	}

	public void get(final String url, final HttpResponseHandler callback) {
		this.get(url, null, callback, true);
	}

	public void get(final String url, Header[] header, final HttpResponseHandler callback) {
		this.get(url, header, callback, true);
	}

	/**
	 * get请求
	 * 
	 * @param url
	 *            请求url
	 * @param header
	 *            请求头
	 * @param callback
	 *            回调接口
	 * @param autoCancel
	 *            是否自动取消 true:同一时间请求一个接口多次 只保留最后一个
	 */
	public void get(final String url, Header[] header, final HttpResponseHandler callback, boolean autoCancel) {
		add(url, getBuilder(url, header, callback), autoCancel);
	}

	private Call getBuilder(String url, Header[] header, HttpResponseHandler responseCallback) {
		url = Util.getMosaicParameter(url, httpConfig.getCommonField());// 拼接公共参数
		// Log.i("ansen","访问的url"+url);
		Request.Builder builder = new Request.Builder();
		builder.url(url);
		builder.get();
		return execute(builder, header, responseCallback);
	}

	public String getSync(String url) {
		return getSync(url, null);
	}

	public String getSync(String url, Header[] header) {
		url = Util.getMosaicParameter(url, httpConfig.getCommonField());// 拼接公共参数
		Request.Builder builder = new Request.Builder();
		builder.url(url);
		builder.get();
		byte[] bytes = execute(builder, header);
		try {
			String str = new String(bytes, "utf-8");
			return str;
		} catch (Exception e) {
			printLog("getSync HTTPCaller:" + e.toString());
		}
		return null;
	}

	public void post(final String url, List<NameValuePair> params, HttpResponseHandler callback) {
		this.post(url, null, params, callback, true);
	}

	public void post(final String url, Header[] header, List<NameValuePair> params, HttpResponseHandler callback) {
		this.post(url, header, params, callback, true);
	}

	/**
	 *
	 * @param clazz
	 *            json对应类的类型
	 * @param url
	 *            请求url
	 * @param header
	 *            请求头
	 * @param params
	 *            参数
	 * @param callback
	 *            回调
	 * @param autoCancel
	 *            是否自动取消 true:同一时间请求一个接口多次 只保留最后一个
	 */
	public void post(final String url, Header[] header, final List<NameValuePair> params, final HttpResponseHandler callback, boolean autoCancel) {
		add(url, postBuilder(url, header, params, callback), autoCancel);
	}
	public void postJson(String url, Header[] header, String jsonParam, final HttpResponseHandler callback, boolean autoCancel) {
		add(url, postJsonBuilder(url, header, jsonParam, callback), autoCancel);
	}

	public String postSync(String url, List<NameValuePair> form) {
		return postSync(url, form, null);
	}

	public String postSync(String url, List<NameValuePair> form, Header[] header) {
		Request.Builder builder = getRequestBuild(url, form);
		byte[] bytes = execute(builder, header);
		try {
			String result = new String(bytes, "utf-8");
			return result;
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return null;
	}

	private Call postJsonBuilder(String url, Header[] header, String jsonParam, HttpResponseHandler callback) {
		try {
			Request.Builder builder = getRequestJsonBuild(url, jsonParam);
			return execute(builder, header, callback);
		} catch (Exception e) {
			if(callback != null) {
				callback.onFailure(-1, e.getMessage().getBytes());
			}
		}
		return null;
	}

	private Call postBuilder(String url, Header[] header, List<NameValuePair> form, HttpResponseHandler responseCallback) {
		try {
			Request.Builder builder = getRequestBuild(url, form);
			return execute(builder, header, responseCallback);
		} catch (Exception e) {
			if (responseCallback != null)
				responseCallback.onFailure(-1, e.getMessage().getBytes());
		}
		return null;
	}

	private Request.Builder getRequestJsonBuild(String url, String jsonParam) {
		RequestBody requestBody = RequestBody.create(MediaType.parse("application/json;charset=utf-8"), jsonParam);
		Request.Builder builder = new Request.Builder();
		builder.url(url);
		builder.post(requestBody);
		return builder;
	}

	private Request.Builder getRequestBuild(String url, List<NameValuePair> form) {
		if (form == null) {
			form = new ArrayList<>();
		}
		form.addAll(httpConfig.getCommonField());// 添加公共字段
		FormBody.Builder formBuilder = new FormBody.Builder();
		for (NameValuePair item : form) {
			if (StringUtils.isBlank(item.getValue())) {
				printLog("字段:" + item.getName() + "的值为null");
				continue;
			}
			formBuilder.add(item.getName(), item.getValue());
		}
		RequestBody requestBody = formBuilder.build();
		Request.Builder builder = new Request.Builder();
		builder.url(url);
		builder.post(requestBody);
		return builder;
	}

	

	// 异步执行
	private Call execute(Request.Builder builder, Header[] header, Callback responseCallback) {
		Call call = getCall(builder, header);
		if (call != null) {
			call.enqueue(responseCallback);
		}
		return call;
	}

	// 同步执行
	private byte[] execute(Request.Builder builder, Header[] header) {
		Call call = getCall(builder, header);
		byte[] body = "".getBytes();
		try {
			Response response = call.execute();
			body = response.body().bytes();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return body;
	}

	private Call getCall(Request.Builder builder, Header[] header) {
		boolean hasUa = false;
		if (header == null) {
			builder.header("Connection", "close");
			builder.header("Accept", "*/*");
		} else {
			for (Header h : header) {
				builder.header(h.getName(), h.getValue());
				if (!hasUa && h.getName().equals("User-Agent")) {
					hasUa = true;
				}
			}
		}
		if (!hasUa && StringUtils.isNotBlank(httpConfig.getUserAgent())) {
			builder.header("User-Agent", httpConfig.getUserAgent());
		}
		Request request = builder.cacheControl(cacheControl).build();
		return client.newCall(request);
	}

	private void autoCancel(String function) {
		Call call = requestHandleMap.remove(function);
		if (call != null) {
			call.cancel();
		}
	}

	/**
	 * 保存请求信息
	 * 
	 * @param url
	 *            请求url
	 * @param call
	 *            http请求call
	 * @param autoCancel
	 *            自动取消
	 */
	private void add(String url, Call call, boolean autoCancel) {
		if (StringUtils.isNotBlank(url)) {
			if (url.contains("?")) {// get请求需要去掉后面的参数
				url = url.substring(0, url.indexOf("?"));
			}
			if (autoCancel) {
				autoCancel(url);// 如果同一时间对api进行多次请求，自动取消之前的
			}
			requestHandleMap.put(url, call);
		}
	}

	private void printLog(String content) {
		if (httpConfig.isDebug()) {
			logger.debug(httpConfig.getTagName(), content);
		}
	}

	// 更新字段值
	public void updateCommonField(String key, String value) {
		httpConfig.updateCommonField(key, value);
	}

	public void removeCommonField(String key) {
		httpConfig.removeCommonField(key);
	}

	public void addCommonField(String key, String value) {
		httpConfig.addCommonField(key, value);
	}


}
