package com.ejianc.framework.core.util;

import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellValue;
import org.apache.poi.xssf.usermodel.*;
import org.springframework.web.multipart.MultipartFile;

import com.ejianc.framework.core.exception.BusinessException;

@SuppressWarnings("resource")
public class ExcelReader {
	
	/**
	 * 读取Excel方法
	 * 
	 * @param fileName
	 * @param file
	 * @return
	 * @throws IOException
	 */
	public static List<List<String>> readExcel(MultipartFile file) {
		String originalFileName = file.getOriginalFilename();
        String extName = null;
        originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
        originalFileName.replaceAll("00.", "");
    	extName = FileUtils.getFileExt(originalFileName, false);
		
		if ("xls".equals(extName)) {
			return read2003Excel(file);
		} else if ("xlsx".equals(extName)) {
			return read2007Excel(file);
		} else {
			throw new BusinessException("不支持的文件类型");
		}
	}

	/**
	 * 读取 office 2003 excel
	 * 
	 * @throws IOException
	 * @throws FileNotFoundException
	 */
	private static List<List<String>> read2003Excel(MultipartFile file) {
		List<List<String>> list = new LinkedList<>();
		HSSFWorkbook hwb;
		try {
			hwb = new HSSFWorkbook(new BufferedInputStream(file.getInputStream()));
		} catch (IOException e) {
			throw new BusinessException(e.getMessage());
		}
		int size = hwb.getNumberOfSheets();
		HSSFSheet sheet ;
		for (int index = 0; index < size; index++) {
			sheet = hwb.getSheetAt(index);
			String value = null;
			HSSFRow row = null;
			HSSFCell cell = null;
			boolean isEmpty = true;
			int rowFirstCellNum = 0;
			int rowLastCellNum = 0;
			for (int i = sheet.getFirstRowNum()+1; i < sheet.getPhysicalNumberOfRows(); i++) {
				isEmpty = true;
				row = sheet.getRow(i);
				if (row == null) {
					continue;
				}
				rowLastCellNum = sheet.getRow(0).getLastCellNum();
				List<String> linked = new LinkedList<>();
				for (int j = rowFirstCellNum; j < rowLastCellNum; j++) {
					cell = row.getCell(j);
					if (cell == null) {
						linked.add("");
						continue;
					}
					switch (cell.getCellType()) {
						//数值型
						case XSSFCell.CELL_TYPE_NUMERIC:
							// 纯数字
							cell.setCellType(Cell.CELL_TYPE_STRING);
			                value = cell.getStringCellValue();
							break;
						//字符串类型
						case XSSFCell.CELL_TYPE_STRING:
							value = cell.getStringCellValue().toString();
							break;
						// 公式类型
						case XSSFCell.CELL_TYPE_FORMULA:
							//读公式计算值
							// 使用HSSFFormulaEvaluator计算公式的值
							HSSFFormulaEvaluator evaluator = new HSSFFormulaEvaluator(hwb);
							CellValue cellValue = evaluator.evaluate(cell);
							// 获取实际值的类型
							int actualType = cellValue.getCellType();
							// 根据实际值的类型获取实际值
							if (actualType == XSSFCell.CELL_TYPE_BOOLEAN) {
								value = cellValue.getBooleanValue()+"";
							} else if (actualType == XSSFCell.CELL_TYPE_NUMERIC) {
								cell.setCellType(Cell.CELL_TYPE_STRING);
								value = cell.getStringCellValue();
							} else if (actualType == XSSFCell.CELL_TYPE_STRING) {
								value = cellValue.getStringValue();
							}
							break;
						// 布尔类型
						case XSSFCell.CELL_TYPE_BOOLEAN:
							value = " "+ cell.getBooleanCellValue();
							break;
						// 空格
						case XSSFCell.CELL_TYPE_BLANK:
							value = "";
							break;
						default:
							value = "Error";
					}
					linked.add(value);
				}

				if(list.size()>0){
					int headLength = list.get(0).size();
					int nowLinkLength = linked.size() ;
					if(nowLinkLength < headLength){
						for(int k=nowLinkLength;k<headLength;k++){
							linked.add(null);
						}
					}
				}

				//判断下linked是否为空
				if(isEmpty) {
					for(Object object:linked) {
						if(object != null && StringUtils.isNotBlank(String.valueOf(object))) {
							isEmpty = false;
							break;
						}
					}
				}
				if(!isEmpty) {
					list.add(linked);
				}else{
					break;
				}
			}
		}
		return list;
	}

	/**
	 * 读取2007数据
	 * 
	 * @param file
	 * @return
	 * @throws IOException
	 */
	private static List<List<String>> read2007Excel(MultipartFile file) {
		List<List<String>> list = new LinkedList<>();
		XSSFWorkbook xwb;
		try {
			xwb = new XSSFWorkbook(new BufferedInputStream(file.getInputStream()));
		} catch (IOException e) {
			throw new BusinessException(e.getMessage());
		}
		int size = xwb.getNumberOfSheets();
		XSSFSheet sheet ;
		for (int index = 0; index < size; index++) {
			sheet = xwb.getSheetAt(index);
			String value = null;
			XSSFRow row = null;
			XSSFCell cell = null;
			boolean isEmpty = true;
			int rowFirstCellNum = 0;
			int rowLastCellNum = 0;
			for (int i = sheet.getFirstRowNum()+1; i < sheet.getPhysicalNumberOfRows(); i++) {
				isEmpty = true;
				row = sheet.getRow(i);
				rowLastCellNum = sheet.getRow(0).getLastCellNum();
				if (row == null) {
					continue;
				}
				List<String> linked = new LinkedList<String>();
				for (int j = rowFirstCellNum; j < rowLastCellNum; j++) {
					cell = row.getCell(j);
					if (cell == null) {
						linked.add("");
						continue;
					}
					switch (cell.getCellType()) {
						//数值型
						case XSSFCell.CELL_TYPE_NUMERIC:
							cell.setCellType(Cell.CELL_TYPE_STRING);
			                value = cell.getStringCellValue();
							break;
						//字符串类型
						case XSSFCell.CELL_TYPE_STRING:
							value = cell.getStringCellValue().toString();
							break;
						// 公式类型
						case XSSFCell.CELL_TYPE_FORMULA:
							//读公式计算值
							// 使用HSSFFormulaEvaluator计算公式的值
							XSSFFormulaEvaluator evaluator = new XSSFFormulaEvaluator(xwb);
							CellValue cellValue = evaluator.evaluate(cell);
							// 获取实际值的类型
							int actualType = cellValue.getCellType();
							// 根据实际值的类型获取实际值
							if (actualType == XSSFCell.CELL_TYPE_BOOLEAN) {
								value = cellValue.getBooleanValue()+"";
							} else if (actualType == XSSFCell.CELL_TYPE_NUMERIC) {
								cell.setCellType(Cell.CELL_TYPE_STRING);
								value = cell.getStringCellValue();
							} else if (actualType == XSSFCell.CELL_TYPE_STRING) {
								value = cellValue.getStringValue();
							}
							break;
						// 布尔类型
						case XSSFCell.CELL_TYPE_BOOLEAN:
							value = " "+ cell.getBooleanCellValue();
							break;
						// 空格
						case XSSFCell.CELL_TYPE_BLANK:
							value = "";
							break;
						default:
							value = "Error";
					}
					linked.add(value);
				}

				if(list.size()>0){
					int headLength = list.get(0).size();
					int nowLinkLength = linked.size() ;
					if(nowLinkLength < headLength){
						for(int k=nowLinkLength;k<headLength;k++){
							linked.add(null);
						}
					}
				}
				//判断下linked是否为空
				if(isEmpty) {
					for(Object object:linked) {
						if(object != null && StringUtils.isNotBlank(String.valueOf(object))) {
							isEmpty = false;
							break;
						}
					}
				}
				if(!isEmpty) {
					list.add(linked);
				}else{
					break;
				}
			}
		}
		return list;
	}
}
