package com.ejianc.framework.core.util;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.NumberFormat;

/***
 * @description: BigDecimal工具类
 * @author songlx
 * @date: 2021/7/15
 */
public class ComputeUtil {

    public static int precision = 8;


    public static BigDecimal safeAdd(BigDecimal decimal1, BigDecimal... addNums) {
        for (BigDecimal temp : addNums) {
            decimal1 = safeAdd(decimal1, temp);
        }
        return decimal1;
    }

    public static BigDecimal safeAdd(BigDecimal decimal1, BigDecimal decimal2) {
        if (decimal1 == null) {
            return decimal2;
        }
        if (decimal2 == null) {
            return decimal1;
        }
        return decimal1.add(decimal2);
    }

    public static BigDecimal safeSub(BigDecimal decimal1, BigDecimal decimal2) {
        if (decimal1 == null) {
            decimal1 = new BigDecimal(0);
        }
        if (decimal2 == null) {
            decimal2 = new BigDecimal(0);
        }
        return decimal1.subtract(decimal2);
    }

    public static BigDecimal safeSub(BigDecimal decimal1, BigDecimal... subNums) {
        for (BigDecimal temp : subNums) {
            decimal1 = safeSub(decimal1, temp);
        }
        return decimal1;
    }


    public static BigDecimal safeDiv(BigDecimal decimal1, BigDecimal decimal2) {
        if (decimal1 == null || decimal2 == null || decimal2.compareTo(BigDecimal.ZERO) == 0) {
            return new BigDecimal(0);
        } else {
            return decimal1.divide(decimal2, precision, BigDecimal.ROUND_HALF_UP);
        }
    }

    public static BigDecimal safeDiv(BigDecimal decimal1, BigDecimal... divNums) {
        for (BigDecimal temp : divNums) {
            decimal1 = safeDiv(decimal1, temp);
        }
        return decimal1.setScale(precision, BigDecimal.ROUND_HALF_UP);
    }


    public static BigDecimal safeMultiply(BigDecimal decimal1, BigDecimal decimal2) {
        if (decimal1 == null) {
            decimal1 = new BigDecimal(0);
        }
        if (decimal2 == null) {
            decimal2 = new BigDecimal(0);
        }
        return decimal1.multiply(decimal2).setScale(precision, BigDecimal.ROUND_HALF_UP);
    }

    public static BigDecimal safeMultiply(BigDecimal decimal1, BigDecimal... mulNums) {
        for (BigDecimal temp : mulNums) {
            decimal1 = safeMultiply(decimal1, temp);
        }
        return decimal1.setScale(precision, BigDecimal.ROUND_HALF_UP);
    }

    public static BigDecimal scaleTwo(BigDecimal src) {
        return scale(src, 2);
    }

    /**
     * 保留小数位数
     *
     * @param x     目标小数
     * @param scale 要保留小数位数
     * @return BigDecimal 结果四舍五入
     */
    public static BigDecimal scale(BigDecimal x, int scale) {
        if (x == null) {
            return null;
        }
        return x.setScale(scale, BigDecimal.ROUND_HALF_UP);
    }



    /**
     * 保留小数位数并去除尾部0
     *
     * @param x     目标小数
     * @param scale 要保留小数位数
     * @return BigDecimal 结果四舍五入
     */
    public static String scaleStripTrailingZeros(BigDecimal x, int scale) {
        if (x == null) {
            return null;
        }
        return x.setScale(scale, BigDecimal.ROUND_HALF_UP).stripTrailingZeros().toPlainString();
    }

    /**
     * 整型转为BigDecimal
     *
     * @param x(可为null)
     * @return BigDecimal (可为null)
     */
    public static BigDecimal toBigDecimal(Integer x) {
        if (x == null) {
            return null;
        }
        return new BigDecimal(x.toString());
    }

    /**
     * 长整型转为BigDecimal
     *
     * @param x(可为null)
     * @return BigDecimal (可为null)
     */
    public static BigDecimal toBigDecimal(Long x) {
        if (x == null) {
            return null;
        }
        return new BigDecimal(x.toString());
    }

    /**
     * 双精度型转为BigDecimal
     *
     * @param x(可为null)
     * @return BigDecimal (可为null)
     */
    public static BigDecimal toBigDecimal(Double x) {
        if (x == null) {
            return null;
        }
        return new BigDecimal(x.toString());
    }

    /**
     * 单精度型转为BigDecimal
     *
     * @param x(可为null)
     * @return BigDecimal (可为null)
     */
    public static BigDecimal toBigDecimal(Float x) {
        if (x == null) {
            return null;
        }
        return new BigDecimal(x.toString());
    }

    /**
     * 字符串型转为BigDecimal
     *
     * @param x(可为null)
     * @return BigDecimal (可为null)
     */
    public static BigDecimal toBigDecimal(String x) {
        if (x == null) {
            return null;
        }
        return new BigDecimal(x);
    }

    /**
     * 对象类型转为BigDecimal
     *
     * @param x(可为null)
     * @return BigDecimal (可为null)
     */
    public static BigDecimal toBigDecimal(Object x) {
        if (x == null) {
            return null;
        }
        BigDecimal result = null;
        try {
            result = new BigDecimal(x.toString());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }


    /**
     * 比较 num1 是否大于 num2
     *
     * @param num1
     * @param num2
     * @return boolean
     */
    public static boolean isGreaterThan(BigDecimal num1, BigDecimal num2) {
        return nullToZero(num1).compareTo(nullToZero(num2)) == 1;
    }

    /**
     * 比较 num1 是否大于等于 num2
     *
     * @param num1
     * @param num2
     * @return boolean
     */
    public static boolean isGreaterOrEqual(BigDecimal num1, BigDecimal num2) {
        return isGreaterThan(num1, num2) || equals(num1, num2);
    }

    /**
     * 比较 num1 是否小于 num2
     *
     * @param num1
     * @param num2
     * @return boolean
     */
    public static boolean isLessThan(BigDecimal num1, BigDecimal num2) {
        return nullToZero(num1).compareTo(nullToZero(num2)) == -1;
    }

    /**
     * 比较 num1 是否小于等于 num2
     *
     * @param num1
     * @param num2
     * @return boolean
     */
    public static boolean isLessOrEqual(BigDecimal num1, BigDecimal num2) {
        return isLessThan(num1, num2) || equals(num1, num2);
    }

    /**
     * 比较 num1 是否等于 num2
     *
     * @param num1
     * @param num2
     * @return
     */
    public static boolean equals(BigDecimal num1, BigDecimal num2) {
        return nullToZero(num1).compareTo(nullToZero(num2)) == 0;
    }

    /**
     * 计算 num1 / num2 的百分比
     * getPercentage(10,100) = 10%
     *
     * @param num1 10
     * @param num2 100
     * @return String 10%
     */
    public static String getPercentage(BigDecimal num1, BigDecimal num2) {
        BigDecimal result = nullToZero(num1).divide(num2, 4, RoundingMode.HALF_UP);
        NumberFormat percent = NumberFormat.getPercentInstance();
        percent.setMaximumFractionDigits(2);
        return percent.format(result.doubleValue());
    }

    /**
     * 计算 num1 / num2 的百分比
     * getPercentage(10, 100, 2) = 10.00
     *
     * @param num1
     * @param num2
     * @param point 保留几位小数
     * @return BigDecimal
     */
    public static BigDecimal bigDecimalPercent(Integer num1, Integer num2, int point) {
        if (num1 == null || num2 == null) {
            return BigDecimal.ZERO;
        }
        if (num2.equals(Integer.valueOf(0))) {
            return BigDecimal.ZERO;
        }
        BigDecimal bigDecimalNum1 = new BigDecimal(num1);
        BigDecimal bigDecimalNum2 = new BigDecimal(num2);
        return bigDecimalPercent(bigDecimalNum1, bigDecimalNum2, point);
    }

    /**
     * 计算 num1 / num2 的百分比
     * getPercentage(10, 100, 2) = 10.00
     *
     * @param num1  10
     * @param num2  100
     * @param point 保留几位小数 2
     * @return BigDecimal 10.00
     */
    public static BigDecimal bigDecimalPercent(BigDecimal num1, BigDecimal num2, int point) {
        if (num1 == null || num2 == null) {
            return BigDecimal.ZERO;
        }
        if (equals(BigDecimal.ZERO, num2)) {
            return BigDecimal.ZERO;
        }
        BigDecimal percent = num1.divide(num2, point + 2, RoundingMode.HALF_UP);
        BigDecimal percent100 = percent.multiply(new BigDecimal(100)).setScale(point);
        return percent100;
    }

    /**
     * 判断num是否为空 或者 零
     *
     * @param num
     * @return String
     */
    public static Boolean isEmpty(BigDecimal num) {
        return null == num || BigDecimal.ZERO.compareTo(num) == 0;
    }

    /**
     * 判断num是否 不等于null 并且不等于零
     *
     * @param num
     * @return String
     */
    public static Boolean isNotEmpty(BigDecimal num) {
        return !isEmpty(num);
    }


    /**
     * 转换为负数
     *
     * @param num
     * @return
     */
    public static BigDecimal convertToMinusNumber(BigDecimal num) {
        if (isLessOrEqual(num, BigDecimal.ZERO)) {
            return num;
        }
        return BigDecimal.ZERO.subtract(num);
    }

    /**
     * null转换为 0
     *
     * @param num
     * @return
     */
    public static BigDecimal nullToZero(BigDecimal num) {
        return isEmpty(num) ? BigDecimal.ZERO : num;
    }

    public static void main(String[] args) {
       /* BigDecimal a = null;
        BigDecimal b = new BigDecimal("2");
        BigDecimal c = new BigDecimal("5");
        System.out.println(safeDiv(a, b, c));*/

        System.out.println(bigDecimalPercent(BigDecimal.TEN, toBigDecimal("100"), 2));
    }

}
