package com.ejianc.framework.core.util;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

public class DateFormater implements Serializable {

    private static final long serialVersionUID = 1L;

    public static final TimeZone DEFAULT_TIME_ZONE = TimeZone.getTimeZone("GMT+08:00");

    public static final String LONG_FORMAT = "yyyy-MM-dd HH:mm:ss";
    public static final String SHORT_FORMAT = "yyyy-MM-dd";
    public static final String YEAR_MONTH = "yyyy-MM";
    public static final String SHORT_TIME = "MM-dd HH:mm";

    private Date date = null;
    private TimeZone timeZone = DEFAULT_TIME_ZONE;

    private DateFormater(TimeZone tz) {
        this.timeZone = tz;
    }

    @Deprecated
    public static DateFormater getInstance() {
        return DateFormater.getInstance(System.currentTimeMillis());
    }

    @Deprecated
    public static DateFormater getInstance(Date date) {
        return DateFormater.getInstance(date.getTime());
    }

    @Deprecated
    public static DateFormater getInstance(long millis) {
        DateFormater formater = new DateFormater(DEFAULT_TIME_ZONE);
        formater.date = new Date(millis);
        return formater;
    }

    @Deprecated
    public static DateFormater getInstance(String dateStr) {
        return getInstance(dateStr, DEFAULT_TIME_ZONE);
    }

    public static DateFormater getInstance(Date date, int timeZone) {
        return DateFormater.getInstance(date.getTime(), DateFormater.getTimeZone(timeZone));
    }

    public static DateFormater getInstance(Date date, TimeZone timeZone) {
        return DateFormater.getInstance(date.getTime(), timeZone);
    }

    public static DateFormater getInstance(long millis, TimeZone timeZone) {
        DateFormater formater = new DateFormater(DEFAULT_TIME_ZONE);
        formater.date = new Date(millis);
        formater.timeZone = timeZone;
        return formater;
    }

    public static DateFormater getInstance(String dateStr, int timeZone) {
        return DateFormater.getInstance(dateStr, DateFormater.getTimeZone(timeZone));
    }

    public static DateFormater getInstance(String dateStr, String timeZone) {
        return getInstance(dateStr, TimeZone.getTimeZone(timeZone));
    }

    public static DateFormater getInstance(String dateStr, TimeZone timeZone) {
        DateFormater formater = new DateFormater(timeZone);
        try {
            if (dateStr == null || "".equals(dateStr)) {
                formater.date = new Date();
            } else if (dateStr.length() == 10 && Utils.isNum(dateStr)) {
                formater.date = new Date(Long.parseLong(dateStr) * 1000);
            } else if (dateStr.length() == 13 && Utils.isNum(dateStr)) {
                formater.date = new Date(Long.parseLong(dateStr));
            } else {
                formater.date = convert2Date(dateStr, timeZone);
            }
        } catch (Throwable t) {
            throw new RuntimeException("日期非法：" + dateStr);
        }

        return formater;
    }

    public static Date convert2Date(String dateStr, TimeZone timeZone) throws ParseException {
        SimpleDateFormat sdf = null;
        if (dateStr.length() == 10) {
            sdf = new SimpleDateFormat(SHORT_FORMAT);
        } else if (dateStr.length() == 7) {
            sdf = new SimpleDateFormat(YEAR_MONTH);
        } else {
            sdf = new SimpleDateFormat(LONG_FORMAT);
        }

        sdf.setTimeZone(timeZone);

        return sdf.parse(dateStr);
    }

    /**
     * 设置时区
     * 
     * @param timeZone
     * @return DateFormater
     */
    public DateFormater setTimeZone(TimeZone timeZone) {
        this.timeZone = timeZone;
        return this;
    }

    /**
     * 设置时区
     * 
     * @param timeZoneId 时区 ID，如：GMT+08 或 GMT+8
     * @return DateFormater
     */
    public DateFormater setTimeZone(String timeZoneId) {
        this.timeZone = TimeZone.getTimeZone(timeZoneId);
        return this;
    }

    /**
     * 设置时区
     * 
     * @param timeZone 时区，如东八区：8
     * @return DateFormater
     */
    public DateFormater setTimeZone(int timeZone) {
        if (timeZone > 12) {
            timeZone = 12;
        } else if (timeZone < -12) {
            timeZone = -12;
        }

        this.timeZone = TimeZone.getTimeZone((timeZone >= 0 ? "GMT+" : "GMT") + timeZone);

        return this;
    }

    public TimeZone getTimeZone() {
        return this.timeZone;
    }

    public Date getDate() {
        return date;
    }

    public DateFormater getOffsetSecond(int seconds) {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        calendar.add(Calendar.SECOND, seconds);

        DateFormater formater = new DateFormater(this.timeZone);
        formater.date = calendar.getTime();

        return formater;
    }

    public DateFormater getOffset(int days) {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        calendar.add(Calendar.DAY_OF_MONTH, days);

        DateFormater formater = new DateFormater(this.timeZone);
        formater.date = calendar.getTime();

        return formater;
    }

    public DateFormater getOffsetMonth(int months) {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, months);

        DateFormater formater = new DateFormater(this.timeZone);
        formater.date = calendar.getTime();

        return formater;
    }

    public int getHour24() {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        return calendar.get(Calendar.HOUR_OF_DAY);
    }

    /**
     * 返回DateFormater代表的日期，是一个月的哪一天（1~31）
     */
    public int getDay() {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        return calendar.get(Calendar.DAY_OF_MONTH);
    }

    public DateFormater setDay(int day) {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, day);
        this.date = calendar.getTime();
        return this;
    }

    public int getDaysOfWeek() {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        return calendar.get(Calendar.DAY_OF_WEEK);
    }

    /**
     * @return
     * @deprecated
     */
    public String getDayOfWeekName() {
        int day = getDaysOfWeek();
        switch (day) {
        case Calendar.MONDAY:
            return "一";
        case Calendar.TUESDAY:
            return "二";
        case Calendar.WEDNESDAY:
            return "三";
        case Calendar.THURSDAY:
            return "四";
        case Calendar.FRIDAY:
            return "五";
        case Calendar.SATURDAY:
            return "六";
        case Calendar.SUNDAY:
            return "日";
        default:
            break;
        }
        return null;
    }

    public int getDaysOfMonth() {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        return calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
    }

    public DateFormater getFirstDateOfMonth() {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, 1);

        DateFormater formater = new DateFormater(this.timeZone);
        formater.date = calendar.getTime();

        return formater;
    }

    public DateFormater getLastDateOfMonth() {
        Calendar calendar = Calendar.getInstance(this.timeZone);
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, getDaysOfMonth());

        DateFormater formater = new DateFormater(this.timeZone);
        formater.date = calendar.getTime();

        return formater;
    }

    public int getDaysBetween(DateFormater df) {
        long differSeconds = Math.abs(this.getDate().getTime() - df.getDate().getTime()) / 1000;
        return (int) Math.ceil(differSeconds * 1.0 / (3600 * 24));
    }

    public DateFormater getBegin() {
        return DateFormater.getInstance(format(SHORT_FORMAT + " 00:00:00"), this.timeZone);
    }

    public DateFormater getEnd() {
        return DateFormater.getInstance(format(SHORT_FORMAT + " 23:59:59"), this.timeZone);
    }

    public DateFormater getNoon() {
        return DateFormater.getInstance(format(SHORT_FORMAT + " 12:00:00"), this.timeZone);
    }

    public boolean before(DateFormater df) {
        return this.getDate().before(df.getDate());
    }

    public boolean after(DateFormater df) {
        return this.getDate().after(df.getDate());
    }

    public int compareTo(Date date) {
        return this.getDate().compareTo(date);
    }

    public int compareTo(DateFormater df) {
        return this.getDate().compareTo(df.getDate());
    }

    @Override
    public boolean equals(Object obj) {
        if (obj != null && (obj instanceof DateFormater || obj instanceof Date)) {
            if (obj instanceof DateFormater) {
                return this.getDate().getTime() == ((DateFormater) obj).getDate().getTime();
            } else {
                return this.getDate().getTime() == ((Date) obj).getTime();
            }
        }

        return false;
    }

    public String getLongFormat() {
        return format(LONG_FORMAT);
    }

    public String getShortFormat() {
        return format(SHORT_FORMAT);
    }

    public String getYearMonth() {
        return format(YEAR_MONTH);
    }

    public String getShortTime() {
        return format(SHORT_TIME);
    }

    public String format(String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        sdf.setTimeZone(this.timeZone);
        return sdf.format(date);
    }

    public static Date getCurrentDate() {
        return new Date(System.currentTimeMillis());
    }

    public static TimeZone getTimeZone(int tz) {
        if (tz > 12) {
            tz = 12;
        } else if (tz < -12) {
            tz = -12;
        }

        return TimeZone.getTimeZone((tz >= 0 ? "GMT+" : "GMT") + tz);
    }

    @Override
    public String toString() {
        return this.getLongFormat();
    }

    public static void main(String[] args) {
        DateFormater now = DateFormater.getInstance();
        System.out.println(now);
        System.out.println(now.getBegin());
        System.out.println(now.getBegin().getDate().getTime());
        System.out.println(now.getDate().getTime());

        now.setTimeZone(-9);
        System.out.println(now);
        System.out.println(now.getBegin());
        System.out.println(now.getBegin().getDate().getTime());
        System.out.println(now.getDate().getTime());
    }

}
