package com.ejianc.business.utils;

import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.util.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellValue;
import org.apache.poi.xssf.usermodel.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

@SuppressWarnings("resource")
public class HuaKangExcelReader {

    // 最后一行表头所在行数，因为表头存在合并行的情况，如果合并了，会导致认为后面是空行不再解析，
    // 此处在原有逻辑加上并且行数大于表头在认为是空行
    private static int TITLE_LAST_ROW_NUM = 6;

    /**
     * 读取Excel方法
     *
     * @param file
     * @return
     * @throws IOException
     */
    public static List<List<String>> readExcel(MultipartFile file, Integer projNameLine, Integer beginLine) {
        String originalFileName = file.getOriginalFilename();
        String extName = null;
        originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
        originalFileName.replaceAll("00.", "");
        extName = FileUtils.getFileExt(originalFileName, false);

        if ("xls".equals(extName)) {
            return read2003Excel(file, projNameLine, beginLine);
        } else if ("xlsx".equals(extName)) {
            return read2007Excel(file, projNameLine, beginLine);
        } else {
            throw new BusinessException("不支持的文件类型");
        }
    }

    /**
     * 读取 office 2003 excel
     *
     * @throws IOException
     * @throws FileNotFoundException
     */
    private static List<List<String>> read2003Excel(MultipartFile file, Integer projNameLine, Integer beginLine) {
        List<List<String>> list = new LinkedList<>();
        HSSFWorkbook hwb;
        try {
            hwb = new HSSFWorkbook(new BufferedInputStream(file.getInputStream()));
        } catch (IOException e) {
            throw new BusinessException(e.getMessage());
        }
        List<Integer> idexList = get2003IdexList(hwb, projNameLine, beginLine);
        int size = hwb.getNumberOfSheets();
        HSSFSheet sheet;
        //循环多个页签
        for (int index = 0; index < size; index++) {
            sheet = hwb.getSheetAt(index);
            String value = null;
            HSSFRow row = null;
            HSSFCell cell = null;
            boolean isEmpty = true;
            int firstRowNum = sheet.getFirstRowNum();
            int physicalNumberOfRows = sheet.getPhysicalNumberOfRows();
            for (int i = firstRowNum ; i < physicalNumberOfRows; i++) {
                isEmpty = true;
                row = sheet.getRow(i);
                if (row == null) {
                    continue;
                }
                if ((i != (projNameLine -1) && i < beginLine - 1)) {
                    continue;
                }
                if (i >= beginLine - 1 && (null == row.getCell(idexList.get(2)) || StringUtils.isEmpty(row.getCell(idexList.get(2)).getStringCellValue()))) {
                    continue;
                }
                List<String> linked = new LinkedList<>();
                linked.add(sheet.getSheetName());
                for (Integer j : idexList) {
                    cell = row.getCell(j);
                    if (cell == null) {
                        linked.add("");
                        continue;
                    }
                    switch (cell.getCellType()) {
                        //数值型
                        case XSSFCell.CELL_TYPE_NUMERIC:
                            // 纯数字
                            cell.setCellType(Cell.CELL_TYPE_STRING);
                            value = cell.getStringCellValue();
                            break;
                        //字符串类型
                        case XSSFCell.CELL_TYPE_STRING:
                            value = cell.getStringCellValue().toString();
                            break;
                        // 公式类型
                        case XSSFCell.CELL_TYPE_FORMULA:
                            //读公式计算值
                            // 使用HSSFFormulaEvaluator计算公式的值
                            HSSFFormulaEvaluator evaluator = new HSSFFormulaEvaluator(hwb);
                            CellValue cellValue = evaluator.evaluate(cell);
                            // 获取实际值的类型
                            int actualType = cellValue.getCellType();
                            // 根据实际值的类型获取实际值
                            if (actualType == XSSFCell.CELL_TYPE_BOOLEAN) {
                                value = cellValue.getBooleanValue() + "";
                            } else if (actualType == XSSFCell.CELL_TYPE_NUMERIC) {
                                cell.setCellType(Cell.CELL_TYPE_STRING);
                                value = cell.getStringCellValue();
                            } else if (actualType == XSSFCell.CELL_TYPE_STRING) {
                                value = cellValue.getStringValue();
                            }
                            break;
                        // 布尔类型
                        case XSSFCell.CELL_TYPE_BOOLEAN:
                            value = " " + cell.getBooleanCellValue();
                            break;
                        // 空格
                        case XSSFCell.CELL_TYPE_BLANK:
                            value = "";
                            break;
                        default:
                            value = "Error";
                    }

                    linked.add(value);
                }

                if (list.size() > 0) {
                    int headLength = list.get(0).size();
                    int nowLinkLength = linked.size();
                    if (nowLinkLength < headLength) {
                        for (int k = nowLinkLength; k < headLength; k++) {
                            linked.add(null);
                        }
                    }
                }

                //判断下linked是否为空
                if (isEmpty) {
                    for (Object object : linked) {
                        if (object != null && StringUtils.isNotBlank(String.valueOf(object))) {
                            isEmpty = false;
                            break;
                        }
                    }
                }
                if (!isEmpty) {
                    list.add(linked);
                } else {
                    if (i > sheet.getPhysicalNumberOfRows() - 3) {
                        break;
                    }
                }
            }
        }
        return list;
    }

    private static List<Integer> get2003IdexList(HSSFWorkbook hwb, Integer projNameLine, Integer beginLine) {
        // 各字段所在列号，因为存在列合并的情况，需要定位到列号
        // 按序号、项目编码、名称、特征描述、计量单位、工程量、单价、合价的顺序存放列号
        List<Integer> indexList = new LinkedList<>();
        // 序号
        int detailIndexCellNum = 0;
        // 项目编码
        int codeCellNum = 1;
        // 名称
        int nameCellNum = 2;
        // 特征描述
        int descCellNum = 3;
        // 计量单位
        int unitCellNum = 4;
        // 工程量
        int numCellNum = 5;
        // 单价
        int taxPriceCellNum = 6;
        // 合价
        int taxMnyCellNum = 7;

        // 识别字段下标
        int indexKnownNum = 0;

        List<List<String>> result = get2003HeadData(hwb, hwb.getSheetAt(0), projNameLine, beginLine);

        Iterator<List<String>> iterator = result.iterator();
        while (iterator.hasNext()) {
            List<String> datas = iterator.next();
            int _indexKnownNum = indexKnownNum;
            for (int j = 0; j < datas.size(); j++) {
                String s = datas.get(j);
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("序号")) {
                    detailIndexCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("项目编码")) {
                    codeCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("项目名称")) {
                    nameCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("项目特征")) {
                    descCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("计量单位")) {
                    unitCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("工程量") || s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("工程数量")) {
                    numCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("综合单价")) {
                    taxPriceCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").equals("合价")) {
                    taxMnyCellNum = j;
                    indexKnownNum++;
                }
            }
            if (indexKnownNum > _indexKnownNum) {
                iterator.remove();
            }
            if (indexKnownNum == 8) {
                break;
            }
        }

        if (indexKnownNum < 8 && result.size() == 0) {
            throw new BusinessException("未识别表头字段信息");
        }

        indexList.add(detailIndexCellNum);
        indexList.add(codeCellNum);
        indexList.add(nameCellNum);
        indexList.add(descCellNum);
        indexList.add(unitCellNum);
        indexList.add(numCellNum);
        indexList.add(taxPriceCellNum);
        indexList.add(taxMnyCellNum);
        return indexList;
    }

    private static List<List<String>> get2003HeadData(HSSFWorkbook hwb, HSSFSheet sheet, Integer projNameLine, Integer beginLine) {
        List<List<String>> list = new LinkedList<>();
        String value = null;
        HSSFRow row = null;
        HSSFCell cell = null;
        boolean isEmpty = true;
        int rowFirstCellNum = 0;
        int rowLastCellNum = 0;
        for (int i = sheet.getFirstRowNum() + 1; i < sheet.getPhysicalNumberOfRows(); i++) {
            isEmpty = true;
            row = sheet.getRow(i);
            if (row == null) {
                continue;
            }
            if (i < projNameLine || i >= beginLine - 1) {
                continue;
            }
            rowLastCellNum = row.getLastCellNum();
            List<String> linked = new LinkedList<>();
            for (int j = rowFirstCellNum; j < rowLastCellNum; j++) {
                cell = row.getCell(j);
                if (cell == null) {
                    linked.add("");
                    continue;
                }
                switch (cell.getCellType()) {
                    //数值型
                    case XSSFCell.CELL_TYPE_NUMERIC:
                        // 纯数字
                        cell.setCellType(Cell.CELL_TYPE_STRING);
                        value = cell.getStringCellValue();
                        break;
                    //字符串类型
                    case XSSFCell.CELL_TYPE_STRING:
                        value = cell.getStringCellValue().toString();
                        break;
                    // 公式类型
                    case XSSFCell.CELL_TYPE_FORMULA:
                        //读公式计算值
                        // 使用HSSFFormulaEvaluator计算公式的值
                        HSSFFormulaEvaluator evaluator = new HSSFFormulaEvaluator(hwb);
                        CellValue cellValue = evaluator.evaluate(cell);
                        // 获取实际值的类型
                        int actualType = cellValue.getCellType();
                        // 根据实际值的类型获取实际值
                        if (actualType == XSSFCell.CELL_TYPE_BOOLEAN) {
                            value = cellValue.getBooleanValue() + "";
                        } else if (actualType == XSSFCell.CELL_TYPE_NUMERIC) {
                            cell.setCellType(Cell.CELL_TYPE_STRING);
                            value = cell.getStringCellValue();
                        } else if (actualType == XSSFCell.CELL_TYPE_STRING) {
                            value = cellValue.getStringValue();
                        }
                        break;
                    // 布尔类型
                    case XSSFCell.CELL_TYPE_BOOLEAN:
                        value = " " + cell.getBooleanCellValue();
                        break;
                    // 空格
                    case XSSFCell.CELL_TYPE_BLANK:
                        value = "";
                        break;
                    default:
                        value = "Error";
                }

                linked.add(value);
            }

            if (list.size() > 0) {
                int headLength = list.get(0).size();
                int nowLinkLength = linked.size();
                if (nowLinkLength < headLength) {
                    for (int k = nowLinkLength; k < headLength; k++) {
                        linked.add(null);
                    }
                }
            }

            //判断下linked是否为空
            if (isEmpty) {
                for (Object object : linked) {
                    if (object != null && StringUtils.isNotBlank(String.valueOf(object))) {
                        isEmpty = false;
                        break;
                    }
                }
            }
            if (!isEmpty) {
                list.add(linked);
            } else {
                if (i >= beginLine) {
                    break;
                }
            }
        }
        return list;
    }

    /**
     * 读取2007数据
     *
     * @param file
     * @param projNameLine
     * @param beginLine
     * @return
     * @throws IOException
     */
    private static List<List<String>> read2007Excel(MultipartFile file, Integer projNameLine, Integer beginLine) {
        List<List<String>> list = new LinkedList<>();
        XSSFWorkbook xwb;
        try {
            xwb = new XSSFWorkbook(new BufferedInputStream(file.getInputStream()));
        } catch (IOException e) {
            throw new BusinessException(e.getMessage());
        }
        List<Integer> idexList = get2007IdexList(xwb, projNameLine, beginLine);
        int size = xwb.getNumberOfSheets();
        XSSFSheet sheet;
        for (int index = 0; index < size; index++) {
            sheet = xwb.getSheetAt(index);
            String value = null;
            XSSFRow row = null;
            XSSFCell cell = null;
            boolean isEmpty = true;
            int firstRowNum = sheet.getFirstRowNum();
            int physicalNumberOfRows = sheet.getPhysicalNumberOfRows();
            for (int i = firstRowNum; i < physicalNumberOfRows; i++) {
                isEmpty = true;
                row = sheet.getRow(i);
                if (row == null) {
                    continue;
                }
                if ((i != (projNameLine -1) && i < beginLine - 1)) {
                    continue;
                }
                if (i >= beginLine - 1 && (null == row.getCell(idexList.get(2)) || StringUtils.isEmpty(row.getCell(idexList.get(2)).getStringCellValue()))) {
                    continue;
                }

                List<String> linked = new LinkedList<String>();
                linked.add(sheet.getSheetName());
                for (Integer j : idexList) {
                    cell = row.getCell(j);
                    if (cell == null) {
                        linked.add("");
                        continue;
                    }
                    switch (cell.getCellType()) {
                        //数值型
                        case XSSFCell.CELL_TYPE_NUMERIC:
                            cell.setCellType(Cell.CELL_TYPE_STRING);
                            value = cell.getStringCellValue();
                            break;
                        //字符串类型
                        case XSSFCell.CELL_TYPE_STRING:
                            value = cell.getStringCellValue().toString();
                            break;
                        // 公式类型
                        case XSSFCell.CELL_TYPE_FORMULA:
                            //读公式计算值
                            // 使用HSSFFormulaEvaluator计算公式的值
                            XSSFFormulaEvaluator evaluator = new XSSFFormulaEvaluator(xwb);
                            CellValue cellValue = evaluator.evaluate(cell);
                            // 获取实际值的类型
                            int actualType = cellValue.getCellType();
                            // 根据实际值的类型获取实际值
                            if (actualType == XSSFCell.CELL_TYPE_BOOLEAN) {
                                value = cellValue.getBooleanValue() + "";
                            } else if (actualType == XSSFCell.CELL_TYPE_NUMERIC) {
                                cell.setCellType(Cell.CELL_TYPE_STRING);
                                value = cell.getStringCellValue();
                            } else if (actualType == XSSFCell.CELL_TYPE_STRING) {
                                value = cellValue.getStringValue();
                            }
                            break;
                        // 布尔类型
                        case XSSFCell.CELL_TYPE_BOOLEAN:
                            value = " " + cell.getBooleanCellValue();
                            break;
                        // 空格
                        case XSSFCell.CELL_TYPE_BLANK:
                            value = "";
                            break;
                        default:
                            value = "Error";
                    }
                    linked.add(value);
                }

                if (list.size() > 0) {
                    int headLength = list.get(0).size();
                    int nowLinkLength = linked.size();
                    if (nowLinkLength < headLength) {
                        for (int k = nowLinkLength; k < headLength; k++) {
                            linked.add(null);
                        }
                    }
                }
                //判断下linked是否为空
                if (isEmpty) {
                    for (Object object : linked) {
                        if (object != null && StringUtils.isNotBlank(String.valueOf(object))) {
                            isEmpty = false;
                            break;
                        }
                    }
                }
                if (!isEmpty) {
                    list.add(linked);
                } else {
                    if (i > sheet.getPhysicalNumberOfRows() - 3) {
                        break;
                    }
                }
            }
        }
        return list;
    }


    private static List<Integer> get2007IdexList(XSSFWorkbook xwb, Integer projNameLine, Integer beginLine) {
        // 各字段所在列号，因为存在列合并的情况，需要定位到列号
        // 按序号、项目编码、名称、特征描述、计量单位、工程量、单价、合价的顺序存放列号
        List<Integer> indexList = new LinkedList<>();
        // 序号
        int detailIndexCellNum = 0;
        // 项目编码
        int codeCellNum = 1;
        // 名称
        int nameCellNum = 2;
        // 特征描述
        int descCellNum = 3;
        // 计量单位
        int unitCellNum = 4;
        // 工程量
        int numCellNum = 5;
        // 单价
        int taxPriceCellNum = 6;
        // 合价
        int taxMnyCellNum = 7;

        // 识别字段下标
        int indexKnownNum = 0;

        List<List<String>> result = get2007HeadData(xwb, xwb.getSheetAt(0), projNameLine, beginLine);

        Iterator<List<String>> iterator = result.iterator();
        while (iterator.hasNext()) {
            List<String> datas = iterator.next();
            int _indexKnownNum = indexKnownNum;
            for (int j = 0; j < datas.size(); j++) {
                String s = datas.get(j);
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("序号")) {
                    detailIndexCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("项目编码")) {
                    codeCellNum = j;
                    indexKnownNum++;
                }
                if (s.trim().replaceAll("\\s*|\\r|\\n|\\t", "").contains("项目名称")) {
                    nameCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("项目特征")) {
                    descCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("计量单位")) {
                    unitCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("工程量") || s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("工程数量")) {
                    numCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").contains("综合单价")) {
                    taxPriceCellNum = j;
                    indexKnownNum++;
                }
                if (s.replaceAll("\\s*|\\r|\\n|\\t", "").equals("合价")) {
                    taxMnyCellNum = j;
                    indexKnownNum++;
                }
            }
            if (indexKnownNum > _indexKnownNum) {
                iterator.remove();
            }
            if (indexKnownNum == 8) {
                break;
            }
        }

        if (indexKnownNum < 8 && result.size() == 0) {
            throw new BusinessException("未识别表头字段信息");
        }

        indexList.add(detailIndexCellNum);
        indexList.add(codeCellNum);
        indexList.add(nameCellNum);
        indexList.add(descCellNum);
        indexList.add(unitCellNum);
        indexList.add(numCellNum);
        indexList.add(taxPriceCellNum);
        indexList.add(taxMnyCellNum);
        return indexList;
    }

    private static List<List<String>> get2007HeadData(XSSFWorkbook xwb, XSSFSheet sheet, Integer projNameLine, Integer beginLine) {
        List<List<String>> list = new LinkedList<>();
        String value = null;
        XSSFRow row = null;
        XSSFCell cell = null;
        boolean isEmpty = true;
        int rowFirstCellNum = 0;
        int rowLastCellNum = 0;
        for (int i = sheet.getFirstRowNum() + 1; i < sheet.getPhysicalNumberOfRows(); i++) {
            isEmpty = true;
            row = sheet.getRow(i);
            if (row == null) {
                continue;
            }
            if (i < projNameLine || i >= beginLine - 1) {
                continue;
            }
            rowLastCellNum = row.getLastCellNum();
            List<String> linked = new LinkedList<>();
            for (int j = rowFirstCellNum; j < rowLastCellNum; j++) {
                cell = row.getCell(j);
                if (cell == null) {
                    linked.add("");
                    continue;
                }
                switch (cell.getCellType()) {
                    //数值型
                    case XSSFCell.CELL_TYPE_NUMERIC:
                        // 纯数字
                        cell.setCellType(Cell.CELL_TYPE_STRING);
                        value = cell.getStringCellValue();
                        break;
                    //字符串类型
                    case XSSFCell.CELL_TYPE_STRING:
                        value = cell.getStringCellValue().toString();
                        break;
                    // 公式类型
                    case XSSFCell.CELL_TYPE_FORMULA:
                        //读公式计算值
                        // 使用HSSFFormulaEvaluator计算公式的值
                        XSSFFormulaEvaluator evaluator = new XSSFFormulaEvaluator(xwb);
                        CellValue cellValue = evaluator.evaluate(cell);
                        // 获取实际值的类型
                        int actualType = cellValue.getCellType();
                        // 根据实际值的类型获取实际值
                        if (actualType == XSSFCell.CELL_TYPE_BOOLEAN) {
                            value = cellValue.getBooleanValue() + "";
                        } else if (actualType == XSSFCell.CELL_TYPE_NUMERIC) {
                            cell.setCellType(Cell.CELL_TYPE_STRING);
                            value = cell.getStringCellValue();
                        } else if (actualType == XSSFCell.CELL_TYPE_STRING) {
                            value = cellValue.getStringValue();
                        }
                        break;
                    // 布尔类型
                    case XSSFCell.CELL_TYPE_BOOLEAN:
                        value = " " + cell.getBooleanCellValue();
                        break;
                    // 空格
                    case XSSFCell.CELL_TYPE_BLANK:
                        value = "";
                        break;
                    default:
                        value = "Error";
                }

                linked.add(value);
            }

            if (list.size() > 0) {
                int headLength = list.get(0).size();
                int nowLinkLength = linked.size();
                if (nowLinkLength < headLength) {
                    for (int k = nowLinkLength; k < headLength; k++) {
                        linked.add(null);
                    }
                }
            }

            //判断下linked是否为空
            if (isEmpty) {
                for (Object object : linked) {
                    if (object != null && StringUtils.isNotBlank(String.valueOf(object))) {
                        isEmpty = false;
                        break;
                    }
                }
            }
            if (!isEmpty) {
                list.add(linked);
            } else {
                if (i >= beginLine) {
                    break;
                }
            }
        }
        return list;
    }

}
