package com.ejianc.business.dxcheck.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.dxcheck.entity.RecordEntity;
import com.ejianc.business.dxcheck.entity.UserEntity;
import com.ejianc.business.dxcheck.model.res.RecordRes;
import com.ejianc.business.dxcheck.model.vo.*;
import com.ejianc.business.dxcheck.service.RecordServer;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.validation.Valid;
import java.util.List;

/**
 * @Author: LCL
 * @Date: 2024/5/29 上午9:43
 * @Description: 考核记录
 */
@Controller
@RequestMapping("record")
@RequiredArgsConstructor
public class RecordController {
    private final RecordServer server;

    /**
     * 新增
     *
     * @param vo RecordAddVo
     * @return RecordAddRes
     */
    @RequestMapping(value = "add", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<RecordRes> add(@RequestBody @Valid RecordAddEditVo vo) {
        return CommonResponse.success("考核记录-新增成功", server.add(vo));
    }

    /**
     * 新增提交
     */
    @RequestMapping(value = "addSubmit", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> addSubmit(@RequestParam(value = "id") Long id) {
        server.addSubmit(id);
        return CommonResponse.success("考核记录-提交成功");
    }

    /**
     * 批量新增
     *
     * @param vo RecordAddVo
     * @return RecordAddRes
     */
    @RequestMapping(value = "batchAdd", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<List<RecordRes>> batchAdd(@RequestBody @Valid RecordBatchAddVo vo) {
        return CommonResponse.success("考核记录-批量新增成功", server.batchAdd(vo));
    }

    /**
     * 删除
     *
     * @param vo RecordDelVo
     * @return 结果
     */
    @RequestMapping(value = "del", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> del(@RequestBody RecordDelVo vo) {
        server.del(vo);
        return CommonResponse.success("考核记录-删除成功");
    }

    /**
     * 详情
     *
     * @param id id
     * @return RecordAddRes
     */
    @RequestMapping(value = "detail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<RecordRes> detail(@RequestParam(value = "id") Long id) {
        return CommonResponse.success("考核记录-详情", server.detail(id));
    }

    /**
     * 列表
     *
     * @param param param
     * @return IPage<RecordEntity>
     */
    @RequestMapping(value = "list", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<RecordEntity>> list(@RequestBody QueryParam param) {
        return CommonResponse.success("考核指标-列表", server.list(param));
    }

    /**
     * 修改
     *
     * @param vo RecordAddEditVo
     * @return RecordAddRes
     */
    @RequestMapping(value = "edit", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<RecordRes> edit(@RequestBody @Valid RecordAddEditVo vo) {
        return CommonResponse.success("考核记录-修改成功", server.edit(vo));
    }

    /**
     * 评分
     *
     * @param vo RecordScoreVo
     * @return 结果
     */
    @RequestMapping(value = "score", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> score(@RequestBody @Valid RecordScoreVo vo) {
        server.score(vo);
        return CommonResponse.success("考核记录-评分成功");
    }

    /**
     * 考核人员确认
     */
    @RequestMapping(value = "check", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<List<UserVO>> check(@RequestParam(value = "org_Id") String OrgId, @RequestParam(value = "dept_id") String DeptId) {
        List<UserVO> vos = server.checkAssessor(OrgId, DeptId);
        if (vos == null) {
            return CommonResponse.error("考核人员未配置");
        }
        return CommonResponse.success("考核记录-考核人员确认成功", vos);
    }

    /**
     * 考核单位列表
     */
    @RequestMapping(value = "assessmentUnitList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<UserVO>> assessmentUnitList(@RequestBody QueryParam param) {
        IPage<UserEntity> page = server.assessmentUnitList(param);
        IPage<UserVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), UserVO.class));
        return CommonResponse.success("考核记录-考核单位列表", pageData);
    }

    /**
     * 考核部门列表
     */
    @RequestMapping(value = "assessmentDeptList", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<UserVO>> assessmentDeptList(@RequestParam String unitName) {
        List<UserVO> list = server.assessmentDeptList(unitName);
        if (list == null) {
            return CommonResponse.error("仅需要考核中国能源建设集团浙江火电建设有限公司本部");
        }
        return CommonResponse.success("考核记录-考核部门列表", list);
    }

}
