package com.ejianc.business.dxcheck.service.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.dxcheck.dao.RecordDao;
import com.ejianc.business.dxcheck.dao.RecordSubDao;
import com.ejianc.business.dxcheck.dao.UserDao;
import com.ejianc.business.dxcheck.entity.RecordEntity;
import com.ejianc.business.dxcheck.entity.RecordSubEntity;
import com.ejianc.business.dxcheck.entity.UserEntity;
import com.ejianc.business.dxcheck.model.res.RecordRes;
import com.ejianc.business.dxcheck.model.vo.RecordAddEditVo;
import com.ejianc.business.dxcheck.model.vo.RecordBatchAddVo;
import com.ejianc.business.dxcheck.model.vo.RecordDelVo;
import com.ejianc.business.dxcheck.model.vo.RecordScoreVo;
import com.ejianc.business.dxcheck.model.vo.UserVO;
import com.ejianc.business.dxcheck.service.RecordServer;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/5/29 上午9:46
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class RecordServerImpl implements RecordServer {
    private final RecordDao recordDao;
    private final RecordSubDao recordSubDao;
    private final SessionManager sessionManager;
    private final UserDao userDao;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public RecordRes add(RecordAddEditVo vo) {
        UserContext userContext = sessionManager.getUserContext();
        /*
          主信息
         */
        RecordEntity recordEntity = new RecordEntity();
        BeanUtils.copyProperties(vo, recordEntity);
        recordEntity.setState(NumberUtils.INTEGER_ZERO);
        recordEntity.setOrgId(userContext.getOrgId());
        recordEntity.setOrgCode(userContext.getOrgCode());
        recordEntity.setOrgName(userContext.getOrgName());
        recordEntity.setParentOrgId(userContext.getDeptId());
        recordEntity.setParentOrgCode(userContext.getDeptCode());
        recordEntity.setParentOrgName(userContext.getDeptName());
        recordDao.save(recordEntity);
        /*
          指标信息
         */
        List<RecordSubEntity> saveRecordSubEntityList = Lists.newArrayList();
        List<RecordAddEditVo.AddEditRecordSubInfo> recordSubEntities = vo.getRecordSubEntities();
        recordSubEntities.forEach(e -> {
            BeanUtils.copyProperties(recordEntity, e);

            RecordSubEntity recordSubEntity = new RecordSubEntity();
            BeanUtils.copyProperties(e, recordSubEntity);
            recordSubEntity.setId(null);
            recordSubEntity.setRecordId(recordEntity.getId());
            recordSubEntity.setState(NumberUtils.INTEGER_ZERO);
            saveRecordSubEntityList.add(recordSubEntity);
        });
        recordSubDao.saveBatch(saveRecordSubEntityList);

        return buildRes(recordEntity, saveRecordSubEntityList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addSubmit(Long id) {
        try {
            recordDao.lambdaUpdate().eq(BaseEntity::getId, id).set(RecordEntity::getState, NumberUtils.INTEGER_ONE).update();
            recordSubDao.lambdaUpdate().eq(RecordSubEntity::getRecordId, id).set(RecordSubEntity::getState, NumberUtils.INTEGER_ONE).update();
        } catch (Exception e) {
            throw new BusinessException("考核记录提交失败，原因:" + e.getMessage());
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<RecordRes> batchAdd(RecordBatchAddVo vo) {
        UserContext userContext = sessionManager.getUserContext();
        String batchAssessmentCode = RandomUtil.randomString(20);
        List<RecordBatchAddVo.BathAddRecordInfo> recordInfos = vo.getRecordInfos();

        String assessmentContent = vo.getAssessmentContent();
        String assessmentEventType = vo.getAssessmentEventType();

        List<RecordRes> resList = Lists.newArrayList();
        List<RecordEntity> recordEntityList = Lists.newArrayList();
        List<RecordSubEntity> saveRecordSubEntityList = Lists.newArrayList();
        for (RecordBatchAddVo.BathAddRecordInfo recordInfo : recordInfos) {
            /*
              主信息
             */
            RecordEntity recordEntity = new RecordEntity();
            BeanUtils.copyProperties(recordInfo, recordEntity);
            recordEntity.setState(NumberUtils.INTEGER_ZERO);
            recordEntity.setOrgId(userContext.getOrgId());
            recordEntity.setOrgCode(userContext.getOrgCode());
            recordEntity.setOrgName(userContext.getOrgName());
            recordEntity.setParentOrgId(userContext.getDeptId());
            recordEntity.setParentOrgCode(userContext.getDeptCode());
            recordEntity.setParentOrgName(userContext.getDeptName());
            if ("考评组发起评分".equals(assessmentEventType) || "专业组直接评分".equals(assessmentEventType)) {
                recordEntity.setBatchAssessmentCode(batchAssessmentCode);
            }

            recordEntity.setAssessmentContent(assessmentContent);
            recordEntity.setAssessmentEventType(assessmentEventType);
            /*
              指标信息
             */
            List<RecordBatchAddVo.BathAddRecordSubInfo> recordSubEntities = vo.getRecordSubEntities();
            boolean isSaveRecord = Boolean.TRUE;
            List<RecordSubEntity> recordSubEntityList = Lists.newArrayList();
            for (RecordBatchAddVo.BathAddRecordSubInfo recordSubEntity : recordSubEntities) {
                BeanUtils.copyProperties(recordEntity, recordSubEntity);
                /*
                  1:本部 2.直管项目 3.分子公司
                 */
                Integer orgType = recordSubEntity.getOrgType();
                if (orgType == 1) {
                    String headquarterDepartment = recordSubEntity.getHeadquarterDepartment();
                    if (normDispose(recordSubEntityList, assessmentContent, assessmentEventType
                            , recordSubEntity, headquarterDepartment)) {
                        continue;
                    }
                }
                if (orgType == 2) {
                    String straightProject = recordSubEntity.getStraightProject();
                    if (normDispose(recordSubEntityList, assessmentContent, assessmentEventType,
                            recordSubEntity, straightProject)) {
                        continue;
                    }
                }
                if (orgType == 3) {
                    String subsidiaryCompany = recordSubEntity.getSubsidiaryCompany();
                    boolean dispose = normDispose(recordSubEntityList, assessmentContent, assessmentEventType
                            , recordSubEntity, subsidiaryCompany);
                    if (!dispose) {
                        isSaveRecord = Boolean.FALSE;
                    }
                }
            }
            /**
             * 判断是否生成主信息和子信息
             */
            if (isSaveRecord) {
                recordDao.save(recordEntity);
                recordEntityList.add(recordEntity);
                Long recordEntityId = recordEntity.getId();
                recordSubEntityList.forEach(e -> e.setRecordId(recordEntityId));
                saveRecordSubEntityList.addAll(recordSubEntityList);
            }
        }

        recordSubDao.saveBatch(saveRecordSubEntityList);

        recordEntityList.forEach(recordEntity -> resList.add(buildRes(recordEntity, saveRecordSubEntityList)));
        return resList;
    }

    /**
     * 指标处理
     * 逻辑:都适用(√)||部分适用(单位名字相同)=====>新增指标
     *
     * @param recordSubEntityList 需要保存的指标集合
     * @param assessmentContent   考核内容
     * @param assessmentEventType 考评事件类型
     * @param recordSubEntity     批量新增的子表信息
     * @param unitInfo            对应类型的指标
     * @return 新的需要保存的指标集合 saveRecordSubEntityList
     */
    private boolean normDispose(List<RecordSubEntity> recordSubEntityList,
                                String assessmentContent, String assessmentEventType,
                                RecordBatchAddVo.BathAddRecordSubInfo recordSubEntity, String unitInfo) {
        //都适用(√)||部分适用(单位名字相同)=====>新增指标
        if ("√".equals(unitInfo) || unitInfo.equals(recordSubEntity.getAssessmentUnitName())) {
            RecordSubEntity saveRecordSubEntity = new RecordSubEntity();
            BeanUtils.copyProperties(recordSubEntity, saveRecordSubEntity);
            saveRecordSubEntity.setId(null);
            saveRecordSubEntity.setAssessmentEventType(assessmentEventType);
            saveRecordSubEntity.setAssessmentContent(assessmentContent);
            saveRecordSubEntity.setState(NumberUtils.INTEGER_ZERO);
            recordSubEntityList.add(saveRecordSubEntity);
            return true;
        }
        return false;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(RecordDelVo vo) {
        List<Long> ids = vo.getIds();
        List<Long> subIdList = vo.getSubIdList();
        /*
          删除主信息
         */
        if (!ids.isEmpty()) {
            try {
                recordDao.removeByIds(ids);
                recordSubDao.lambdaUpdate().in(RecordSubEntity::getRecordId, ids).remove();
                return;
            } catch (Exception e) {
                throw new BusinessException("列表删除失败，原因:" + e.getMessage());
            }
        }
        /*
          删除指标
         */
        if (!subIdList.isEmpty()) {
            try {
                recordSubDao.removeByIds(subIdList);
                return;
            } catch (Exception e) {
                throw new BusinessException("明细删除失败，原因:" + e.getMessage());
            }
        }
        throw new BusinessException("列表删除失败，请选择删除的条目");
    }

    @Override
    public RecordRes detail(Long id) {
        RecordEntity recordEntity = recordDao.lambdaQuery().eq(BaseEntity::getId, id).one();
        if (Objects.isNull(recordEntity)) {
            throw new BusinessException("查询失败,数据不存在");
        }
        List<RecordSubEntity> recordSubEntityList = recordSubDao.lambdaQuery().eq(RecordSubEntity::getRecordId, id).list();

        return buildRes(recordEntity, recordSubEntityList);
    }

    @Override
    public IPage<RecordEntity> list(QueryParam param) {
        //模糊
        param.getFuzzyFields().addAll(Arrays.asList());
        //排序
        param.getOrderMap().put("createTime", QueryParam.DESC);
        /*
          隔离？
         */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        return recordDao.queryPage(param, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public RecordRes edit(RecordAddEditVo vo) {
        /*
          修改主信息
         */
        RecordEntity recordEntity = recordDao.getById(vo.getId());
        BeanUtils.copyProperties(vo, recordEntity);
        if (StringUtils.isNotEmpty(vo.getModificationTime())) {
            recordEntity.setModificationTime(DateUtil.parse(vo.getModificationTime()));
        }
        recordDao.updateById(recordEntity);

        List<RecordAddEditVo.AddEditRecordSubInfo> recordSubEntities = vo.getRecordSubEntities();
        if (recordSubEntities.isEmpty()) {
            throw new BusinessException("至少需要一条指标");
        }

        /*
          修改指标
         */
        List<RecordSubEntity> saveRecordSubEntityList = Lists.newArrayList();
        List<RecordSubEntity> editRecordSubEntityList = Lists.newArrayList();
        List<Long> recordSubIdList = recordSubEntities.stream()
                .map(RecordAddEditVo.AddEditRecordSubInfo::getId)
                .collect(Collectors.toList());
        Map<Long, RecordSubEntity> recordSubIdAndEntityMap = recordSubDao.lambdaQuery().in(BaseEntity::getId, recordSubIdList).list()
                .stream()
                .collect(Collectors.toMap(RecordSubEntity::getId, e -> e));


        recordSubEntities.forEach(e -> {
            Long recordSubId = e.getId();
            if (!recordSubIdAndEntityMap.containsKey(recordSubId)) {
                /*
                  修改-新增新指标
                 */
                BeanUtils.copyProperties(recordEntity, e);
                e.setRecordId(recordEntity.getId());

                RecordSubEntity saveRecordSubEntity = new RecordSubEntity();
                BeanUtils.copyProperties(e, saveRecordSubEntity);
                saveRecordSubEntity.setModifierJobNum(vo.getModifierJobNum());
                saveRecordSubEntity.setState(NumberUtils.INTEGER_ZERO);
                saveRecordSubEntityList.add(saveRecordSubEntity);
            } else {
                /*
                  修改-修改原指标
                 */
                RecordSubEntity recordSubEntity = recordSubIdAndEntityMap.get(recordSubId);
                recordSubEntity.setAssessmentUnitCode(vo.getAssessmentUnitCode());
                recordSubEntity.setAssessmentUnitName(vo.getAssessmentUnitName());
                recordSubEntity.setAssessmentEventType(vo.getAssessmentEventType());
                recordSubEntity.setAssessmentContent(vo.getAssessmentContent());
                recordSubEntity.setAssessorJobNum(vo.getAssessorJobNum());
                recordSubEntity.setAssessorName(vo.getAssessorName());

                recordSubEntity.setScoreOfYear(e.getScoreOfYear());
                recordSubEntity.setScoreOfSeason(e.getScoreOfSeason());
                recordSubEntity.setAssessmentSubject(e.getAssessmentSubject());
                recordSubEntity.setSelfScore(e.getSelfScore());
                recordSubEntity.setSelfScoreReason(e.getSelfScoreReason());
                recordSubEntity.setSupportMaterialSelf(e.getSupportMaterialSelf());
                recordSubEntity.setSuggestScore(e.getSuggestScore());
                recordSubEntity.setSuggestReason(e.getSuggestReason());
                recordSubEntity.setSupportMaterialMajor(e.getSupportMaterialMajor());
                recordSubEntity.setPreliminaryScore(e.getPreliminaryScore());
                recordSubEntity.setPreliminaryReason(e.getPreliminaryReason());
                recordSubEntity.setSupportMaterialPrelim(e.getSupportMaterialPrelim());
                recordSubEntity.setFinalScore(e.getFinalScore());
                recordSubEntity.setFinalReason(e.getFinalReason());
                editRecordSubEntityList.add(recordSubEntity);
            }
        });

        List<RecordSubEntity> recordSubEntityList = Lists.newArrayList();
        recordSubEntityList.addAll(saveRecordSubEntityList);
        recordSubDao.saveBatch(saveRecordSubEntityList);
        if (!editRecordSubEntityList.isEmpty()) {
            recordSubDao.updateBatchById(editRecordSubEntityList);
            recordSubEntityList.addAll(editRecordSubEntityList);

        }

        /*
          修改-删除指标
         */
        List<Long> removeRecordSubIdList = vo.getRemoveRecordSubIdList();
        if (Objects.nonNull(removeRecordSubIdList) && !removeRecordSubIdList.isEmpty()) {
            recordSubDao.lambdaUpdate().in(BaseEntity::getId, removeRecordSubIdList)
                    .set(RecordSubEntity::getModifierJobNum, vo.getModifierJobNum())
                    .update();
            recordSubDao.removeByIds(removeRecordSubIdList);
        }

        return buildRes(recordEntity, recordSubEntityList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void score(RecordScoreVo vo) {
        List<RecordScoreVo.ScoreVo> scoreList = vo.getScoreList();
        try {
            for (RecordScoreVo.ScoreVo scoreVo : scoreList) {
                Long recordSubId = scoreVo.getRecordSubId();
                /*
                  建议评分（专业组建议）
                 */
                Integer suggestScore = scoreVo.getSuggestScore();
                if (Objects.nonNull(suggestScore)) {
                    recordSubDao.lambdaUpdate().eq(BaseEntity::getId, recordSubId)
                            .set(RecordSubEntity::getState, 2)
                            .set(RecordSubEntity::getSuggestScore, suggestScore)
                            .set(RecordSubEntity::getSuggestReason, scoreVo.getSuggestReason())
                            .set(RecordSubEntity::getSupportMaterialMajor, scoreVo.getSupportMaterialMajor())
                            .update();
                }
                /*
                  初评分（初评组初审）
                 */
                Integer preliminaryScore = scoreVo.getPreliminaryScore();
                if (Objects.nonNull(preliminaryScore)) {
                    recordSubDao.lambdaUpdate().eq(BaseEntity::getId, recordSubId)
                            .set(RecordSubEntity::getState, 3)
                            .set(RecordSubEntity::getPreliminaryScore, preliminaryScore)
                            .set(RecordSubEntity::getPreliminaryReason, scoreVo.getPreliminaryReason())
                            .set(RecordSubEntity::getSupportMaterialPrelim, scoreVo.getSupportMaterialPrelim())
                            .update();
                }
                /*
                  终评分（考评组长终评）
                 */
                Integer finalScore = scoreVo.getFinalScore();
                if (Objects.nonNull(finalScore)) {
                    recordSubDao.lambdaUpdate().eq(BaseEntity::getId, recordSubId)
                            .set(RecordSubEntity::getState, 4)
                            .set(RecordSubEntity::getFinalScore, finalScore)
                            .set(RecordSubEntity::getFinalReason, scoreVo.getFinalReason())
                            .update();
                }
            }
        } catch (Exception e) {
            throw new BusinessException("评分失败，原因：:" + e.getMessage());
        } finally {
            Long recordId = vo.getRecordId();
            /*
              评分后更新主表状态信息  以最小为准
             */
            Integer minState = recordSubDao.lambdaQuery().eq(RecordSubEntity::getRecordId, recordId).list()
                    .stream().map(RecordSubEntity::getState).min(Integer::compareTo).orElse(0);

            recordDao.lambdaUpdate().eq(BaseEntity::getId, recordId).set(RecordEntity::getState, minState).update();
        }

    }

    @Override
    public IPage<UserEntity> checkAssessor(QueryParam param) {
        Page<UserEntity> page = new Page<>(param.getPageIndex(), param.getPageSize());
        LambdaQueryWrapper<UserEntity> wrapper = new LambdaQueryWrapper<>();
        if (StringUtils.isNotBlank(param.getSearchText())) {
            wrapper.eq(UserEntity::getAssessmentUnitId, param.getSearchText());
        }
        IPage<UserEntity> paged = userDao.page(page, wrapper);

        return paged;
    }


    @Override
    public IPage<UserEntity> assessmentUnitList(QueryParam param, Integer OrgType) {
        Page<UserEntity> page = new Page<>(param.getPageIndex(), param.getPageSize());
        LambdaQueryWrapper<UserEntity> wrapper = new LambdaQueryWrapper<>();
        if (OrgType != null) {
            wrapper.eq(UserEntity::getOrgType, OrgType);
        }
        wrapper.groupBy(UserEntity::getAssessmentUnitName);
        IPage<UserEntity> paged = userDao.page(page, wrapper);

        return paged;
    }

    @Override
    public List<UserVO> assessmentDeptList(String unitName) {
        LambdaQueryWrapper<UserEntity> wrapper = new LambdaQueryWrapper<>();
        List<UserEntity> list = new ArrayList<>();
        if (StringUtils.isNotBlank(unitName)) {
            if ("中国能源建设集团浙江火电建设有限公司本部".equals(unitName)) {
                wrapper.eq(UserEntity::getAssessmentUnitName, unitName);
                wrapper.groupBy(UserEntity::getAssessmentDeptName);
                list = userDao.list(wrapper);
            } else {
                return null;
            }
            List<UserVO> vos = new ArrayList<>();
            for (UserEntity userEntity : list) {
                UserVO userVO = new UserVO();
                BeanUtils.copyProperties(userEntity, userVO);
                userVO.setAssessmentDeptName(userEntity.getAssessmentDeptName());
                userVO.setAssessmentDeptId(userEntity.getAssessmentDeptId());
                userVO.setAssessmentUnitName(userEntity.getAssessmentUnitName());
                userVO.setAssessmentUnitId(String.valueOf(userEntity.getAssessmentUnitId()));
                vos.add(userVO);
            }
            return vos;
        } else {
            return null;
        }

    }

    /**
     * 构建返回
     *
     * @param recordEntity            主信息
     * @param saveRecordSubEntityList 指标信息
     * @return RecordRes 返回
     * @see RecordRes
     */
    private RecordRes buildRes(RecordEntity recordEntity, List<RecordSubEntity> saveRecordSubEntityList) {
        RecordRes recordRes = new RecordRes();
        BeanUtils.copyProperties(recordEntity, recordRes);
        String approveUserInfo = recordEntity.getApproveUserInfo();
        List<com.ejianc.foundation.usercenter.vo.UserVO> approveUserInfoList =
                JSONObject.parseArray(approveUserInfo, com.ejianc.foundation.usercenter.vo.UserVO.class);
        recordRes.setApproveUserInfo(approveUserInfoList);

        List<RecordRes.RecordSubInfoRes> subEntityList = saveRecordSubEntityList.stream().map(recordSubEntity -> {
            RecordRes.RecordSubInfoRes resRecordSubEntity = new RecordRes.RecordSubInfoRes();

            BeanUtils.copyProperties(recordSubEntity, resRecordSubEntity);
            return resRecordSubEntity;
        }).collect(Collectors.toList());
        recordRes.setRecordSubEntitiyList(subEntityList);
        return recordRes;
    }
}
