package com.ejianc.business.dxcheck.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.dxcheck.dao.RecordDao;
import com.ejianc.business.dxcheck.dao.RecordSubDao;
import com.ejianc.business.dxcheck.entity.RecordEntity;
import com.ejianc.business.dxcheck.entity.RecordSubEntity;
import com.ejianc.business.dxcheck.enums.OrgTypeEnums;
import com.ejianc.business.dxcheck.model.res.AssessmentRankRes;
import com.ejianc.business.dxcheck.model.res.RecordStatisticsRes;
import com.ejianc.business.dxcheck.service.StatisticsServer;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/6/4 下午3:36
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class StatisticsServerImpl implements StatisticsServer {
    private final RecordDao recordDao;
    private final RecordSubDao recordSubDao;
    /**
     * 本季开始时间
     */
    private static final DateTime QUARTER_START_TIME = DateUtil.beginOfQuarter(new Date());
    /**
     * 本季结束时间
     */
    private static final DateTime QUARTER_END_TIME = DateUtil.endOfQuarter(new Date());
    /**
     * 本年开始时间
     */
    private static final DateTime YEAR_START_TIME = DateUtil.beginOfYear(new Date());
    /**
     * 本年结束时间
     */
    private static final DateTime YEAR_END_TIME = DateUtil.endOfYear(new Date());

    @Override
    public JSONArray getTree() {
        QueryWrapper<RecordEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.select("org_type", "assessment_unit_name", "assessment_unit_code");
        queryWrapper.groupBy("org_type", "assessment_unit_name");
        List<RecordEntity> recordEntities = recordDao.getBaseMapper().selectList(queryWrapper);

        return buildTree(recordEntities);
    }

    @Override
    public AssessmentRankRes assessmentRank() {
        /*
          当前季度
         */
        List<RecordSubEntity> nowQuarterList = recordSubDao.lambdaQuery()
                .eq(BaseEntity::getDr, 0)
                .isNotNull(RecordSubEntity::getFinalScore)
                .between(BaseEntity::getCreateTime, QUARTER_START_TIME, QUARTER_END_TIME)
                .list();
        /*
          当年
         */
        List<RecordSubEntity> nowYearList = recordSubDao.lambdaQuery()
                .eq(BaseEntity::getDr, 0)
                .isNotNull(RecordSubEntity::getFinalScore)
                .between(BaseEntity::getCreateTime, YEAR_START_TIME, YEAR_END_TIME)
                .list();

        /*
          1:本部
         */
        List<RecordSubEntity> bbYearList = nowYearList.stream().filter(e -> e.getOrgType() == 1).collect(Collectors.toList());
        List<RecordSubEntity> bbQuarterList = nowQuarterList.stream().filter(e -> e.getOrgType() == 1).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> bbInfoList = disposeScores(bbQuarterList, bbYearList);
        /*
          2:直管项目
         */
        List<RecordSubEntity> zgxmYearList = nowYearList.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<RecordSubEntity> zgxmQuarterList = nowQuarterList.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> zgxmInfoList = disposeScores(zgxmQuarterList, zgxmYearList);
        /*
          3:分子公司
         */
        List<RecordSubEntity> fzgsYearList = nowYearList.stream().filter(e -> e.getOrgType() == 3).collect(Collectors.toList());
        List<RecordSubEntity> fzgsQuarterList = nowQuarterList.stream().filter(e -> e.getOrgType() == 3).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> fzgsInfoList = disposeScores(fzgsQuarterList, fzgsYearList);

        return AssessmentRankRes.builder().bb(bbInfoList).zgxm(zgxmInfoList).fzgs(fzgsInfoList).build();
    }

    @Override
    public RecordStatisticsRes record(String unitCode) {
        List<RecordSubEntity> recordSubEntityList = recordSubDao.lambdaQuery()
                .eq(RecordSubEntity::getAssessmentUnitCode, unitCode)
                .isNotNull(RecordSubEntity::getFinalScore)
                .list();
        if (recordSubEntityList.isEmpty()) {
            return RecordStatisticsRes.builder().build();
        }
        ////本季得分
        BigDecimal quarterScore = recordSubEntityList.stream()
                //过滤
                .filter(e -> e.getCreateTime().after(QUARTER_START_TIME) && e.getCreateTime().before(QUARTER_END_TIME))
                .collect(Collectors.toList())
                .stream()
                //求和
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        //本年度加减分
        BigDecimal yearScore = recordSubEntityList.stream()
                //过滤
                .filter(e -> e.getCreateTime().after(YEAR_START_TIME) && e.getCreateTime().before(YEAR_END_TIME))
                .collect(Collectors.toList())
                .stream()
                //求和
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        //年度总分
        BigDecimal totalAnnualScore = NumberUtil.add(yearScore, 100);
        /*
          板块平均得分(本部/分子公司/项目)
         */
        //获取板块信息
        Integer orgType = recordSubEntityList.get(0).getOrgType();
        List<RecordSubEntity> orgTypeRecordSubList = recordSubDao.lambdaQuery()
                .eq(RecordSubEntity::getOrgType, orgType)
                .isNotNull(RecordSubEntity::getFinalScore)
                .list();
        //板块总分
        BigDecimal plateScoreSum = orgTypeRecordSubList.stream()
                //求和
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        //板块数量
        Map<String, List<RecordSubEntity>> collect = orgTypeRecordSubList.stream().collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode));
        int plateSize = collect.size();
        //板块平均得分
        BigDecimal plateAverageScore = plateScoreSum.divide(BigDecimal.valueOf(plateSize)).setScale(2, RoundingMode.HALF_UP);
        //排名
        int rank = orgTypeRecordSubList.stream()
                //汇总
                .collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode,
                        Collectors.reducing(BigDecimal.ZERO, record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO, BigDecimal::add)))
                .entrySet()
                .stream()
                //排序
                .sorted(Map.Entry.<String, BigDecimal>comparingByValue(Comparator.reverseOrder()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList())
                //获取当前部门排名
                .indexOf(unitCode) + 1;

        List<RecordStatisticsRes.RecordSubStatisticsRes> recordSubList = Lists.newArrayList();
        /**
         * 加扣明细
         */
        recordSubEntityList.forEach(e->{
            RecordStatisticsRes.RecordSubStatisticsRes recordSubStatisticsRes = new RecordStatisticsRes.RecordSubStatisticsRes();
            BeanUtils.copyProperties(e, recordSubStatisticsRes);
            recordSubStatisticsRes.setDeductionDetails(e.getFinalScore());
            recordSubList.add(recordSubStatisticsRes);
        });
        /**
         * 当年加扣
         */
        Map<Long, BigDecimal> normIdAndSumMap = recordSubEntityList.stream()
                //过滤
                .filter(e -> e.getCreateTime().after(YEAR_START_TIME) && e.getCreateTime().before(YEAR_END_TIME))
                .collect(Collectors.groupingBy(RecordSubEntity::getNormId, Collectors.reducing(BigDecimal.ZERO, RecordSubEntity::getFinalScore, BigDecimal::add)));
        recordSubList.forEach(e->{
            e.setDeductionYear(normIdAndSumMap.get(e.getNormId()));
        });
        /**
         * 加扣比例 = (当年加扣-扣分上限)/(加分上限-扣分上限)
         */
        recordSubList.forEach(e->{
            //当年加扣
            BigDecimal deductionYear = e.getDeductionYear();
            if (Objects.isNull(deductionYear)){
                return;
            }
            //扣分上限
            BigDecimal reduceLimit = NumberUtil.toBigDecimal(e.getReduceLimit());
            //加分上限
            BigDecimal plusLimit = NumberUtil.toBigDecimal(e.getPlusLimit());

            BigDecimal dividend = NumberUtil.sub(plusLimit, reduceLimit);
            if (dividend.compareTo(BigDecimal.ZERO) == 0 ){
                e.setDeductionRate("0%");
            }else {
                BigDecimal deductionRate = NumberUtil.div(NumberUtil.sub(deductionYear, reduceLimit), dividend)
                        .multiply(BigDecimal.valueOf(100))
                        .setScale(2, RoundingMode.HALF_UP);
                e.setDeductionRate(deductionRate + "%");
            }
        });

        return RecordStatisticsRes.builder()
                .quarterScore(quarterScore)
                .yearScore(yearScore)
                .totalAnnualScore(totalAnnualScore)
                .plateAverageScore(plateAverageScore)
                .rank(rank)
                .total(plateSize)
                .recordSubList(recordSubList)
                .build();
    }

    /**
     * 构建树
     *
     * @param list 数据
     * @return JSONArray 树
     */
    private static JSONArray buildTree(List<RecordEntity> list) {
        Map<Integer, List<RecordEntity>> orgTypeAndUnitInfoListMap = list.stream().collect(Collectors.groupingBy(RecordEntity::getOrgType));
        JSONArray jsonArray = JSONUtil.createArray();
        for (Map.Entry<Integer, List<RecordEntity>> orgTypeAndUnitInfoList : orgTypeAndUnitInfoListMap.entrySet()) {
            Integer orgType = orgTypeAndUnitInfoList.getKey();
            JSONObject node = JSONUtil.createObj();
            node.put("title", OrgTypeEnums.getNameByCode(orgType));
            node.put("key", orgType);

            List<RecordEntity> unitInfoList = orgTypeAndUnitInfoList.getValue();
            JSONArray childrenArray = JSONUtil.createArray();
            for (RecordEntity recordEntity : unitInfoList) {
                JSONObject childNode = JSONUtil.createObj();
                childNode.put("title", recordEntity.getAssessmentUnitName());
                childNode.put("key", recordEntity.getAssessmentUnitCode());
                childrenArray.add(childNode);
            }

            node.put("children", childrenArray);
            jsonArray.add(node);

        }
        return jsonArray;
    }

    /**
     * 分数处理 年度总分 = 初始分 + 当年分数(可能为负数)
     *
     * @param quarterList 当前季度  只是用来显示，不参与分数计算
     * @param yearList    当年  纬度最大，以这个为基础计算总分 用来参与分数计算  加上年度总分初始赋分
     * @return AssessmentRankRes.AssessmentRankInfo
     */
    private List<AssessmentRankRes.AssessmentRankInfo> disposeScores(List<RecordSubEntity> quarterList, List<RecordSubEntity> yearList) {
        List<AssessmentRankRes.AssessmentRankInfo> infoList = Lists.newArrayList();
        Map<String, List<RecordSubEntity>> unitYearMap = yearList.stream().collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode));
        Map<String, List<RecordSubEntity>> unitQuarterMap = quarterList.stream().collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode));

        for (Map.Entry<String, List<RecordSubEntity>> unitRecordYearList : unitYearMap.entrySet()) {
            String unitCode = unitRecordYearList.getKey();
            //年度总分初始分
            BigDecimal totalAnnualScore = BigDecimal.valueOf(100);
            List<RecordSubEntity> recordSubList = unitRecordYearList.getValue();
            //当前年度分
            BigDecimal yearScore = recordSubList.stream()
                    //求和
                    .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                    .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2,RoundingMode.HALF_UP);
            //该部门年度总分
            totalAnnualScore = totalAnnualScore.add(yearScore).setScale(2,RoundingMode.HALF_UP);
            //当前季度分
            BigDecimal quarterScore = BigDecimal.ZERO;
            if (unitQuarterMap.containsKey(unitCode)) {
                List<RecordSubEntity> recordSubEntityList = unitQuarterMap.get(unitCode);
                if (recordSubEntityList != null) {
                    quarterScore = recordSubEntityList.stream()
                            //求和
                            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                            .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2,RoundingMode.HALF_UP);
                }
            }
            AssessmentRankRes.AssessmentRankInfo infoRes = AssessmentRankRes.AssessmentRankInfo.builder()
                    .unitCode(unitCode)
                    .unitName(recordSubList.get(0).getAssessmentUnitName())
                    .quarterScore(quarterScore)
                    .yearScore(yearScore)
                    .totalAnnualScore(totalAnnualScore)
                    .build();
            infoList.add(infoRes);
        }
        //排序
        infoList.sort(Comparator.comparing(AssessmentRankRes.AssessmentRankInfo::getTotalAnnualScore).reversed());
        return infoList;
    }
}
