package com.ejianc.business.dxcheck.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.dxcheck.dao.NormDao;
import com.ejianc.business.dxcheck.dao.RecordDao;
import com.ejianc.business.dxcheck.dao.RecordSubDao;
import com.ejianc.business.dxcheck.dao.UserDao;
import com.ejianc.business.dxcheck.entity.NormEntity;
import com.ejianc.business.dxcheck.entity.RecordEntity;
import com.ejianc.business.dxcheck.entity.RecordSubEntity;
import com.ejianc.business.dxcheck.entity.UserEntity;
import com.ejianc.business.dxcheck.enums.OrgTypeEnums;
import com.ejianc.business.dxcheck.model.res.AssessmentRankRes;
import com.ejianc.business.dxcheck.model.res.RecordStatisticsRes;
import com.ejianc.business.dxcheck.service.StatisticsServer;
import com.ejianc.business.dxcheck.util.DateExtUtil;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/6/4 下午3:36
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class StatisticsServerImpl implements StatisticsServer {
    private final RecordDao recordDao;
    private final RecordSubDao recordSubDao;
    private final UserDao userDao;
    private final NormDao normDao;

    private final SessionManager sessionManager;

    @Override
    public JSONArray getTree() {

        @SuppressWarnings("unchecked")
        List<UserEntity> userEntities = userDao.lambdaQuery()
                .select(UserEntity::getOrgType, UserEntity::getAssessmentUnitName, UserEntity::getAssessmentUnitCode)
                .eq(UserEntity::getStatus, "启用")
                .groupBy(UserEntity::getOrgType, UserEntity::getAssessmentUnitCode)
                .list();

        return buildTree(userEntities);
    }


    @Override
    public AssessmentRankRes assessmentRank(String year) {
        Pair<DateTime, DateTime> yearStartEndTime = DateExtUtil.getYearStartEndTime(year);

        @SuppressWarnings("unchecked")
        List<UserEntity> userEntities = userDao.lambdaQuery()
                .select(UserEntity::getOrgType, UserEntity::getAssessmentUnitName,UserEntity::getAssessmentUnitShortName, UserEntity::getAssessmentUnitCode)
                .eq(UserEntity::getStatus, "启用")
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
                .groupBy(UserEntity::getOrgType, UserEntity::getAssessmentUnitCode)
                .list();
        /*
          当年
         */
        List<RecordSubEntity> nowYearList = recordSubDao.lambdaQuery()
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
                .list();
        /*
          1:本部
         */
        List<UserEntity> bbList = userEntities.stream().filter(e -> e.getOrgType() == 1).collect(Collectors.toList());
        List<RecordSubEntity> bbYearList = nowYearList.stream().filter(e -> e.getOrgType() == 1).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> bbInfoList = disposeScores(bbList, bbYearList, year);
        /*
          2:直管项目
         */
        List<UserEntity> zgxmList = userEntities.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<RecordSubEntity> zgxmYearList = nowYearList.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> zgxmInfoList = disposeScores(zgxmList, zgxmYearList, year);
        /*
          3:分子公司
         */
        List<UserEntity> fzgList = userEntities.stream().filter(e -> e.getOrgType() == 3).collect(Collectors.toList());
        List<RecordSubEntity> fzgsYearList = nowYearList.stream().filter(e -> e.getOrgType() == 3).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> fzgsInfoList = disposeScores(fzgList, fzgsYearList, year);

        return AssessmentRankRes.builder()
                .bb(bbInfoList)
                .zgxm(zgxmInfoList)
                .fzgs(fzgsInfoList)
                .build();
    }

    @Override
    public JSONArray record(String unitCode, String year) {
        JSONArray res = JSONUtil.createArray();
        JSONObject other = JSONUtil.createObj();
        if (StrUtil.isBlank(unitCode) || "null".equals(unitCode)) {
            return res;
        }

        //获取时间
        Pair<DateTime, DateTime> yearStartEndTime = DateExtUtil.getYearStartEndTime(year);
        /*
          板块平均得分(本部/分子公司/项目)
         */
        //获取板块信息
        List<Map<String, Object>> orgTypeInfoByUnitCode = userDao.getOrgTypeInfoByUnitCode(unitCode);

        Integer orgType = (Integer) orgTypeInfoByUnitCode.get(0).get("orgType");
        List<RecordSubEntity> orgTypeRecordSubList = recordSubDao.lambdaQuery()
                .eq(RecordSubEntity::getOrgType, orgType)
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
                .list();
        //本年度板块得分
        BigDecimal plateScore = orgTypeRecordSubList.stream()
                //求和
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        //本年度板块数量
        Long plateSize = (Long) orgTypeInfoByUnitCode.get(0).get("count");
        //本年度板块总分
        BigDecimal plateSum = plateScore.add(BigDecimal.valueOf(plateSize * 100L));
        //本年度板块平均得分
        BigDecimal plateAverageScore = plateSum.divide(plateSize.equals(NumberUtils.LONG_ZERO) ? BigDecimal.ONE : BigDecimal.valueOf(plateSize),
                2, RoundingMode.HALF_UP);
        //排名
        int rank = orgTypeRecordSubList.stream()
                //汇总
                .collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode,
                        Collectors.reducing(BigDecimal.ZERO,
                                record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO,
                                BigDecimal::add)))
                .entrySet()
                .stream()
                //排序
                .sorted(Map.Entry.comparingByValue(Comparator.reverseOrder()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList())
                //获取当前部门排名
                .indexOf(unitCode) + 1;

        List<RecordSubEntity> recordSubEntityList = recordSubDao.lambdaQuery()
                .eq(RecordSubEntity::getAssessmentUnitCode, unitCode)
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
                .list();
        //没打分=没排名
        int size = (int) recordSubEntityList.stream().filter(e -> e.getFinalScore() == null).count();

        //本年度加减分
        BigDecimal yearScore = recordSubEntityList.stream()
                //求和
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        //本年度总分
        BigDecimal totalAnnualScore = NumberUtil.add(yearScore, 100);

        DateExtUtil.QuarterStartEndTime quarterStartEndTime = DateExtUtil.getQuarterStartEndTime(year);
        //季度加减分
        BigDecimal q1 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ1StartTime())
                        && e.getCreateTime().before(quarterStartEndTime.getQ1EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        BigDecimal q2 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ2StartTime())
                        && e.getCreateTime().before(quarterStartEndTime.getQ2EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        BigDecimal q3 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ3StartTime())
                        && e.getCreateTime().before(quarterStartEndTime.getQ3EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        BigDecimal q4 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ4StartTime())
                        && e.getCreateTime().before(quarterStartEndTime.getQ4EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        other.put("q1", q1);
        other.put("q2", q2);
        other.put("q3", q3);
        other.put("q4", q4);
        other.put("yearScore", yearScore);
        other.put("totalAnnualScore", totalAnnualScore);
        other.put("plateAverageScore", plateAverageScore);
        other.put("rank", rank);
        other.put("total", Math.toIntExact(plateSize));
        if (size == recordSubEntityList.size()) {
            other.put("rank", null);
        }

        List<NormEntity> normList = normDao.getFirstSecondListOrder();
        Map<String, Map<String, List<NormEntity>>> collect1 = normList.stream()
                .collect(Collectors.groupingBy(NormEntity::getFirstClassify,
                        Collectors.groupingBy(NormEntity::getSecondClassify)));
        List<String> firstSet = normList.stream().map(NormEntity::getFirstClassify).distinct().collect(Collectors.toList());
        JSONArray normArray = JSONUtil.createArray();

        String userCode = String.valueOf(sessionManager.getUserContext().getUserCode());
        Map<Long, List<RecordSubEntity>> normIdAndSubListMap = recordSubEntityList.stream()
                .filter(e -> e.getNormId() != null)
                .collect(Collectors.groupingBy(RecordSubEntity::getNormId));
        for (String first : firstSet) {
            JSONObject firstNode = JSONUtil.createObj();
            firstNode.put("id", RandomUtil.randomString(7));
            firstNode.put("key", first);
            firstNode.put("normContent", first);
            Map<String, List<NormEntity>> stringListMap = collect1.get(first);
            JSONArray childrenArray = JSONUtil.createArray();
            Set<String> secondSet = stringListMap.keySet();
            for (String second : secondSet) {
                JSONObject secondNode = JSONUtil.createObj();
                secondNode.put("id", RandomUtil.randomString(7));
                secondNode.put("key", second);
                secondNode.put("normContent", second);
                if (stringListMap.containsKey(second)) {
                    List<NormEntity> secondList = stringListMap.get(second);
                    List<RecordStatisticsRes.NormRes> normRes = BeanMapper.mapList(secondList, RecordStatisticsRes.NormRes.class);
                    for (RecordStatisticsRes.NormRes normRes1 : normRes) {
                        Long normId = normRes1.getId();
                        List<RecordStatisticsRes.RecordSubStatisticsRes> recordSubList = Lists.newArrayList();
                        normRes1.setRecordSubList(recordSubList);

                        if (!normIdAndSubListMap.containsKey(normId)) {
                            continue;
                        }
                        List<RecordSubEntity> subList = normIdAndSubListMap.get(normId);
                        subList.forEach(e -> {
                            RecordStatisticsRes.RecordSubStatisticsRes recordSubStatisticsRes = new RecordStatisticsRes.RecordSubStatisticsRes();
                            BeanUtils.copyProperties(e, recordSubStatisticsRes);
                            recordSubList.add(recordSubStatisticsRes);
                        });
                        BigDecimal deductionYear = subList.stream()
                                //求和
                                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                                .reduce(BigDecimal.ZERO, BigDecimal::add);
                        normRes1.setDeductionYear(deductionYear);
                        //扣分上限
                        BigDecimal reduceLimit = NumberUtil.toBigDecimal(recordSubList.get(0).getReduceLimit());
                        //加分上限
                        BigDecimal plusLimit = NumberUtil.toBigDecimal(recordSubList.get(0).getPlusLimit());

                        BigDecimal dividend = NumberUtil.sub(plusLimit, reduceLimit);
                        BigDecimal deductionRate;
                        if (dividend.compareTo(BigDecimal.ZERO) == 0) {
                            deductionRate = BigDecimal.ZERO;
                            normRes1.setDeductionRate(deductionRate + "%");
                        } else {
                            deductionRate = NumberUtil.div(NumberUtil.sub(deductionYear, reduceLimit), dividend)
                                    .multiply(BigDecimal.valueOf(100))
                                    .setScale(2, RoundingMode.HALF_UP);
                            normRes1.setDeductionRate(deductionRate + "%");
                        }
                        recordSubList.forEach(e -> {
                            e.setDeductionYear(deductionYear);
                            e.setDeductionRate(deductionRate + "%");
                            e.setIsH(false);
                            if (StrUtil.isNotEmpty(e.getApproveUserCode())) {
                                List<String> list = JSONUtil.toList(JSONUtil.parseArray(e.getApproveUserCode()), String.class);
                                if (list.contains(userCode)) {
                                    e.setIsH(true);
                                }
                            }
                        });
                        normRes1.setRecordSubList(recordSubList);
                    }
                    secondNode.put("children", normRes);
                }
                childrenArray.add(secondNode);
            }
            firstNode.put("children", childrenArray);
            normArray.put(firstNode);
        }
        res.add(normArray);
        res.add(other);
        return res;
    }

    @Override
    public JSONArray tips(String year) {
        Pair<DateTime, DateTime> yearStartEndTime = DateExtUtil.getYearStartEndTime(year);

        JSONArray jsonArray = JSONUtil.createArray();
        JSONObject node = JSONUtil.createObj();
        Long deptId = sessionManager.getUserContext().getDeptId();
        String userCode = sessionManager.getUserContext().getUserCode();
        Long userId = sessionManager.getUserContext().getUserId();

        List<RecordSubEntity> agent = recordSubDao.lambdaQuery()
                .like(RecordSubEntity::getApproveUserCode, userCode)
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
                .list();
        //当前代办
        node.put("agent", agent.size());
        if (!agent.isEmpty()) {
            int count = 0;
            for (RecordSubEntity recordSubEntity : agent) {
                if (deptId != null && !recordSubEntity.getAssessmentUnitId().equals(String.valueOf(deptId))) {
                    count++;
                }
            }
            //单位
            node.put("unit", count);
        }

        List<RecordEntity> comingAgent = recordDao.lambdaQuery()
                .like(RecordEntity::getProfessionalGroupMembers, userId)
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
                .or()
                .like(RecordEntity::getEvaluationGroupMembers, userId)
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
                .list();
        //代办即将到达
        node.put("comingAgent", comingAgent.size());
        jsonArray.put(node);

        return jsonArray;
    }

    @Override
    public JSONArray getNewRecord() {
        List<RecordSubEntity> recordSubEntityList = recordSubDao.lambdaQuery().isNotNull(RecordSubEntity::getFinalScore).list();

        JSONArray jsonArray = JSONUtil.createArray();
        JSONObject node = JSONUtil.createObj();
        /*
          加分
         */
        List<RecordSubEntity> add = recordSubEntityList.stream()
                .filter(e -> e.getFinalScore().compareTo(BigDecimal.ZERO) >= 0)
                .sorted(Comparator.comparing(RecordSubEntity::getFinalScore).reversed())
                .collect(Collectors.toList());
        /*
          减分
         */
        List<RecordSubEntity> subtract = recordSubEntityList.stream()
                .filter(e -> e.getFinalScore().compareTo(BigDecimal.ZERO) < 0)
                .sorted(Comparator.comparing(RecordSubEntity::getFinalScore).reversed())
                .collect(Collectors.toList());
        node.put("add", add);
        node.put("subtract", subtract);
        jsonArray.put(node);
        return jsonArray;
    }

    /**
     * 构建树
     *
     * @param list 数据
     * @return JSONArray 树
     */
    private JSONArray buildTree(List<UserEntity> list) {
        String deptId = String.valueOf(sessionManager.getUserContext().getDeptId());

        Map<Integer, List<UserEntity>> orgTypeAndUnitInfoListMap = list.stream().collect(Collectors.groupingBy(UserEntity::getOrgType));
        JSONArray jsonArray = JSONUtil.createArray();
        for (Map.Entry<Integer, List<UserEntity>> orgTypeAndUnitInfoList : orgTypeAndUnitInfoListMap.entrySet()) {
            Integer orgType = orgTypeAndUnitInfoList.getKey();
            JSONObject node = JSONUtil.createObj();
            node.put("title", OrgTypeEnums.getNameByCode(orgType));
            node.put("key", orgType);

            List<UserEntity> unitInfoList = orgTypeAndUnitInfoList.getValue();
            JSONArray childrenArray = JSONUtil.createArray();
            for (UserEntity userEntity : unitInfoList) {
                JSONObject childNode = JSONUtil.createObj();
                childNode.put("title", userEntity.getAssessmentUnitName());
                childNode.put("key", userEntity.getAssessmentUnitCode());
                if (userEntity.getAssessmentUnitCode().equals(deptId)) {
                    childNode.put("isH", true);
                } else {
                    childNode.put("isH", false);
                }
                childrenArray.add(childNode);
            }
            node.put("children", childrenArray);
            jsonArray.add(node);
        }
        return jsonArray;
    }

    /**
     * 分数处理 年度总分 = 初始分 + 当年分数(可能为负数)
     */
    private List<AssessmentRankRes.AssessmentRankInfo> disposeScores(List<UserEntity> userList, List<RecordSubEntity> recordSubList, String year) {
        List<AssessmentRankRes.AssessmentRankInfo> infoList = Lists.newArrayList();

        Map<String, List<RecordSubEntity>> unitRecordSubMap = recordSubList.stream()
                .collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode));
        for (UserEntity userEntity : userList) {
            String assessmentUnitCode = userEntity.getAssessmentUnitCode();
            String assessmentUnitName = userEntity.getAssessmentUnitName();
            String assessmentUnitShortName = userEntity.getAssessmentUnitShortName();
            AssessmentRankRes.AssessmentRankInfo res = AssessmentRankRes.AssessmentRankInfo.builder()
                    .unitCode(assessmentUnitCode)
                    .unitName(assessmentUnitName)
                    .unitShortName(assessmentUnitShortName)
                    .build();
            if (!unitRecordSubMap.containsKey(assessmentUnitCode)) {
                res.setQ1(BigDecimal.ZERO);
                res.setQ2(BigDecimal.ZERO);
                res.setQ3(BigDecimal.ZERO);
                res.setQ4(BigDecimal.ZERO);
                //当年加减分
                res.setYearScore(BigDecimal.ZERO);
                //年度总分
                res.setTotalAnnualScore(BigDecimal.valueOf(100));
                infoList.add(res);
                continue;
            }
            List<RecordSubEntity> recordSubEntityList = unitRecordSubMap.get(assessmentUnitCode);
            //年度总分初始分
            BigDecimal totalAnnualScore = BigDecimal.valueOf(100);
            //当年加减分
            BigDecimal yearScore = recordSubEntityList.stream()
                    //求和
                    .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                    .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            //该部门年度总分
            totalAnnualScore = totalAnnualScore.add(yearScore).setScale(2, RoundingMode.HALF_UP);
            DateExtUtil.QuarterStartEndTime quarterStartEndTime = DateExtUtil.getQuarterStartEndTime(year);
            //季度加减分
            BigDecimal q1 = recordSubEntityList.stream()
                    .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ1StartTime())
                            && e.getCreateTime().before(quarterStartEndTime.getQ1EndTime()))
                    .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                    .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            BigDecimal q2 = recordSubEntityList.stream()
                    .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ2StartTime())
                            && e.getCreateTime().before(quarterStartEndTime.getQ2EndTime()))
                    .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                    .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            BigDecimal q3 = recordSubEntityList.stream()
                    .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ3StartTime())
                            && e.getCreateTime().before(quarterStartEndTime.getQ3EndTime()))
                    .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                    .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            BigDecimal q4 = recordSubEntityList.stream()
                    .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ4StartTime())
                            && e.getCreateTime().before(quarterStartEndTime.getQ4EndTime()))
                    .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                    .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            res.setQ1(q1);
            res.setQ2(q2);
            res.setQ3(q3);
            res.setQ4(q4);
            //当年加减分
            res.setYearScore(yearScore);
            //年度总分
            res.setTotalAnnualScore(totalAnnualScore);
            infoList.add(res);
        }
        //排序
        infoList.sort(Comparator.comparing(AssessmentRankRes.AssessmentRankInfo::getTotalAnnualScore).reversed());
        return infoList;
    }
}
