package com.ejianc.business.dxcheck.service.impl;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.dxcheck.dao.NormDao;
import com.ejianc.business.dxcheck.dao.RecordDao;
import com.ejianc.business.dxcheck.dao.RecordSubDao;
import com.ejianc.business.dxcheck.dao.TeamUserDao;
import com.ejianc.business.dxcheck.dao.UserDao;
import com.ejianc.business.dxcheck.entity.NormEntity;
import com.ejianc.business.dxcheck.entity.RecordEntity;
import com.ejianc.business.dxcheck.entity.RecordSubEntity;
import com.ejianc.business.dxcheck.entity.TeamUserEntity;
import com.ejianc.business.dxcheck.entity.UserEntity;
import com.ejianc.business.dxcheck.enums.OrgTypeEnums;
import com.ejianc.business.dxcheck.model.res.AssessmentRankRes;
import com.ejianc.business.dxcheck.model.res.NormScoreRes;
import com.ejianc.business.dxcheck.model.res.RecordStatisticsRes;
import com.ejianc.business.dxcheck.model.vo.NormScoreVo;
import com.ejianc.business.dxcheck.service.StatisticsServer;
import com.ejianc.business.dxcheck.util.DateExtUtil;
import com.ejianc.foundation.permission.api.IRoleApi;
import com.ejianc.foundation.permission.vo.RoleUserRelationVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.template.BaseEntity;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * @Author: LCL
 * @Date: 2024/6/4 下午3:36
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class StatisticsServerImpl implements StatisticsServer {
    private final RecordDao recordDao;
    private final RecordSubDao recordSubDao;
    private final UserDao userDao;
    private final NormDao normDao;
    private final TeamUserDao teamUserDao;

    private final SessionManager sessionManager;

    @Qualifier("com.ejianc.foundation.permission.api.IRoleApi")
    private final IRoleApi iRoleApi;

    @Override
    public JSONArray getTree() {

        @SuppressWarnings("unchecked")
        List<UserEntity> userEntities = userDao.lambdaQuery()
            .select(UserEntity::getOrgType, UserEntity::getAssessmentUnitName, UserEntity::getAssessmentUnitCode)
            .eq(UserEntity::getStatus, "启用").groupBy(UserEntity::getOrgType, UserEntity::getAssessmentUnitCode).list();

        return buildTree(userEntities);
    }

    @Override
    public AssessmentRankRes assessmentRank(String year) {
        Pair<DateTime, DateTime> yearStartEndTime = DateExtUtil.getYearStartEndTime(year);

        @SuppressWarnings("unchecked")
        List<UserEntity> userEntities = userDao.lambdaQuery()
            .select(UserEntity::getOrgType, UserEntity::getAssessmentUnitName, UserEntity::getAssessmentUnitShortName,
                UserEntity::getAssessmentUnitCode)
            .eq(UserEntity::getStatus, "启用")
            .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight())
            .groupBy(UserEntity::getOrgType, UserEntity::getAssessmentUnitCode).list();
        /*
          当年
         */
        List<RecordSubEntity> nowYearList = recordSubDao.lambdaQuery()
            .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight()).list();
        /*
          1:本部
         */
        List<UserEntity> bbList = userEntities.stream().filter(e -> e.getOrgType() == 1).collect(Collectors.toList());
        List<RecordSubEntity> bbYearList =
            nowYearList.stream().filter(e -> e.getOrgType() == 1).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> bbInfoList = disposeScores(bbList, bbYearList, year);
        /*
          2:直管项目
         */
        List<UserEntity> zgxmList = userEntities.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<RecordSubEntity> zgxmYearList =
            nowYearList.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> zgxmInfoList = disposeScores(zgxmList, zgxmYearList, year);
        /*
          3:分子公司
         */
        List<UserEntity> fzgList = userEntities.stream().filter(e -> e.getOrgType() == 3).collect(Collectors.toList());
        List<RecordSubEntity> fzgsYearList =
            nowYearList.stream().filter(e -> e.getOrgType() == 3).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> fzgsInfoList = disposeScores(fzgList, fzgsYearList, year);

        return AssessmentRankRes.builder().bb(bbInfoList).zgxm(zgxmInfoList).fzgs(fzgsInfoList).build();
    }

    @Override
    public JSONArray record(String unitCode, String year) {
        JSONArray res = JSONUtil.createArray();
        JSONObject other = JSONUtil.createObj();
        if (StrUtil.isBlank(unitCode) || "null".equals(unitCode)) {
            // 获取所有考核指标
            List<NormEntity> normList = normDao.getFirstSecondListOrder();
            Map<String, Map<String, List<NormEntity>>> collect1 = normList.stream().collect(Collectors
                .groupingBy(NormEntity::getFirstClassify, Collectors.groupingBy(NormEntity::getSecondClassify)));
            // 一级分类
            List<String> firstSet =
                normList.stream().map(NormEntity::getFirstClassify).distinct().collect(Collectors.toList());

            JSONArray normArray = JSONUtil.createArray();
            // 当前登录人账号
            for (String first : firstSet) {
                JSONArray childrenArray = JSONUtil.createArray();
                JSONObject firstNode = JSONUtil.createObj();
                firstNode.put("id", RandomUtil.randomString(7));
                firstNode.put("key", first);
                firstNode.put("normContent", first);
                // 获取一级分类下的所有二级分类Map
                Map<String, List<NormEntity>> stringListMap = collect1.get(first);
                // 二级分类排序
                TreeMap<String, List<NormEntity>> sort = MapUtil.sort(stringListMap);
                Set<String> secondSet = sort.keySet();
                for (String second : secondSet) {
                    // 二级node
                    JSONObject secondNode = JSONUtil.createObj();
                    secondNode.put("id", RandomUtil.randomString(7));
                    secondNode.put("key", second);
                    secondNode.put("normContent", second);
                    if (stringListMap.containsKey(second)) {
                        List<NormEntity> secondList = stringListMap.get(second);
                        List<RecordStatisticsRes.NormRes> normRes =
                            BeanMapper.mapList(secondList, RecordStatisticsRes.NormRes.class);
                        normRes.forEach(e -> e.setRecordSubList(Collections.emptyList()));
                        secondNode.put("children", normRes);
                    }
                    childrenArray.add(secondNode);
                }
                firstNode.put("children", childrenArray);
                normArray.put(firstNode);
            }
            res.add(normArray);
            other.put("q1", 0);
            other.put("q2", 0);
            other.put("q3", 0);
            other.put("q4", 0);
            other.put("yearScore", 0);
            other.put("totalAnnualScore", 0);
            other.put("plateAverageScore", 0);
            other.put("rank", 0);
            other.put("total", 0);
            res.add(other);
            return res;
        }
        // 获取时间
        Pair<DateTime, DateTime> yearStartEndTime = DateExtUtil.getYearStartEndTime(year);
        /*
          板块平均得分(本部/分子公司/项目)
         */
        // 获取板块信息
        UserEntity one =
            userDao.lambdaQuery().eq(UserEntity::getAssessmentUnitCode, unitCode).eq(UserEntity::getYear, year).one();
        if (Objects.isNull(one)) {
            // 获取所有考核指标
            List<NormEntity> normList = normDao.getFirstSecondListOrder();
            Map<String, Map<String, List<NormEntity>>> collect1 = normList.stream().collect(Collectors
                .groupingBy(NormEntity::getFirstClassify, Collectors.groupingBy(NormEntity::getSecondClassify)));
            // 一级分类
            List<String> firstSet =
                normList.stream().map(NormEntity::getFirstClassify).distinct().collect(Collectors.toList());

            JSONArray normArray = JSONUtil.createArray();
            // 当前登录人账号
            for (String first : firstSet) {
                JSONArray childrenArray = JSONUtil.createArray();
                JSONObject firstNode = JSONUtil.createObj();
                firstNode.put("id", RandomUtil.randomString(7));
                firstNode.put("key", first);
                firstNode.put("normContent", first);
                // 获取一级分类下的所有二级分类Map
                Map<String, List<NormEntity>> stringListMap = collect1.get(first);
                // 二级分类排序
                TreeMap<String, List<NormEntity>> sort = MapUtil.sort(stringListMap);
                Set<String> secondSet = sort.keySet();
                for (String second : secondSet) {
                    // 二级node
                    JSONObject secondNode = JSONUtil.createObj();
                    secondNode.put("id", RandomUtil.randomString(7));
                    secondNode.put("key", second);
                    secondNode.put("normContent", second);
                    if (stringListMap.containsKey(second)) {
                        List<NormEntity> secondList = stringListMap.get(second);
                        List<RecordStatisticsRes.NormRes> normRes =
                            BeanMapper.mapList(secondList, RecordStatisticsRes.NormRes.class);
                        normRes.forEach(e -> e.setRecordSubList(Collections.emptyList()));
                        secondNode.put("children", normRes);
                    }
                    childrenArray.add(secondNode);
                }
                firstNode.put("children", childrenArray);
                normArray.put(firstNode);
            }
            res.add(normArray);
            other.put("q1", 0);
            other.put("q2", 0);
            other.put("q3", 0);
            other.put("q4", 0);
            other.put("yearScore", 0);
            other.put("totalAnnualScore", 0);
            other.put("plateAverageScore", 0);
            other.put("rank", 0);
            other.put("total", 0);
            res.add(other);
            return res;
        }
        Integer orgType = one.getOrgType();

        List<RecordSubEntity> orgTypeRecordSubList = recordSubDao.lambdaQuery().eq(RecordSubEntity::getOrgType, orgType)
            .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight()).list();
        // 本年度板块得分
        BigDecimal plateScore = orgTypeRecordSubList.stream()
            // 求和
            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 本年度板块数量
        List<UserEntity> orgTypeList = userDao.lambdaQuery().eq(UserEntity::getOrgType, orgType)
            .eq(UserEntity::getYear, year).orderByAsc(UserEntity::getSerialNum).list();
        // 本年度板块总分
        BigDecimal plateSum = plateScore.add(BigDecimal.valueOf(orgTypeList.size() * 100L));
        // 本年度板块平均得分
        BigDecimal plateAverageScore = plateSum.divide(
            orgTypeList.isEmpty() ? BigDecimal.ONE : BigDecimal.valueOf(orgTypeList.size()), 2, RoundingMode.HALF_UP);
        // 排名
        LinkedHashMap<String, BigDecimal> assessmentMap = new LinkedHashMap<>();
        orgTypeList.forEach(e -> assessmentMap.put(e.getAssessmentUnitCode(), BigDecimal.valueOf(100L)));

        Map<String,
            BigDecimal> scoreMap = orgTypeRecordSubList.stream()
                .collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode,
                    Collectors.mapping(
                        record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO,
                        Collectors.reducing(BigDecimal.ZERO, BigDecimal::add))));
        for (Map.Entry<String, BigDecimal> stringIntegerEntry : assessmentMap.entrySet()) {
            String unitCodeKey = stringIntegerEntry.getKey();
            if (!scoreMap.containsKey(unitCodeKey)) {
                continue;
            }
            BigDecimal score = scoreMap.get(unitCodeKey);
            BigDecimal value = stringIntegerEntry.getValue();
            BigDecimal add = NumberUtil.add(value, score);
            stringIntegerEntry.setValue(add);
        }
        int rank = 0;
        if (CollUtil.isNotEmpty(assessmentMap)) {
            rank = assessmentMap.entrySet().stream()
                // 排序
                .sorted(Map.Entry.comparingByValue(Comparator.reverseOrder())).map(Map.Entry::getKey)
                .collect(Collectors.toList())
                // 获取当前部门排名
                .indexOf(unitCode) + 1;
        }
        List<RecordSubEntity> recordSubEntityList =
            recordSubDao.lambdaQuery().eq(RecordSubEntity::getAssessmentUnitCode, unitCode)
                .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight()).list();

        // 本年度加减分
        BigDecimal yearScore = recordSubEntityList.stream()
            // 求和
            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 本年度总分
        BigDecimal totalAnnualScore = NumberUtil.add(yearScore, 100);

        DateExtUtil.QuarterStartEndTime quarterStartEndTime = DateExtUtil.getQuarterStartEndTime(year);
        // 季度加减分
        BigDecimal q1 = recordSubEntityList.stream()
            .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ1StartTime())
                && e.getCreateTime().before(quarterStartEndTime.getQ1EndTime()))
            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
            .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        BigDecimal q2 = recordSubEntityList.stream()
            .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ2StartTime())
                && e.getCreateTime().before(quarterStartEndTime.getQ2EndTime()))
            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
            .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        BigDecimal q3 = recordSubEntityList.stream()
            .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ3StartTime())
                && e.getCreateTime().before(quarterStartEndTime.getQ3EndTime()))
            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
            .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        BigDecimal q4 = recordSubEntityList.stream()
            .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ4StartTime())
                && e.getCreateTime().before(quarterStartEndTime.getQ4EndTime()))
            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
            .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
        other.put("q1", q1);
        other.put("q2", q2);
        other.put("q3", q3);
        other.put("q4", q4);
        other.put("yearScore", yearScore);
        other.put("totalAnnualScore", totalAnnualScore);
        other.put("plateAverageScore", plateAverageScore);
        other.put("rank", rank);
        other.put("total", Math.toIntExact(orgTypeList.size()));
        // 获取所有考核指标
        List<NormEntity> normList = normDao.getFirstSecondListOrder();
        Map<String, Map<String, List<NormEntity>>> collect1 = normList.stream().collect(
            Collectors.groupingBy(NormEntity::getFirstClassify, Collectors.groupingBy(NormEntity::getSecondClassify)));
        // 一级分类
        List<String> firstSet =
            normList.stream().map(NormEntity::getFirstClassify).distinct().collect(Collectors.toList());

        JSONArray normArray = JSONUtil.createArray();
        // 指标——考核记录Map
        Map<Long, List<RecordSubEntity>> normIdAndSubListMap = recordSubEntityList.stream()
            .filter(e -> e.getNormId() != null).collect(Collectors.groupingBy(RecordSubEntity::getNormId));
        // 当前登录人账号
        String userCode = String.valueOf(sessionManager.getUserContext().getUserCode());
        for (String first : firstSet) {
            JSONArray childrenArray = JSONUtil.createArray();
            JSONObject firstNode = JSONUtil.createObj();
            firstNode.put("id", RandomUtil.randomString(7));
            firstNode.put("key", first);
            firstNode.put("normContent", first);
            // 获取一级分类下的所有二级分类Map
            Map<String, List<NormEntity>> stringListMap = collect1.get(first);
            // 二级分类排序
            TreeMap<String, List<NormEntity>> sort = MapUtil.sort(stringListMap);
            Set<String> secondSet = sort.keySet();
            for (String second : secondSet) {
                // 二级node
                JSONObject secondNode = JSONUtil.createObj();
                secondNode.put("id", RandomUtil.randomString(7));
                secondNode.put("key", second);
                secondNode.put("normContent", second);
                if (stringListMap.containsKey(second)) {
                    List<NormEntity> secondList = stringListMap.get(second);
                    List<RecordStatisticsRes.NormRes> normRes =
                        BeanMapper.mapList(secondList, RecordStatisticsRes.NormRes.class);
                    for (RecordStatisticsRes.NormRes normRes1 : normRes) {
                        Long normId = normRes1.getId();
                        List<RecordStatisticsRes.RecordSubStatisticsRes> recordSubList = Lists.newArrayList();
                        normRes1.setRecordSubList(recordSubList);

                        if (!normIdAndSubListMap.containsKey(normId)) {
                            continue;
                        }
                        List<RecordSubEntity> subList = normIdAndSubListMap.get(normId);
                        subList.forEach(e -> {
                            RecordStatisticsRes.RecordSubStatisticsRes recordSubStatisticsRes =
                                new RecordStatisticsRes.RecordSubStatisticsRes();
                            BeanUtils.copyProperties(e, recordSubStatisticsRes);
                            recordSubList.add(recordSubStatisticsRes);
                        });
                        BigDecimal deductionYear = subList.stream()
                            // 求和
                            .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                            .reduce(BigDecimal.ZERO, BigDecimal::add);
                        normRes1.setDeductionYear(deductionYear);
                        // 扣分上限
                        BigDecimal reduceLimit = NumberUtil.toBigDecimal(recordSubList.get(0).getReduceLimit());
                        // 加分上限
                        BigDecimal plusLimit = NumberUtil.toBigDecimal(recordSubList.get(0).getPlusLimit());

                        BigDecimal dividend = NumberUtil.sub(plusLimit, reduceLimit);
                        BigDecimal deductionRate;
                        if (dividend.compareTo(BigDecimal.ZERO) == 0) {
                            deductionRate = BigDecimal.ZERO;
                            normRes1.setDeductionRate(deductionRate + "%");
                        } else {
                            deductionRate = NumberUtil.div(NumberUtil.sub(deductionYear, reduceLimit), dividend)
                                .multiply(BigDecimal.valueOf(100)).setScale(2, RoundingMode.HALF_UP);
                            normRes1.setDeductionRate(deductionRate + "%");
                        }
                        recordSubList.forEach(e -> {
                            e.setDeductionYear(deductionYear);
                            e.setDeductionRate(deductionRate + "%");
                            e.setIsH(false);
                            if (StrUtil.isNotEmpty(e.getApproveUserCode())) {
                                List<String> list = Lists.newArrayList(e.getApproveUserCode().split(","));
                                Integer state = e.getState();
                                if (list.contains(userCode) && state != 6) {
                                    e.setIsH(true);
                                }
                            }
                        });
                        normRes1.setRecordSubList(recordSubList);
                    }
                    secondNode.put("children", normRes);
                }
                childrenArray.add(secondNode);
            }
            firstNode.put("children", childrenArray);
            normArray.put(firstNode);
        }
        res.add(normArray);
        res.add(other);
        return res;
    }

    @Override
    public JSONArray tips(String year) {
        Pair<DateTime, DateTime> yearStartEndTime = DateExtUtil.getYearStartEndTime(year);
        JSONArray jsonArray = JSONUtil.createArray();
        JSONObject node = JSONUtil.createObj();
        String userCode = sessionManager.getUserContext().getUserCode();
        Long userId = sessionManager.getUserContext().getUserId();

        List<RecordSubEntity> agent = recordSubDao.lambdaQuery().ne(RecordSubEntity::getState, 6)
            .like(RecordSubEntity::getApproveUserCode, userCode)
            .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight()).list();
        Set<String> collect = agent.stream().map(RecordSubEntity::getAssessmentUnitName).collect(Collectors.toSet());
        // 当前代办
        node.put("agent", agent.size());
        if (!agent.isEmpty()) {
            int count = collect.size();
            // 单位
            node.put("unit", count);
        }

        List<RecordEntity> comingAgent = recordDao.lambdaQuery().like(RecordEntity::getProfessionalGroupMembers, userId)
            .ne(RecordEntity::getState, 6)
            .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight()).or()
            .like(RecordEntity::getEvaluationGroupMembers, userId).ne(RecordEntity::getState, 6)
            .between(BaseEntity::getCreateTime, yearStartEndTime.getLeft(), yearStartEndTime.getRight()).list();
        // 代办即将到达
        node.put("comingAgent", comingAgent.size());
        // 代办情况
        // 考评组长id
        Set<Long> teamLeaderIdSet;
        // 各个环境roleId不一样 测试roleId 856893004038828089 正式roleId 860556490367316055
        List<Long> roleIdList = iRoleApi.toString().contains("test.ztpc.com")
            ? Collections.singletonList(856893004038828089L) : Collections.singletonList(860556490367316055L);
        CommonResponse<List<RoleUserRelationVO>> roleUser;
        try {
            roleUser = iRoleApi.getRoleUser(roleIdList, null);
            teamLeaderIdSet =
                roleUser.getData().stream().map(RoleUserRelationVO::getUserId).collect(Collectors.toSet());

            List<TeamUserEntity> allInfoListByTeamType = teamUserDao.getAllInfoListByTeamType(1);
            Set<Long> teamUserIdSet =
                allInfoListByTeamType.stream().map(TeamUserEntity::getTeamUserId).collect(Collectors.toSet());
            if (teamUserIdSet.contains(userId) || teamLeaderIdSet.contains(userId)) {
                node.put("isView", true);
            } else {
                node.put("isView", false);
            }
            jsonArray.put(node);
            return jsonArray;
        } catch (Exception e) {
            return jsonArray;
        }
    }

    @Override
    public JSONArray getNewRecord(Integer pageNo, Integer pageSize) {
        JSONArray jsonArray = JSONUtil.createArray();
        JSONObject node = JSONUtil.createObj();

        List<RecordSubEntity> recordSubEntityList =
            recordSubDao.lambdaQuery().isNotNull(RecordSubEntity::getFinalScore).list();
        /*
          加分
         */
        List<RecordSubEntity> addList =
            recordSubEntityList.stream().filter(e -> e.getFinalScore().compareTo(BigDecimal.ZERO) > 0)
                .sorted(Comparator.comparing(RecordSubEntity::getFinalScore).reversed()).collect(Collectors.toList());
        List<RecordSubEntity> addSubList = addList.subList(0, Math.min(addList.size(), 5));
        /*
          减分
         */
        List<RecordSubEntity> subtractList =
            recordSubEntityList.stream().filter(e -> e.getFinalScore().compareTo(BigDecimal.ZERO) < 0)
                .sorted(Comparator.comparing(RecordSubEntity::getFinalScore).reversed()).collect(Collectors.toList());
        List<RecordSubEntity> subtractSubList = subtractList.subList(0, Math.min(subtractList.size(), 5));

        node.put("add", addSubList);
        node.put("subtract", subtractSubList);

        recordSubEntityList.sort(Comparator.comparing(RecordSubEntity::getCreateTime).reversed());

        IPage<RecordSubEntity> page =
            new Page<>(Objects.isNull(pageNo) ? 1 : pageNo, Objects.isNull(pageSize) ? 10 : pageSize);
        IPage<RecordSubEntity> allPage =
            recordSubDao.lambdaQuery().isNotNull(RecordSubEntity::getFinalScore).page(page);

        node.put("all", allPage);

        jsonArray.put(node);
        return jsonArray;
    }

    @Override
    public JSONArray normScore(NormScoreVo vo) {
        String year = vo.getYear();
        HashSet<Long> normIds = new HashSet<>(vo.getNormIds());
        HashSet<String> unitIds = new HashSet<>(vo.getUnitIds());
        if (CollUtil.isEmpty(normIds) || CollUtil.isEmpty(unitIds)) {
            throw new BusinessException("指标信息或单位信息不能为空");
        }

        List<RecordSubEntity> recordSubEntityList =
            recordSubDao.lambdaQuery().eq(BaseEntity::getDr, 0L).eq(RecordSubEntity::getScoreOfYear, year)
                .in(RecordSubEntity::getAssessmentUnitId, unitIds).in(RecordSubEntity::getNormId, normIds).list();
        JSONArray array = JSONUtil.createArray();

        if (CollUtil.isEmpty(recordSubEntityList)) {
            return array;
        }
        Map<Long, NormEntity> normMap = normDao.lambdaQuery().eq(BaseEntity::getDr, 0L).in(BaseEntity::getId, normIds)
            .list().stream().collect(Collectors.toMap(BaseEntity::getId, Function.identity()));

        // 针对每个指标进行处理
        for (Long normId : normIds) {
            JSONObject obj = JSONUtil.createObj();
            Map<String, List<RecordSubEntity>> groupedByUnit =
                recordSubEntityList.stream().filter(e -> e.getNormId().equals(normId))
                    .collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitId));
            NormEntity normEntity = normMap.get(normId);
            String plusLimit = normEntity.getPlusLimit();
            String reduceLimit = normEntity.getReduceLimit();
            List<NormScoreRes> objects = Lists.newArrayList();

            // 针对每个单位的记录进行比较
            for (Map.Entry<String, List<RecordSubEntity>> entry : groupedByUnit.entrySet()) {
                List<RecordSubEntity> recordsForUnit = entry.getValue();

                BigDecimal add = recordsForUnit.stream().map(RecordSubEntity::getFinalScore)
                    .filter(finalScore -> finalScore.compareTo(BigDecimal.ZERO) > 0)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
                BigDecimal sub = recordsForUnit.stream().map(RecordSubEntity::getFinalScore)
                    .filter(finalScore -> finalScore.compareTo(BigDecimal.ZERO) < 0)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
                BigDecimal s1 = NumberUtil.sub(plusLimit, String.valueOf(add));
                BigDecimal s2 = NumberUtil.sub(reduceLimit, String.valueOf(sub));
                NormScoreRes build = NormScoreRes.builder().aa(s1).ss(s2).build();
                objects.add(build);
            }
            BigDecimal minAa = objects.stream().map(NormScoreRes::getAa).filter(Objects::nonNull)
                .min(BigDecimal::compareTo).orElse(BigDecimal.ZERO);
            BigDecimal minSs = objects.stream().map(NormScoreRes::getSs).filter(Objects::nonNull)
                .max(BigDecimal::compareTo).orElse(BigDecimal.ZERO);
            obj.put("minAa", minAa);
            obj.put("minSs", minSs);
            obj.put("normId", normId);
            array.add(obj);
        }
        return array;
    }

    /**
     * 构建树
     *
     * @param list 数据
     * @return JSONArray 树
     */
    private JSONArray buildTree(List<UserEntity> list) {
        String deptId = String.valueOf(sessionManager.getUserContext().getDeptId());

        Map<Integer, List<UserEntity>> orgTypeAndUnitInfoListMap =
            list.stream().collect(Collectors.groupingBy(UserEntity::getOrgType));
        JSONArray jsonArray = JSONUtil.createArray();
        for (Map.Entry<Integer, List<UserEntity>> orgTypeAndUnitInfoList : orgTypeAndUnitInfoListMap.entrySet()) {
            Integer orgType = orgTypeAndUnitInfoList.getKey();
            JSONObject node = JSONUtil.createObj();
            node.put("title", OrgTypeEnums.getNameByCode(orgType));
            node.put("key", orgType);

            List<UserEntity> unitInfoList = orgTypeAndUnitInfoList.getValue();
            JSONArray childrenArray = JSONUtil.createArray();
            for (UserEntity userEntity : unitInfoList) {
                JSONObject childNode = JSONUtil.createObj();
                childNode.put("title", userEntity.getAssessmentUnitName());
                childNode.put("key", userEntity.getAssessmentUnitCode());
                if (userEntity.getAssessmentUnitCode().equals(deptId)) {
                    childNode.put("isH", true);
                } else {
                    childNode.put("isH", false);
                }
                childrenArray.add(childNode);
            }
            node.put("children", childrenArray);
            jsonArray.add(node);
        }
        return jsonArray;
    }

    /**
     * 分数处理 年度总分 = 初始分 + 当年分数(可能为负数)
     */
    private List<AssessmentRankRes.AssessmentRankInfo> disposeScores(List<UserEntity> userList,
        List<RecordSubEntity> recordSubList, String year) {
        List<AssessmentRankRes.AssessmentRankInfo> infoList = Lists.newArrayList();

        Map<String, List<RecordSubEntity>> unitRecordSubMap =
            recordSubList.stream().collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode));
        for (UserEntity userEntity : userList) {
            String assessmentUnitCode = userEntity.getAssessmentUnitCode();
            String assessmentUnitName = userEntity.getAssessmentUnitName();
            String assessmentUnitShortName = userEntity.getAssessmentUnitShortName();
            AssessmentRankRes.AssessmentRankInfo res =
                AssessmentRankRes.AssessmentRankInfo.builder().unitCode(assessmentUnitCode).unitName(assessmentUnitName)
                    .unitShortName(assessmentUnitShortName).build();
            if (!unitRecordSubMap.containsKey(assessmentUnitCode)) {
                res.setQ1(BigDecimal.ZERO);
                res.setQ2(BigDecimal.ZERO);
                res.setQ3(BigDecimal.ZERO);
                res.setQ4(BigDecimal.ZERO);
                // 当年加减分
                res.setYearScore(BigDecimal.ZERO);
                // 年度总分
                res.setTotalAnnualScore(BigDecimal.valueOf(100));
                infoList.add(res);
                continue;
            }
            List<RecordSubEntity> recordSubEntityList = unitRecordSubMap.get(assessmentUnitCode);
            // 年度总分初始分
            BigDecimal totalAnnualScore = BigDecimal.valueOf(100);
            // 当年加减分
            BigDecimal yearScore = recordSubEntityList.stream()
                // 求和
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            // 该部门年度总分
            totalAnnualScore = totalAnnualScore.add(yearScore).setScale(2, RoundingMode.HALF_UP);
            DateExtUtil.QuarterStartEndTime quarterStartEndTime = DateExtUtil.getQuarterStartEndTime(year);
            // 季度加减分
            BigDecimal q1 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ1StartTime())
                    && e.getCreateTime().before(quarterStartEndTime.getQ1EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            BigDecimal q2 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ2StartTime())
                    && e.getCreateTime().before(quarterStartEndTime.getQ2EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            BigDecimal q3 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ3StartTime())
                    && e.getCreateTime().before(quarterStartEndTime.getQ3EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            BigDecimal q4 = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(quarterStartEndTime.getQ4StartTime())
                    && e.getCreateTime().before(quarterStartEndTime.getQ4EndTime()))
                .map(record -> record.getFinalScore() != null ? record.getFinalScore() : BigDecimal.ZERO)
                .reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            res.setQ1(q1);
            res.setQ2(q2);
            res.setQ3(q3);
            res.setQ4(q4);
            // 当年加减分
            res.setYearScore(yearScore);
            // 年度总分
            res.setTotalAnnualScore(totalAnnualScore);
            infoList.add(res);
        }
        // 排序
        infoList.sort(Comparator.comparing(AssessmentRankRes.AssessmentRankInfo::getTotalAnnualScore).reversed());
        return infoList;
    }
}
