package com.ejianc.foundation.file.util;

import com.aspose.pdf.Document;
import com.aspose.pdf.devices.*;
import com.ejianc.framework.core.exception.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

@Component
public class PdfProcessor {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private final Executor pdfTaskExecutor;

    public static final String PDF_TO_JPEG_IMG = "JEPG";
    public static final String PDF_TO_BMP_IMG = "BMP";
    public static final String PDF_TO_PNG_IMG = "PNG";

    @Autowired
    public PdfProcessor(Executor pdfTaskExecutor) {
        this.pdfTaskExecutor = pdfTaskExecutor;
    }

    /**
     * 将 PDF 按页拆分为多个独立的 PDF 文件
     */
    public List<byte[]> splitPdfToPages(byte[] pdfBytes) {
        return processAsync(pdfBytes, (doc, pageIndex) -> {
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            Document newDoc = new Document();
            try {
                newDoc.getPages().add(doc.getPages().get_Item(pageIndex + 1));
                newDoc.save(output);
                return output.toByteArray();
            } catch (Exception e) {
                logger.error("将 PDF 按页拆分异常", e);
                throw new BusinessException("PDF文件拆分异常");
            } finally {
                newDoc.close(); // 显式关闭资源
            }
        });
    }

    /**
     * 将 PDF 按页转换为图片
     */
    public List<byte[]> convertPdfToImages(byte[] pdfBytes, String imageFormat, int dpi) {
        return processAsync(pdfBytes, (doc, pageIndex) -> {
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            ImageDevice device = createImageDevice(imageFormat, new Resolution(dpi));
            try {
                device.process(doc.getPages().get_Item(pageIndex + 1), output);
                return output.toByteArray();
            } finally {
                output.close(); // 显式关闭资源（ImageDevice 无需关闭）
            }
        });
    }

    /**
     * 通用异步处理模板（确保顺序一致）
     */
    private <T> List<T> processAsync(byte[] pdfBytes, PdfPageProcessor<T> processor) {
        Document doc = null;
        try {
            doc = new Document(new ByteArrayInputStream(pdfBytes));
            int totalPages = doc.getPages().size();
            List<CompletableFuture<T>> futures = new ArrayList<>(totalPages);

            // 提交异步任务
            for (int pageIndex = 0; pageIndex < totalPages; pageIndex++) {
                final int currentPageIndex = pageIndex;
                futures.add(processPageAsync(pdfBytes, currentPageIndex, processor));
            }

            // 按顺序收集结果
            List<T> results = new ArrayList<>(totalPages);
            for (CompletableFuture<T> future : futures) {
                results.add(future.join());
            }
            return results;
        } catch (Exception e) {
            logger.error("PDF处理异常****************ERROR**************", e);
            throw new RuntimeException("PDF处理异常", e);
        } finally {
            if (doc != null) {
                doc.close(); // 显式关闭主文档资源
            }
        }
    }

    @Async("pdfTaskExecutor")
    public <T> CompletableFuture<T> processPageAsync(byte[] pdfBytes, int pageIndex, PdfPageProcessor<T> processor) {
        Document doc = null;
        try {
            doc = new Document(new ByteArrayInputStream(pdfBytes));
            return CompletableFuture.completedFuture(processor.process(doc, pageIndex));
        } catch (Exception e) {
            logger.error("PDF处理异常", e);
            throw new BusinessException("PDF处理异常", e);
        } finally {
            if (doc != null) {
                doc.close(); // 显式关闭子文档资源
            }
        }
    }

    /**
     * 创建图片设备
     */
    private ImageDevice createImageDevice(String format, Resolution resolution) {
         switch (format.toUpperCase()) {
             case "PNG":
                 return new PngDevice(resolution);
             case "BMP":
                 return new BmpDevice(resolution);
             default:
                 return new JpegDevice(resolution); // 默认为 JPEG
        }
    }

    /**
     * PDF 页面处理接口（函数式）
     */
    @FunctionalInterface
    private interface PdfPageProcessor<T> {
        T process(Document doc, int pageIndex) throws IOException;
    }

}
