package com.ejianc.util;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.ejianc.entity.Result;


import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * 增强版远程调用工具类（支持统一返回体）
 */
public class RemoteCallUtil {
    private static final Logger logger = LoggerFactory.getLogger(RemoteCallUtil.class);
    private static final ObjectMapper objectMapper = new ObjectMapper();

    // 默认连接超时时间(毫秒)
    private static final int DEFAULT_CONNECT_TIMEOUT = 5000;
    // 默认请求超时时间(毫秒)
    private static final int DEFAULT_SOCKET_TIMEOUT = 10000;

    private RemoteCallUtil() {
        // 工具类，防止实例化
    }

    /**
     * 发送GET请求并返回统一结果
     * @param url 请求地址
     * @param headers 请求头
     * @param dataType 数据类型
     * @return 统一返回体
     * @throws RemoteCallException 远程调用异常
     */
    public static <T> Result<T> doGet(String url, Map<String, String> headers, Class<T> dataType) throws RemoteCallException {
        String response = doGet(url, headers, DEFAULT_CONNECT_TIMEOUT, DEFAULT_SOCKET_TIMEOUT);
        return parseResult(response, dataType);
    }

    /**
     * 发送GET请求
     * @param url 请求地址
     * @param headers 请求头
     * @param connectTimeout 连接超时时间(毫秒)
     * @param socketTimeout 请求超时时间(毫秒)
     * @return 响应内容
     * @throws RemoteCallException 远程调用异常
     */
    private static String doGet(String url, Map<String, String> headers,
                                int connectTimeout, int socketTimeout) throws RemoteCallException {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        HttpGet httpGet = new HttpGet(url);

        // 设置请求头
        if (headers != null && !headers.isEmpty()) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                httpGet.setHeader(entry.getKey(), entry.getValue());
            }
        }

        // 设置超时时间
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(connectTimeout)
                .setSocketTimeout(socketTimeout)
                .build();
        httpGet.setConfig(requestConfig);

        try (CloseableHttpResponse response = httpClient.execute(httpGet)) {
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                throw new RemoteCallException("HTTP请求失败，状态码: " + statusCode);
            }

            HttpEntity entity = response.getEntity();
            if (entity != null) {
                return EntityUtils.toString(entity, StandardCharsets.UTF_8);
            }
            return null;
        } catch (IOException e) {
            logger.error("GET请求失败: {}", url, e);
            throw new RemoteCallException("GET请求失败: " + e.getMessage(), e);
        } finally {
            try {
                httpClient.close();
            } catch (IOException e) {
                logger.error("关闭HTTP客户端失败", e);
            }
        }
    }

    /**
     * 发送POST请求(对象作为请求体)
     * @param url 请求地址
     * @param requestBody 请求体对象
     * @param headers 请求头
     * @param dataType 数据类型
     * @return 统一返回体
     * @throws RemoteCallException 远程调用异常
     */
    public static <T, R> Result<R> doPost(String url, T requestBody, Map<String, String> headers, Class<R> dataType) throws RemoteCallException {
        String jsonParams = serializeRequest(requestBody);
        String response = doPostJson(url, jsonParams, headers, DEFAULT_CONNECT_TIMEOUT, DEFAULT_SOCKET_TIMEOUT);
        return parseResult(response, dataType);
    }

    /**
     * 发送POST请求(对象作为请求体)
     * @param url 请求地址
     * @param requestBody 请求体对象
     * @param headers 请求头
     * @param connectTimeout 连接超时时间(毫秒)
     * @param socketTimeout 请求超时时间(毫秒)
     * @param dataType 数据类型
     * @return 统一返回体
     * @throws RemoteCallException 远程调用异常
     */
    public static <T, R> Result<R> doPost(String url, T requestBody, Map<String, String> headers,
                                          int connectTimeout, int socketTimeout, Class<R> dataType) throws RemoteCallException {
        String jsonParams = serializeRequest(requestBody);
        String response = doPostJson(url, jsonParams, headers, connectTimeout, socketTimeout);
        return parseResult(response, dataType);
    }

    /**
     * 发送POST请求(JSON格式)
     * @param url 请求地址
     * @param jsonParams JSON参数
     * @param headers 请求头
     * @return 响应内容
     * @throws RemoteCallException 远程调用异常
     */
    private static String doPostJson(String url, String jsonParams, Map<String, String> headers) throws RemoteCallException {
        return doPostJson(url, jsonParams, headers, DEFAULT_CONNECT_TIMEOUT, DEFAULT_SOCKET_TIMEOUT);
    }

    /**
     * 发送POST请求(JSON格式)
     * @param url 请求地址
     * @param jsonParams JSON参数
     * @param headers 请求头
     * @param connectTimeout 连接超时时间(毫秒)
     * @param socketTimeout 请求超时时间(毫秒)
     * @return 响应内容
     * @throws RemoteCallException 远程调用异常
     */
    private static String doPostJson(String url, String jsonParams, Map<String, String> headers,
                                     int connectTimeout, int socketTimeout) throws RemoteCallException {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        HttpPost httpPost = new HttpPost(url);

        // 设置请求头
        if (headers != null && !headers.isEmpty()) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                httpPost.setHeader(entry.getKey(), entry.getValue());
            }
        } else {
            // 默认JSON请求头
            httpPost.setHeader("Content-Type", "application/json;charset=UTF-8");
        }

        // 设置请求体
        if (jsonParams != null) {
            StringEntity entity = new StringEntity(jsonParams, StandardCharsets.UTF_8);
            httpPost.setEntity(entity);
        }

        // 设置超时时间
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(connectTimeout)
                .setSocketTimeout(socketTimeout)
                .build();
        httpPost.setConfig(requestConfig);

        try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                throw new RemoteCallException("HTTP请求失败，状态码: " + statusCode);
            }

            HttpEntity entity = response.getEntity();
            if (entity != null) {
                return EntityUtils.toString(entity, StandardCharsets.UTF_8);
            }
            return null;
        } catch (IOException e) {
            logger.error("POST请求失败: {}", url, e);
            throw new RemoteCallException("POST请求失败: " + e.getMessage(), e);
        } finally {
            try {
                httpClient.close();
            } catch (IOException e) {
                logger.error("关闭HTTP客户端失败", e);
            }
        }
    }

    /**
     * 序列化请求对象为JSON字符串
     */
    private static <T> String serializeRequest(T requestBody) throws RemoteCallException {
        if (requestBody == null) {
            return null;
        }
        try {
            return objectMapper.writeValueAsString(requestBody);
        } catch (Exception e) {
            throw new RemoteCallException("请求对象序列化失败", e);
        }
    }

    /**
     * 解析响应字符串为统一返回体
     */
    private static <T> Result<T> parseResult(String response, Class<T> dataType) throws RemoteCallException {
        if (response == null) {
            throw new RemoteCallException("响应为空");
        }
        try {
            // 使用TypeReference处理泛型类型
            return objectMapper.readValue(response,
                    objectMapper.getTypeFactory().constructParametricType(Result.class, dataType));
        } catch (Exception e) {
            throw new RemoteCallException("响应解析失败: " + response, e);
        }
    }

    /**
     * 自定义异常类
     */
    public static class RemoteCallException extends Exception {
        public RemoteCallException(String message) {
            super(message);
        }

        public RemoteCallException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}