package com.ejianc.controller;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.entity.TemplateEntity;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.service.ITemplateCategoryService;
import com.ejianc.service.ITemplateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;

/**
 * @author:刘家乐
 * @version: 1
 * @document:
 */

@Api(tags = "模板管理-封装写法")
@RestController
@RequestMapping("template")
public class TemplateController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /**此处需要根据实际修改*/
    private static final String BILL_CODE = "LJL_TEMPLATE";

    @Value("${common.env.base-host}")
    private String BaseHost;

    @Autowired
    private ITemplateService templateService;

    @Autowired
    private ITemplateCategoryService categoryService;

    /**
     * 查询接口
     * @param params 查询参数
     * @return 返回分页数据
     */
    @ApiOperation(value = "分页查询模板列表（封装写法）")
    @RequestMapping(value = "/queryPage", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<TemplateEntity>> queryPage(@RequestBody Map<String, Object> params){
        // 从Map中获取分页参数
        long current = Long.parseLong(params.get("pageIndex").toString());
        long size = Long.parseLong(params.get("pageSize").toString());

        // 创建分页对象
        Page<TemplateEntity> page = new Page<>(current, size);

        // 创建查询条件
        LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();

        // 根据Map中的参数添加查询条件
        if (params.containsKey("templateName") && StringUtils.isNotEmpty((String) params.get("templateName"))) {
            queryWrapper.like(TemplateEntity::getTemplateName, params.get("templateName"));
        }

        if (params.containsKey("categoryId") && params.get("categoryId") != null) {
            queryWrapper.eq(TemplateEntity::getCategoryId, params.get("categoryId"));
        }

        if (params.containsKey("enableStatus") && params.get("enableStatus") != null) {
            queryWrapper.eq(TemplateEntity::getEnableStatus, params.get("enableStatus"));
        }

        if (params.containsKey("fileType") && StringUtils.isNotEmpty((String) params.get("fileType"))) {
            queryWrapper.eq(TemplateEntity::getFileType, params.get("fileType"));
        }

        // 添加排序条件
        queryWrapper.orderByDesc(TemplateEntity::getCreateTime);

        // 执行分页查询
        IPage<TemplateEntity> result = templateService.page(page, queryWrapper);

        return CommonResponse.success("查询列表数据成功！", result);
    }

    /**
     * 复杂查询条件的原生MyBatis-Plus写法
     * @param current 当前页
     * @param size 每页大小
     * @param templateName 模板名称（模糊查询）
     * @param categoryId 分类ID（精确查询）
     * @param fileTypes 文件类型列表（IN查询）
     * @param minFileSize 最小文件大小
     * @param maxFileSize 最大文件大小
     * @return 分页数据
     */
    @ApiOperation(value = "复杂查询条件的原生MyBatis-Plus写法")
    @GetMapping("/queryPageComplex")
    @ResponseBody
    public CommonResponse<IPage<TemplateEntity>> queryPageComplex(
            @ApiParam(value = "当前页", defaultValue = "1") @RequestParam(defaultValue = "1") long current,
            @ApiParam(value = "每页大小", defaultValue = "10") @RequestParam(defaultValue = "10") long size,
            @ApiParam(value = "模板名称（模糊查询）") @RequestParam(required = false) String templateName,
            @ApiParam(value = "分类ID（精确查询）") @RequestParam(required = false) Long categoryId,
            @ApiParam(value = "文件类型列表（IN查询）") @RequestParam(required = false) String[] fileTypes,
            @ApiParam(value = "最小文件大小") @RequestParam(required = false) Integer minFileSize,
            @ApiParam(value = "最大文件大小") @RequestParam(required = false) Integer maxFileSize) {

        // 创建分页对象
        Page<TemplateEntity> page = new Page<>(current, size);

        // 创建查询条件
        LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();

        // 模板名称模糊查询
        if (StringUtils.isNotEmpty(templateName)) {
            queryWrapper.like(TemplateEntity::getTemplateName, templateName);
        }

        // 分类ID精确查询
        if (categoryId != null) {
            queryWrapper.eq(TemplateEntity::getCategoryId, categoryId);
        }

        // 文件类型IN查询
        if (fileTypes != null && fileTypes.length > 0) {
            queryWrapper.in(TemplateEntity::getFileType, (Object[]) fileTypes);
        }

        // 文件大小范围查询
        if (minFileSize != null) {
            queryWrapper.ge(TemplateEntity::getFileSize, minFileSize);
        }
        if (maxFileSize != null) {
            queryWrapper.le(TemplateEntity::getFileSize, maxFileSize);
        }

        // 只查询启用的模板
        queryWrapper.eq(TemplateEntity::getEnableStatus, 1);

        // 添加排序条件
        queryWrapper.orderByDesc(TemplateEntity::getCreateTime)
                .orderByAsc(TemplateEntity::getSequence);

        // 执行分页查询
        IPage<TemplateEntity> result = templateService.page(page, queryWrapper);

        return CommonResponse.success("查询列表数据成功！", result);
    }

    /**
     * 使用原生MyBatis-Plus的selectMaps方法进行统计查询
     * @return 统计结果
     */
    @ApiOperation(value = "使用原生MyBatis-Plus的selectMaps方法进行统计查询")
    @GetMapping("/statistics")
    @ResponseBody
    public CommonResponse<Map<String, Object>> getStatistics() {

        LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();

        // 统计总模板数
        long totalCount = templateService.count(queryWrapper);

        // 统计各文件类型的模板数量
        queryWrapper.select(TemplateEntity::getFileType, TemplateEntity::getId);
        queryWrapper.groupBy(TemplateEntity::getFileType);

        // 统计启用的模板数量
        LambdaQueryWrapper<TemplateEntity> enabledWrapper = new LambdaQueryWrapper<>();
        enabledWrapper.eq(TemplateEntity::getEnableStatus, 1);
        long enabledCount = templateService.count(enabledWrapper);

        // 构建统计结果
        Map<String, Object> statistics = new HashMap<>();
        statistics.put("totalCount", totalCount);
        statistics.put("enabledCount", enabledCount);
        statistics.put("disabledCount", totalCount - enabledCount);

        return CommonResponse.success("统计查询成功！", statistics);
    }



}
