package com.ejianc.controller;

import com.ejianc.entity.UpLoad;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentRequestVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

@RestController
@RequestMapping(value = "/upload")
public class UploadFileController {

    private static final Logger logger = LoggerFactory.getLogger(UploadFileController.class);

    @Autowired
    private IAttachmentApi attachmentApi;

    @PostMapping(value = "/uploadFile")
    public CommonResponse<List<AttachmentVO>> uploadFile(@RequestBody UpLoad upload) {
        logger.info("开始处理文件上传，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId());
        
        try {
            // 参数验证
            if (!validateUploadParams(upload)) {
                return CommonResponse.error("参数验证失败");
            }
            
            // 构建数据URI
            String dataUri = buildDataUri(upload.getFileName(), upload.getBase64String());
            
            // 删除已有文件
            deleteExistingFiles(upload);
            
            // 上传新文件
            return uploadNewFile(upload, dataUri);
            
        } catch (Exception e) {
            logger.error("文件上传失败，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId(), e);
            return CommonResponse.error("文件上传失败: " + e.getMessage());
        }
    }

    /**
     * 验证上传参数
     */
    private boolean validateUploadParams(UpLoad upload) {
        if (upload == null) {
            logger.error("上传参数为空");
            return false;
        }
        
        if (StringUtils.isBlank(upload.getBase64String())) {
            logger.error("base64字符串为空");
            return false;
        }
        
        if (StringUtils.isBlank(upload.getBillType()) || 
            StringUtils.isBlank(upload.getSourceId()) || 
            StringUtils.isBlank(upload.getSourceType())) {
            logger.error("必要参数缺失: billType={}, sourceId={}, sourceType={}", 
                upload.getBillType(), upload.getSourceId(), upload.getSourceType());
            return false;
        }
        
        if (!isValidBase64(upload.getBase64String())) {
            logger.error("无效的base64字符串");
            return false;
        }
        
        return true;
    }
    
    /**
     * 删除已有文件
     */
    private void deleteExistingFiles(UpLoad upload) {
        try {
            Long sourceId = Long.valueOf(upload.getSourceId());
            attachmentApi.deleteFileByParam(sourceId, upload.getBillType(), upload.getSourceType());
            logger.debug("已删除已有文件，来源ID: {}", upload.getSourceId());
        } catch (NumberFormatException e) {
            logger.warn("来源ID格式错误，跳过删除操作: {}", upload.getSourceId());
        } catch (Exception e) {
            logger.warn("删除已有文件失败，继续上传: {}", e.getMessage());
        }
    }
    
    /**
     * 上传新文件
     */
    private CommonResponse<List<AttachmentVO>> uploadNewFile(UpLoad upload, String dataUri) {
        List<String> fileList = new ArrayList<>();
        fileList.add(dataUri);
        
        AttachmentRequestVO request = new AttachmentRequestVO();
        request.setBillType(upload.getBillType());
        request.setSourceId(upload.getSourceId());
        request.setSourceType(upload.getSourceType());
        request.setFileList(fileList);
        
        CommonResponse<List<AttachmentVO>> response = attachmentApi.uploadForBase64(request);
        
        if (response.isSuccess()) {
            logger.info("文件上传成功，单据类型: {}, 来源ID: {}, 文件数量: {}", 
                upload.getBillType(), upload.getSourceId(), 
                response.getData() != null ? response.getData().size() : 0);
        } else {
            logger.error("文件上传失败，单据类型: {}, 来源ID: {}, 错误信息: {}", 
                upload.getBillType(), upload.getSourceId(), response.getData());
        }
        
        return response;
    }

    /**
     * 验证base64字符串是否有效
     *
     * @param base64String base64字符串
     * @return 是否有效
     */
    private boolean isValidBase64(String base64String) {
        if (StringUtils.isBlank(base64String)) {
            return false;
        }

        try {
            Base64.getDecoder().decode(base64String);
            return true;
        } catch (IllegalArgumentException e) {
            logger.debug("无效的base64字符串: {}", e.getMessage());
            return false;
        }
    }
    
    /**
     * 构建base64数据URI
     * 
     * @param fileName 文件名
     * @param base64String base64编码的文件内容
     * @return 完整的数据URI
     */
    private String buildDataUri(String fileName, String base64String) {
        String extension = getFileExtension(fileName);
        return "data:application/" + extension + ";base64," + base64String;
    }
    
    /**
     * 获取文件扩展名
     * 
     * @param fileName 文件名
     * @return 文件扩展名
     */
    private String getFileExtension(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return "tmp";
        }

        int lastDotIndex = fileName.lastIndexOf('.');
        if (lastDotIndex > 0 && lastDotIndex < fileName.length() - 1) {
            return fileName.substring(lastDotIndex + 1);
        }

        return "docx";
    }
    


    @PostMapping(value = "/uploadFile2")
    public void uploadFile2(@RequestBody UpLoad upload) throws IOException {
        logger.info("开始测试文件上传");

        try {
            // 读取测试文件
            Path path = Paths.get("C:\\Users\\fc\\IdeaProjects\\pageoffice6-springboot2-master\\target\\classes\\static\\doc\\DataTagEdit\\111111.docx");
            byte[] fileBytes = Files.readAllBytes(path);

            // 转换为base64并构建数据URI
            String base64String = Base64.getEncoder().encodeToString(fileBytes);
            String dataUri = buildDataUri("111111.docx", base64String);

            // 准备上传参数
            List<String> fileList = new ArrayList<>();
            fileList.add(dataUri);

            AttachmentRequestVO request = new AttachmentRequestVO();
            request.setBillType("BT220216000000003");
            request.setSourceId("1956275048645922818");
            request.setSourceType("subContractFile");
            request.setFileList(fileList);

            // 删除已有文件
            CommonResponse<String> deleteResponse = attachmentApi.deleteFileByParam(1956275048645922818L, "BT220216000000003", "subContractFile");
            logger.info("删除响应: {}", deleteResponse);

            // 上传新文件
            CommonResponse<List<AttachmentVO>> uploadResponse = attachmentApi.uploadForBase64(request);
            logger.info("上传响应: {}", uploadResponse.getData());

        } catch (Exception e) {
            logger.error("测试文件上传失败", e);
        }
    }

}



