package com.ejianc.foundation.front.business.ide.service.impl;

import java.util.Date;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejianc.foundation.front.business.ide.bo.IdeAccountBo;
import com.ejianc.foundation.front.business.ide.entity.IdeUser;
import com.ejianc.foundation.front.business.ide.repository.IdeUserRepo;
import com.ejianc.foundation.front.business.ide.service.IdeAccountService;
import com.ejianc.foundation.front.util.MD5Util;
import com.ejianc.foundation.front.util.RandomUtils;
import com.ejianc.foundation.front.util.TimeUtils;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.support.idworker.util.IdWorker;


/**
 * 账号相关服务的实现
 */
@Transactional
@Service
public class IdeAccountServiceImpl implements IdeAccountService {
    @Autowired
    IdeUserRepo ideUserRepo;
    /**
     * 直接生成一个新账号
     */
	@Override
	public IdeAccountBo getNewAccount() throws Exception{
		//生成新帐号
		IdeUser newUser = generateNewAccount();
		//返回新生成的账号密码
		IdeAccountBo bo = new IdeAccountBo();
		bo.setGid(newUser.getId() + "");
		bo.setCode(newUser.getCode());
		bo.setPassword(newUser.getUnencryptPwd());//设置未加密密码
		bo.setPasswordMd5(newUser.getPassword());//设置加密密码
		return bo;
	}
	/**
	 * 生成一个新的账号信息
	 * @param defaultPwd 是否使用缺省密码
	 * @param qqAccessToken
	 * @param qqOpenId
	 * @param nickName
	 * @param sex
	 * @param avatarUrl
	 * @return
	 */
	private IdeUser generateNewAccount(boolean defaultPwd,String qqAccessToken,String qqOpenId,String nickName,int sex,String avatarUrl)
		throws Exception{
		//新生成的账号
		String newAccount = RandomUtils.getRandomNumbersAndLetters(3)+new Date().getTime();
		//确保生成的账号未被使用
		boolean inUse = false;
		int tryCount = 1;//尝试重新生成的次数
		do{
			inUse = checkAccountInUse(newAccount);
			if(inUse){
				newAccount=newAccount+1;
				tryCount = tryCount +1;
			}
			if(tryCount>100){
				return null;//尝试超过100次则终止生成。
			}
		}while(inUse);
		
		//生成新的密码
		String newPwd = defaultPwd?ACCOUNT_DEFAULT_PASSWORD:RandomUtils.getRandomNumbers(6);
		IdeUser newUser = createNewAccount(newAccount,newPwd,qqAccessToken,qqOpenId,nickName,sex,avatarUrl);
		return newUser;
	}
	//创建一个新账号信息
	private IdeUser createNewAccount(String code,String pwd,String qqAccessToken,String qqOpenId,String nickName,int sex,String avatarUrl)
	throws Exception{
		if(StringUtils.isNotBlank(code)&&StringUtils.isNotBlank(pwd)){
			if(checkAccountInUse(code)){
				throw new Exception("账号["+code+"]已被注册！");//确保账号不存在，已存在则返回null
			}
			//生成新帐号
			IdeUser newUser = new IdeUser();
			newUser.setId(IdWorker.getId());
			newUser.setCode(code);//设置账号
			newUser.setUnencryptPwd(pwd);//缓存未加密密码
			newUser.setPassword(MD5Util.MD5(pwd));//设置加密密码
			newUser.setCreateDate(new Date());
			newUser.setSex(sex);//设置性别
			if(!StringUtils.isBlank(nickName)){
				newUser.setName(nickName);
			}
			if(!StringUtils.isBlank(qqOpenId)){
				newUser.setQqOpenId(qqOpenId);
			}
			if(!StringUtils.isBlank(qqAccessToken)){
				newUser.setQqAccessToken(qqAccessToken);
			}
			if(!StringUtils.isBlank(avatarUrl)){
				newUser.setQqAvatarUrl(avatarUrl);
			}
			newUser.setCreateId(InvocationInfoProxy.getUserid());
			newUser.setIsDelete(0);
			//保存新的账号
			ideUserRepo.save(newUser);
			return newUser;
		}
		return null;
	}
	//生成一个新的账号信息
	private IdeUser generateNewAccount()throws Exception{
		return this.generateNewAccount(false,null, null, null, 0, null);
	}
	@Override
	public IdeUser regAccount(String code,String password) throws Exception{
		IdeUser newUser = this.createNewAccount(code,password,null, null, null, 0, null);
		return newUser;
	}
	@Override
	public boolean loginAccount(String code, String password) {
		String pwdMd5 = MD5Util.MD5(password);
		boolean result = loginAccountMd5(code,pwdMd5);
		return result;
	}
	@Override
	public boolean loginAccountMd5(String code, String pwdMd5) {
		IdeUser user = ideUserRepo.selectAccount(code, pwdMd5);
		if(user!=null){
				boolean isExpire = checkAccountIsExpire(user);
				if(isExpire){
					//账号已过期，则无法登录
					return false;
				}else{
					//记录最后一次登录的时间
					user.setLoginDate(new Date());
					ideUserRepo.update(user);
					return true;
				}
		}else{//账号不存在或密码不正确
			return false;
		}
	}
	
	@Override
	public boolean checkAccount(String code, String password) {
		String pwdMd5 = MD5Util.MD5(password);
		IdeUser user = ideUserRepo.selectAccount(code, pwdMd5);
		return (user!=null);
	}

	@Override
	public boolean checkAccountExpire(String code) {
		IdeUser user = findUserByCode(code);
		return checkAccountIsExpire(user);
	}
	
	//检查账号是否过期
	private boolean checkAccountIsExpire(IdeUser user){
		if(user==null)return true;
		Date date = user.getExpireDate();
		if(date!=null){
			Date expireDate = TimeUtils.stringToDate(TimeUtils.dateToString(date, "yyyy-MM-dd"),"yyyy-MM-dd");
			Date nowDate = TimeUtils.stringToDate(TimeUtils.dateToString(new Date(), "yyyy-MM-dd"),"yyyy-MM-dd");
			return nowDate.after(expireDate);
		}else{//不存在过期时间则永久有效
			return false;
		}
	}
	@Override
	public IdeUser findUserByCode(String code){
		List<IdeUser> user = ideUserRepo.selectAccountByCode(code);
		if(user!=null&&user.size()>0){//存在重复的多个取最后一个（注册异常的情况下会导致多个重复账号产生）
			return user.get(user.size()-1);//取最后一个
		}
		return null;
	}
	@Override
	public boolean checkAccountInUse(String code) {
		IdeUser user = findUserByCode(code);
		return (user!=null);
	}

	@Override
	public String changePassword(String code, String oldPwd, String newPwd) {
		if(StringUtils.isEmpty(oldPwd)){
			return "请输入原始密码";
		}
		String pwdMd5 = MD5Util.MD5(oldPwd);
		IdeUser user = ideUserRepo.selectAccount(code, pwdMd5);
		if(user==null){
			return "原始密码不正确"; 
		}
		if(StringUtils.isEmpty(newPwd)){
			return "请输入新的密码";
		}
		user.setPassword(MD5Util.MD5(newPwd));
		ideUserRepo.update(user);
		return "success";
	}

	@Override
	public IdeUser findAccount(String code, String password) {
		String pwdMd5 = MD5Util.MD5(password);
		IdeUser user = ideUserRepo.selectAccount(code, pwdMd5);
		return user;
	}

	@Override
	public IdeUser findByQQOpenId(String qqOpenId) {
		IdeUser user = ideUserRepo.selectByQQOpenId(qqOpenId);
		return user;
	}

	@Override
	public IdeUser generateNewAccountByQQ(String qqAccessToken,String qqOpenId,String nickName,int sex,String avatarUrl)throws Exception {
		IdeUser user = generateNewAccount(true,qqAccessToken, qqOpenId, nickName, sex, avatarUrl);
		return user;
	}
	/**
	 * 存储或更新用户原始信息
	 * @param user
	 */
	public void saveOrUpdate(IdeUser user){
		if(user.getId() != null && user.getId() > 0) {
			ideUserRepo.update(user);
		}else{
			ideUserRepo.save(user);
		}
	}
	/**
	 * 设置当前登录用户信息
	 * @param mv
	 * @param acc
	 */
	public void setCurrUser(HttpServletRequest request,IdeAccountBo acc){
		if(request!=null && request.getSession()!=null){
			//保存当前登录用户到Session
			request.getSession().setAttribute(SESSION_USER_KEY,acc);
		}
	}

	/**
	 * 获取当前用户基础信息
	 * @return
	 */
	public IdeAccountBo getCurrUser(HttpServletRequest request){
		if(request!=null && request.getSession()!=null){
			return (IdeAccountBo)request.getSession().getAttribute(SESSION_USER_KEY);
		}
		return null;
	}
	/**
	 * 清除当前登录用户信息
	 */
	public void clearCurrUser(HttpServletRequest request){
		if(request!=null && request.getSession()!=null){
			//从Session中移除当前用户
			request.getSession().removeAttribute(SESSION_USER_KEY);
			//注销会话Session
			request.getSession().invalidate();
		}
	}
	
	/**
	 * 修改用户原始信息
	 * @param user
	 */
	@Override
	public IdeUser userUpdate(IdeUser user){
		IdeUser ideUser = ideUserRepo.selectAccountByCode(user.getCode()).get(0);
		if (ideUser != null) {
			ideUser.setCode(user.getCode());
			ideUser.setEmail(user.getEmail());
			ideUser.setName(user.getName());
			ideUser.setPhone(user.getPhone());
			ideUserRepo.update(ideUser);
			return ideUser;
		}else {
			return null;
		}
	}
}
