package com.ejianc.foundation.front.business.ide.service.impl;

import com.ejianc.foundation.front.business.ide.bo.IdeUiDefineBO;
import com.ejianc.foundation.front.business.ide.bo.IdeUiFolderBO;
import com.ejianc.foundation.front.business.ide.entity.IdeUiFolder;
import com.ejianc.foundation.front.business.ide.repository.IdeUiFolderRepo;
import com.ejianc.foundation.front.business.ide.service.IdeUiDefineService;
import com.ejianc.foundation.front.business.ide.service.IdeUiFolderService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author andy
 */
@Service
public class IdeUiFolderServiceImpl implements IdeUiFolderService {
    @Autowired
    private IdeUiFolderRepo repo;
    @Autowired
    private IdeUiDefineService defineService;

    /**
     * 创建文件夹
     *
     * @param folder 文件夹
     * @return 文件夹id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public String create(IdeUiFolderBO folder) {
        IdeUiFolder entity = new IdeUiFolder();
        entity.setLibraryId(Long.parseLong(folder.getLibraryId()));
        entity.setName(folder.getName());
        if (StringUtils.isEmpty(folder.getParentId())) {
            entity.setIsRoot(true);
        }
        entity.setParentId(Long.parseLong(folder.getParentId()));
        repo.save(entity);
        return entity.getId().toString();
    }

    /**
     * 更新文件夹
     *
     * @param folder 文件夹
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(IdeUiFolderBO folder) {
        IdeUiFolder entity = repo.findOne(folder.getId());
        entity.setName(folder.getName());
        repo.update(entity);
    }

    /**
     * 删除文件夹
     *
     * @param id id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(String id) {
        deleteParent(id);
    }

    private void deleteParent(String parentId) {
        List<IdeUiFolder> children = repo.findByParentId(parentId);
        if (!CollectionUtils.isEmpty(children)) {
            children.forEach(child -> deleteParent(child.getId().toString()));
        }

        repo.delete(parentId);
        // 删除文件下所有组件
        defineService.deleteByFolderId(parentId);
    }

    /**
     * 删除库下所有文件夹
     *
     * @param libraryId 库id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deleteByLibraryId(String libraryId) {
        List<IdeUiFolder> folders = repo.findByLibraryIdAndIsRootTrue(libraryId);
        folders.forEach(folder -> delete(folder.getId().toString()));
    }

    /**
     * 查询文件夹详情
     *
     * @param id id
     * @param searchText 模糊匹配文本
     * @return 文件夹详情
     */
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    @Override
    public IdeUiFolderBO queryDetail(String id, String searchText) {
        IdeUiFolder entity = repo.findOne(id);

        if (entity == null) {
            return null;
        }

        IdeUiFolderBO vo = new IdeUiFolderBO();
        BeanUtils.copyProperties(entity, vo);

        setDetails(vo, vo, true, searchText);

        return vo;
    }

    /**
     * 查找库下所有文件夹及组件
     *
     * @param libraryId 库id
     * @param searchText 模糊匹配文本
     * @return 文件夹列表
     */
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    @Override
    public List<IdeUiFolderBO> findByLibraryId(String libraryId, String searchText) {
        return repo.findByLibraryIdAndIsRootTrue(libraryId).stream().map(item -> {
            IdeUiFolderBO vo = new IdeUiFolderBO();
            BeanUtils.copyProperties(item, vo);

            setDetails(vo, vo, true, searchText);

            return vo;
        }).collect(Collectors.toList());
    }

    /**
     * 设置子文件夹及组件详情
     *
     * @param root 根文件夹
     * @param parent 父文件夹
     * @param containsDefine 是否包含组件
     * @param searchText 模糊匹配文本
     */
    private void setDetails(IdeUiFolderBO root, IdeUiFolderBO parent, boolean containsDefine, String searchText) {
        if (containsDefine && CollectionUtils.isEmpty(root.getDefines())) {
            root.setDefines(Optional.ofNullable(defineService.findByFolderId(root.getId(), searchText)).orElse(new ArrayList<>()));
        }

        List<IdeUiFolder> children = repo.findByParentId(parent.getId());
        if (!CollectionUtils.isEmpty(children)) {
            parent.setChildren(children.stream().map(item -> {
                IdeUiFolderBO vo = new IdeUiFolderBO();
                BeanUtils.copyProperties(item, vo);

                if (containsDefine) {
                    List<IdeUiDefineBO> defines = root.getDefines();
                    defines.addAll(defineService.findByFolderId(item.getId().toString(), searchText));
                    root.setDefines(defines);
                }

                setDetails(root, vo, containsDefine, searchText);

                return vo;
            }).collect(Collectors.toList()));
        }
    }

    /**
     * 查找库下下所有文件夹
     *
     * @param libraryId 库id
     * @return 文件夹列表
     */
    @Override
    public List<IdeUiFolderBO> findLibraryFolders(String libraryId) {
        return repo.findByLibraryIdAndIsRootTrue(libraryId).stream().map(item -> {
            IdeUiFolderBO vo = new IdeUiFolderBO();
            BeanUtils.copyProperties(item, vo);

            setDetails(vo, vo, false, null);

            return vo;
        }).collect(Collectors.toList());
    }
}
