package com.ejianc.foundation.front.business.ide.service.impl;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejianc.foundation.front.business.ide.bo.IdeTeamBo;
import com.ejianc.foundation.front.business.ide.bo.IdeUserBo;
import com.ejianc.foundation.front.business.ide.entity.IdeTeam;
import com.ejianc.foundation.front.business.ide.entity.IdeTeamUser;
import com.ejianc.foundation.front.business.ide.entity.IdeUser;
import com.ejianc.foundation.front.business.ide.entity.JoinStatus;
import com.ejianc.foundation.front.business.ide.entity.RoleType;
import com.ejianc.foundation.front.business.ide.repository.IdeAppRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeTeamRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeTeamUserRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeUserRepo;
import com.ejianc.foundation.front.business.ide.service.IdeCommonService;
import com.ejianc.foundation.front.business.ide.service.IdeTeamService;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.support.idworker.util.IdWorker;

/**
 * Created by cdr_c on 2017/7/1.
 */
@Transactional(readOnly = true)
@Service
public class IdeTeamServiceImpl implements IdeTeamService {
    @Autowired
    private IdeAppRepo appRepo;
    @Autowired
    private IdeTeamRepo teamRepo;
    @Autowired
    private IdeTeamUserRepo teamUserRepo;
    @Autowired
    IdeCommonService ideCommonService;
    @Autowired
    private IdeUserRepo userRepo;

    /**
     * 保存团队
     *
     * @param team 团队
     * @return 团队
     */
    @Transactional
    @Override
    public IdeTeam save(IdeTeam team) {
        if (team.getId() != null && team.getId() > 0) {//创建团队
            IdeTeam _team = teamRepo.findOne(String.valueOf(team.getId()));
            _team.setName(team.getName());
            _team.setModifyDate(new Date());
            teamRepo.update(_team);
        } else {//修改团队
        	team.setId(IdWorker.getId());
        	team.setCode(ideCommonService.generateTeamCode());
            team.setMemberCount(1);
            team.setCreateDate(new Date());
            teamRepo.save(team); 
            
            IdeTeamUser teamUser = new IdeTeamUser();
            teamUser.setId(IdWorker.getId());
            teamUser.setTeamId(team.getId());
            teamUser.setUserId(team.getCreateId());
            teamUser.setJoinStatus(JoinStatus.IN.getValue());
            teamUser.setRoleType(RoleType.SUPER_ADMIN.getValue());
            teamUserRepo.save(teamUser);
        }
        return team;

    }

    /**
     * 更新团队服务地址配置
     *
     * @param id         主键
     * @param serverUrls 服务地址列表
     * @return 团队
     */
    @Transactional
    @Override
    public IdeTeam updateServerUrls(String id, String serverUrls) {
    	IdeTeam team = teamRepo.findOne(id);
        team.setServerUrls(serverUrls);
        team.setModifyDate(new Date());
        teamRepo.update(team);
        return team;
    }

    /**
     * 获取团队
     *
     * @param id 主键
     * @return 团队
     */
    @Override
    public IdeTeam findById(String id) {
        return teamRepo.findOne(id);
    }

    /**
     * 删除团队
     *
     * @param id 主键
     */
    @Transactional
    @Override
    public void deleteById(String id) {
    	IdeTeam team = teamRepo.findOne(id);
        if (team == null) {
            throw new BusinessException("该团队不存在或已被删除!");
        }
        if (appRepo.countApps(id) > 0) {
            throw new BusinessException("该团队下存在应用，不能删除!");
        }
        if (teamRepo.countTeamUsers(id) > 0) {
            throw new BusinessException("该团队下存在用户，不能删除!");
        }
        teamRepo.logicDel(String.valueOf(team.getId()));
        teamUserRepo.deleteByTeamId(String.valueOf(team.getId()));
    }

    /**
     * 分页查询团队
     *
     * @param key      关键字
     * @param userId   用户id
     * @param prodType 产品类型
     * @param pageable 分页信息
     * @return 团队分页列表
     */
    @Override
    public Page<IdeTeamBo> queryForPage(String key, String userId, int prodType, Pageable pageable) {
        Integer pageIndex = pageable.getPageNumber() * pageable.getPageSize();
        Integer pageSize = pageable.getPageSize();
        List<IdeTeamBo> teamBoList = teamRepo.queryForPageList(key,prodType,userId,pageIndex,pageSize);
        Long teamBoCount = teamRepo.queryForPageCount(key,prodType,userId);
        Page<IdeTeamBo> page = new PageImpl<>(teamBoList, pageable, teamBoCount);
        return page;
    }

    /**
     * 查询用户归属的团队
     *
     * @param userId 用户id
     * @param prodType 产品类型
     * @return 团队分页列表
     */
    @Override
    public List<IdeTeamBo> queryUserTeams(String userId, int prodType) {
    	List<IdeTeamBo> ideTeamBoList = teamRepo.queryUserTeams(userId, prodType);
        return ideTeamBoList;
    }

    /**
     * 获取团队下用户集合
     *
     * @param teamId 团队id
     * @return 用户集合
     */
    @Override
    public List<IdeUserBo> findUsers(String teamId) {
    	List<IdeUserBo> ideUserBoList = userRepo.findUsers(teamId);
    	return ideUserBoList;
    }

    /**
     * 申请加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     */
    @Transactional
    @Override
    public void applyToJoin(String teamId, String userId) {
    	IdeTeamUser teamUser = new IdeTeamUser();
    	teamUser.setId(IdWorker.getId());
        teamUser.setTeamId(Long.parseLong(teamId));
        teamUser.setUserId(Long.parseLong(userId));
        teamUser.setJoinStatus(JoinStatus.APPLYING.getValue());
        teamUser.setRoleType(RoleType.MEMBER.getValue());
        teamUserRepo.save(teamUser);
    }

    /**
     * 同意用户加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     */
    @Transactional
    @Override
    public void agreeToJoin(String teamId, String userId) {
    	IdeTeam team = teamRepo.findOne(teamId);
        if (team == null) {
            throw new BusinessException(String.format("团队不存在，团队id: %s", teamId));
        }
        IdeTeamUser teamUser = teamUserRepo.findByTeamIdAndUserId(teamId, userId);
        if (teamUser == null) {
            throw new BusinessException(String.format("团队-用户关联不存在，团队id: %s，用户id: %s", teamId, userId));
        }
        teamUser.setJoinStatus(JoinStatus.IN.getValue());
        teamUser.setRoleType(RoleType.MEMBER.getValue());
        teamUserRepo.update(teamUser);
        //更新成员数
        int usersNum = teamUserRepo.countUsers(teamId);
        team.setMemberCount(usersNum);
        teamRepo.update(team);
    }

    /**
     * 拒绝用户加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     */
    @Transactional
    @Override
    public void refuseToJoin(String teamId, String userId) {
    	IdeTeamUser teamUser = teamUserRepo.findByTeamIdAndUserId(teamId, userId);
        if (teamUser == null) {
            throw new BusinessException(String.format("团队-用户关联不存在，团队id: %s，用户id: %s", teamId, userId));
        }
        teamUser.setJoinStatus(JoinStatus.OUT.getValue());
        teamUserRepo.update(teamUser);

        IdeTeam team = teamRepo.findOne(teamId);
        if (team == null) {
            throw new BusinessException(String.format("团队不存在，团队id: %s", teamId));
        }
        teamRepo.update(team);
    }

    /**
     * 通过账号手动添加用户到团队
     *
     * @param teamId 团队id
     * @param code   账号
     */
    @Transactional
    @Override
    public void agreeToJoinByCode(String teamId, String code) {
    	IdeTeam team = teamRepo.findOne(teamId);
        if (team == null) {
            throw new BusinessException(String.format("团队不存在，团队id: %s", teamId));
        }
        List<IdeUser> users = userRepo.selectAccountByCode(code);
        if (CollectionUtils.isEmpty(users)) {
            throw new BusinessException(String.format("账号%s不存在", code));
        }
        String userId = users.get(0).getId().toString();
        IdeTeamUser teamUser = teamUserRepo.findByTeamIdAndUserId(teamId, userId);
        if (teamUser == null) {
            teamUser = new IdeTeamUser();
            teamUser.setId(IdWorker.getId());
            teamUser.setTeamId(Long.parseLong(teamId));
            teamUser.setUserId(Long.parseLong(userId));
            teamUser.setJoinStatus(JoinStatus.IN.getValue());
            teamUser.setRoleType(RoleType.MEMBER.getValue());
            teamUserRepo.save(teamUser);
        } else if (teamUser.getJoinStatus() != JoinStatus.IN.getValue()) {
            teamUser.setJoinStatus(JoinStatus.IN.getValue());
            teamUser.setRoleType(RoleType.MEMBER.getValue());
            teamUserRepo.update(teamUser);
        }
        //更新成员数
        int usersNum = teamUserRepo.countUsers(teamId);
        team.setMemberCount(usersNum);
        teamRepo.update(team);
    }

    /**
     * 从团队移除用户
     *
     * @param teamId  团队id
     * @param userIds 用户id数组
     */
    @Transactional
    @Override
    public void removeUsers(String teamId, String[] userIds) {
    	IdeTeam team = teamRepo.findOne(teamId);
        if (team == null) {
            throw new BusinessException(String.format("团队不存在，团队id: %s", teamId));
        }
        String userIdStr = "";
        for(String userId:userIds) {
        	userIdStr += userId + ",";
        }
        if(StringUtils.isNotBlank(userIdStr)) {
        	userIdStr = userIdStr.substring(0, userIdStr.length()-1);
        	teamUserRepo.deleteUsers(teamId, userIdStr);
        }
        //更新成员数
        int usersNum = teamUserRepo.countUsers(teamId);
        team.setMemberCount(usersNum);
        teamRepo.update(team);
    }

    /**
     * 提升用户为创建者
     *
     * @param teamId     团队id
     * @param userId 用户id
     */
    @Transactional
    @Override
    public void updateTeamCreator(String teamId, String userId) {
    	IdeTeamUser teamUser = teamUserRepo.getTeamUser(teamId, userId);
        if (teamUser == null) {
            throw new BusinessException(String.format("用户不在该团队中，用户id：%s, 团队id：%s", userId, teamId));
        } else if (teamUser.getJoinStatus() != JoinStatus.IN.getValue()) {
            throw new BusinessException(String.format("用户尚未加入该团队，用户id：%s, 团队id：%s", userId, teamId));
        }

        IdeTeamUser creator = teamUserRepo.getTeamCreator(teamId);
        if (creator != null) {
            creator.setRoleType(RoleType.MEMBER.getValue());
            teamUserRepo.update(creator);
        }

        teamUser.setRoleType(RoleType.SUPER_ADMIN.getValue());
        teamUserRepo.update(teamUser);
    }

    /**
     * 提升用户为管理员
     *
     * @param teamId      团队id
     * @param userIds 用户id数组
     */
    @Transactional
    @Override
    public void setTeamManagers(String teamId, String[] userIds) {
    	String userIdStr = "";
    	for(String userId:userIds) {
    		userIdStr += userId + ",";
    	}
    	if(StringUtils.isNotBlank(userIdStr)) {
    		userIdStr = userIdStr.substring(0, userIdStr.length()-1);
    	}
    	List<IdeTeamUser> teamUsers = teamUserRepo.getTeamUsers(teamId, userIdStr);
        if (CollectionUtils.isEmpty(teamUsers)) {
            throw new BusinessException(String.format("用户都不在该团队中，用户ids：%s，团队id：%s", userIdStr, teamId));
        }
        List<Long> notJoinTeamUserIds = teamUsers.stream().filter(teamUser -> teamUser.getJoinStatus() != JoinStatus.IN.getValue()).map(IdeTeamUser::getUserId).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(notJoinTeamUserIds)) {
            throw new BusinessException(String.format("存在用户尚未加入该团队，用户ids：%s, 团队id：%s", notJoinTeamUserIds.toString(),
                    teamId));
        }
        for(IdeTeamUser ideTeamUser:teamUsers) {
        	ideTeamUser.setRoleType(RoleType.ADMIN.getValue());
        	teamUserRepo.update(ideTeamUser);
        }
    }

    /**
     * 撤销用户管理员角色
     *
     * @param teamId      团队id
     * @param userIds 用户id数组
     */
    @Transactional
    @Override
    public void cancelTeamManagers(String teamId, String[] userIds) {
    	String userIdStr = "";
    	for(String userId:userIds) {
    		userIdStr += userId + ",";
    	}
    	if(StringUtils.isNotBlank(userIdStr)) {
    		userIdStr = userIdStr.substring(0, userIdStr.length()-1);
    	}
    	List<IdeTeamUser> teamUsers = teamUserRepo.getTeamUsers(teamId, userIdStr);
        if (CollectionUtils.isEmpty(teamUsers)) {
            throw new BusinessException(String.format("用户都不在该团队中，用户ids：%s，团队id：%s", userIdStr, teamId));
        }
        List<Long> notJoinTeamUserIds = teamUsers.stream().filter(teamUser -> teamUser.getJoinStatus() != JoinStatus.IN.getValue()).map(IdeTeamUser::getUserId).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(notJoinTeamUserIds)) {
            throw new BusinessException(String.format("存在用户尚未加入该团队，用户ids：%s, 团队id：%s", notJoinTeamUserIds.toString(),
                    teamId));
        }
        for(IdeTeamUser ideTeamUser:teamUsers) {
        	ideTeamUser.setRoleType(RoleType.MEMBER.getValue());
        	teamUserRepo.update(ideTeamUser);
        }
    }

	@Override
	public Page<IdeTeamBo> findPage(String searchText,int prodType,boolean isDelete, Pageable pageable) {
		Integer pageIndex = pageable.getPageNumber() * pageable.getPageSize();
		Integer pageSize = pageable.getPageSize();
		
		List<IdeTeamBo> teamBoList = teamRepo.queryTeamList(searchText,prodType,isDelete,pageIndex,pageSize);
		Long teamBoCount = teamRepo.queryTeamCount(searchText,prodType,isDelete);
		Page<IdeTeamBo> page = new PageImpl<>(teamBoList, pageable, teamBoCount);
		return page;
	}

	@Override
	public List<IdeTeam> findTeamByProdType(int prodType) {
		return teamRepo.findTeamByProdType(prodType);
	}
	@Transactional
	@Override
	public void deleteByIds(List<String> ids, boolean isRemove) throws BusinessException, Exception {
		String idStr = "";
		if(ids != null && ids.size() > 0) {
			for(String id:ids) {
				idStr += id + ",";
			}
			
			idStr = idStr.substring(0, idStr.length()-1);
		}
		if (isRemove) {
			teamRepo.removeByIds(idStr);
			teamUserRepo.deleteUsersByTeamIds(idStr);
		}else {
	    	List<IdeTeam> teams = teamRepo.findByIds(idStr);
	    	for(IdeTeam team : teams){
	    	   if (appRepo.countApps(String.valueOf(team.getId())) > 0) {
	    		    throw new BusinessException("团队["+team.getName()+"]下存在应用，不能删除!");
	    		}
	    	   if (teamRepo.countTeamUsers(String.valueOf(team.getId())) > 0) {
	    		    throw new BusinessException("团队["+team.getName()+"]下存在用户，不能删除!");
	    		 }
	    	   }
	    	 //执行批量删除
	    	teamRepo.deleteByIds(idStr);
		}
	}
	
	@Transactional
	@Override
	public void recoverByIds(List<String> ids) throws BusinessException, Exception {
		String idStr = "";
		if(ids != null && ids.size() > 0) {
			for(String id:ids) {
				idStr += id + ",";
			}
			
			idStr = idStr.substring(0, idStr.length()-1);
			teamRepo.recoverByIds(idStr);
		}
	}

	@Override
	public Page<IdeUserBo> findUsersByTeamId(String searchText, boolean isDelete, String teamId, Pageable pageable) {
		
		return null;
	}

	@Transactional
	@Override
	public void deleteUsersByTeamId(List<String> ids, String teamId, boolean isRemove) throws BusinessException, Exception {
		String idStr = "";
		if(ids != null && ids.size() > 0) {
			for(String id:ids) {
				idStr += id + ",";
			}
			idStr = idStr.substring(0, idStr.length() - 1);
		}
		if (isRemove) {
			teamUserRepo.deleteUsers(teamId, idStr);
		}else {
			IdeTeam team = teamRepo.findOne(teamId);
	        if (team == null) {
	            throw new BusinessException(String.format("团队不存在，团队id: %s", teamId));
	        }
	        teamUserRepo.deleteUsersByTeamId(teamId, idStr);
	        //更新成员数
	        int usersNum = teamUserRepo.countUsers(teamId);
	        team.setMemberCount(usersNum);
	        teamRepo.update(team);
		}

	}

	@Transactional
	@Override
	public void recoverUsersByIds(List<String> ids, String teamId) throws BusinessException, Exception {
		IdeTeam team = teamRepo.findOne(teamId);
        if (team == null) {
            throw new BusinessException(String.format("团队不存在，团队id: %s", teamId));
        }
        String idStr = "";
		if(ids != null && ids.size() > 0) {
			for(String id:ids) {
				idStr += id + ",";
			}
			idStr = idStr.substring(0, idStr.length() - 1);
			teamUserRepo.recoverUsers(teamId, idStr);
		}
        //更新成员数
        int usersNum = teamUserRepo.countUsers(teamId);
        team.setMemberCount(usersNum);
        teamRepo.update(team);
	}
}
