package com.ejianc.foundation.front.business.ide.service.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.http.HttpResponse;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.front.business.ide.Constant;
import com.ejianc.foundation.front.business.ide.bo.IdeAppBo;
import com.ejianc.foundation.front.business.ide.bo.IdeModuleBo;
import com.ejianc.foundation.front.business.ide.bo.IdeProductionBo;
import com.ejianc.foundation.front.business.ide.bo.IdeTeamBo;
import com.ejianc.foundation.front.business.ide.entity.IdeApp;
import com.ejianc.foundation.front.business.ide.entity.IdeModule;
import com.ejianc.foundation.front.business.ide.entity.IdeOnline;
import com.ejianc.foundation.front.business.ide.entity.IdeRelease;
import com.ejianc.foundation.front.business.ide.entity.IdeReleasePage;
import com.ejianc.foundation.front.business.ide.entity.IdeTeam;
import com.ejianc.foundation.front.business.ide.repository.IdeModuleProRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeModuleRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeOnlineRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeReleasePageRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeReleaseRepo;
import com.ejianc.foundation.front.business.ide.repository.IdeTeamRepo;
import com.ejianc.foundation.front.business.ide.service.IdeAppService;
import com.ejianc.foundation.front.business.ide.service.IdeEnvironmentService;
import com.ejianc.foundation.front.business.ide.service.IdeModuleProService;
import com.ejianc.foundation.front.business.ide.service.IdeModuleService;
import com.ejianc.foundation.front.business.ide.service.IdeReleaseService;
import com.ejianc.foundation.front.common.http.HttpClientUtil;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.support.idworker.util.IdWorker;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * 版本管理service impl
 */
@Transactional(readOnly = true)
@Service
public class IdeReleaseServiceImpl implements IdeReleaseService {
    private static final Logger log = LoggerFactory.getLogger(IdeReleaseServiceImpl.class);

    @Autowired
    private IdeModuleService moduleService;
    @Autowired
    private IdeAppService ideAppService;
    @Autowired
    private IdeEnvironmentService ideEnvironmentService;
    @Autowired
    private IdeModuleService ideModuleService;
    @Autowired
    private IdeModuleProService ideModuleProService;
    @Autowired
    private IdeReleaseRepo releaseRepo;
    @Autowired
    private IdeReleasePageRepo releasePageRepo;
    @Autowired
    private IdeOnlineRepo onlineRepo;
    @Autowired
    private IdeTeamRepo ideTeamRepo;
    @Autowired
    private IdeModuleRepo ideModuleRepo;
    @Autowired
    IdeModuleProRepo ideModuleProRepo;

    /**
     * 发版
     *
     * @param appId   应用id
     * @param version 版本号
     * @param pageIds 页面ids
     */
    @Transactional
    @Override
    public void release(String appId, String version, List<String> pageIds,String releaseGroupId) {
    	IdeRelease ideRelease = releaseRepo.findByVersion(version,appId);
    	if(ideRelease != null){
    		throw new BusinessException("该版本号已存在，不可发版！");
    	}
        Long releaseId = IdWorker.getId();
        
        IdeRelease release = new IdeRelease();
        release.setId(releaseId);
        release.setVersion(version);
        release.setReleaseTime(new Date());
        release.setAppId(Long.parseLong(appId));
        release.setReleaseGroupId(Long.parseLong(releaseGroupId));
        releaseRepo.save(release);

        List<IdeModule> modules = moduleService.findByIds(pageIds);
        if(modules != null && modules.size() > 0) {
        	for(IdeModule module:modules) {
        		IdeReleasePage page = new IdeReleasePage();
        		page.setId(IdWorker.getId());
        		page.setReleaseId(release.getId());
	            page.setReleaseGroupId(Long.parseLong(releaseGroupId));
	            page.setPageId(module.getId());
	            page.setPageCode(module.getCode());
	            page.setPageName(module.getName());
	            page.setPageData(module.getData());
	            page.setAppId(module.getAppId());
	            releasePageRepo.save(page);
        	}
        }
    }

    /**
     * 查看应用下所有发版
     *
     * @param appId 应用id
     * @return 应用下所有发版
     */
    @Override
    public List<IdeRelease> findReleases(String appId) {
        return releaseRepo.findByAppIdOrderByReleaseTimeDesc(appId);
    }

    /**
     * 查看某版本下所有页面
     *
     * @param releaseGroupId 发版id
     * @return 所有页面
     */
    @Override
    public List<IdeReleasePage> findReleasePages(String releaseGroupId) {
        return releasePageRepo.findByReleaseGroupIdOrderByPageCodeAsc(releaseGroupId);
    }

    /**
     * 上线
     *
     * @param releaseId 发版id
     * @param envUrl    环境地址
     */
    @Transactional
    @Override
    public void online(String releaseGroupId, String envUrl) {
        List<IdeRelease> releases = releaseRepo.findByGroupId(releaseGroupId);
        List<IdeReleasePage> releasePages = releasePageRepo.findByReleaseGroupIdOrderByPageCodeAsc(releaseGroupId);
        if(releases != null&&releases.size()>0){
        	List<IdeOnline> onlines = new ArrayList<>();
        	List<IdeProductionBo> postDatas = new ArrayList<>();
        	for (IdeRelease release : releases) {
        		IdeApp app = ideAppService.findById(String.valueOf(release.getAppId()));
                if (!ideEnvironmentService.isInWhiteList(envUrl, String.valueOf(app.getTeamId()))) {
                    throw new BusinessException(String.format("本团队被限制发布应用到[%s]环境，可联系超级管理员进行授权！", envUrl));
                }
                IdeTeamBo team = new IdeTeamBo(ideTeamRepo.findOne(String.valueOf(app.getTeamId())));
                IdeOnline online = new IdeOnline();
                online.setId(IdWorker.getId());
                online.setAppId(app.getId());
                online.setReleaseGroupId(Long.parseLong(releaseGroupId));
                online.setOnlineTime(new Date());
                online.setOnlineEnvUrl(envUrl);
                onlines.add(online);
                onlineRepo.save(online);
                
                IdeProductionBo postData = new IdeProductionBo();
                postData.setIdeApp(new IdeAppBo(app));
                postData.setIdeTeam(team);
                postData.setToName(envUrl);
                postDatas.add(postData);
                
			}
        	 List<IdeModule> publishModules = releasePages.stream()
                     .map(page -> {
                         IdeModule ideModule = ideModuleRepo.findById(String.valueOf(page.getPageId()));
                         if (ideModule != null) {
                             ideModule.setCode(page.getPageCode());
                             ideModule.setName(page.getPageName());
                             ideModule.setData(page.getPageData());
                             IdeApp app = ideAppService.findById(ideModule.getAppId()+"");
                             ideModule.setAppCode(app.getCode());
                         }
                         return ideModule;
                     })
                     .filter(Objects::nonNull)
                     .collect(Collectors.toList());
             publishModules = ideModuleProService.findDiffLastPublish(publishModules, envUrl);

             if (CollectionUtils.isEmpty(publishModules)) {
                 throw new BusinessException("所发布页面与最新版本相同，无需重复发布");
             }
             for (IdeProductionBo postData : postDatas) {
            	 publishModules.stream()
            	 .filter(module -> postData.getIdeApp().getId().equals(String.valueOf(module.getAppId())))
            	 .forEach(module -> 
            		 postData.getIdeModules().add(new IdeModuleBo(module))
            	 );
                 String postJson = JSONObject.toJSONString(postData);
                 log.info("postJson: {}", postJson);
                 HttpResponse httpresponse = HttpClientUtil.postJSON(envUrl + Constant.PUBLISH_SYNC_URL, postJson);

                 if (httpresponse == null) {
                     throw new BusinessException("发布失败，目标服务器响应异常");
                 }

                 int resCode = httpresponse.getStatusLine().getStatusCode();
                 if (resCode != 200) {
                     throw new BusinessException("发布失败，目标服务器异常，响应状态码：" + resCode);
                 }

                 String result;
                 try {
                     result = EntityUtils.toString(httpresponse.getEntity());
                 } catch (IOException e) {
                     throw new BusinessException("json序列化失败", e);
                 }
                 JSONObject resultObj = (JSONObject) JSONObject.parse(result);

                 boolean success = resultObj.getBoolean("success");
                 if (!success) {
                     log.error("发布失败: {}", result);
                     throw new BusinessException("发布失败");
                 }
			}
           
            publishModules.forEach(module -> ideModuleService.publishHistory(module, "", envUrl));
        }
        
    }

    /**
     * 查找上线历史
     *
     * @param appId 应用id
     * @return 各环境上线历史记录
     */
    @Override
    public Map<String, List<IdeOnline>> viewOnlineHistory(String appId) {
        Map<String, List<IdeOnline>> histories = onlineRepo.findByAppId(appId).stream()
                .collect(Collectors.groupingBy(IdeOnline::getOnlineEnvUrl));

        IdeApp app = ideAppService.findById(appId);
        String teamId = String.valueOf(app.getTeamId());
        IdeTeam ideTeam = ideTeamRepo.findOne(teamId);
        if(ideTeam != null){
        	String teamUrls = ideTeam.getServerUrls();
        	if(StringUtils.isEmpty(teamUrls)){
        		return histories;
        	}else {
        		List<String> teamUrlList = Arrays.asList(teamUrls.split(","));
            	return teamUrlList.stream().
            			collect(Collectors.toMap(url -> url, url -> histories.getOrDefault(url, new ArrayList<>())));
			}	
        }else {
			return histories;
		}
        
        
    }

	@SuppressWarnings("unchecked")
	@Transactional
	@Override
	public void releaseAppIds(Map<String, Object> appList, String version) {
		Long releaseGroupId = IdWorker.getId();
		appList.keySet().forEach(key -> this.release(key, version, (List<String>)appList.get(key),releaseGroupId.toString()));
		
	}

	@Transactional
	@Override
	public void deleteVersion(String releaseGroupId) {
		onlineRepo.deleteByGroupId(releaseGroupId);
		releasePageRepo.deleteByGroupId(releaseGroupId);
		releaseRepo.deleteByGroupId(releaseGroupId);	
	}
}
