package com.ejianc.foundation.front.business.ide.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.ejianc.foundation.front.business.ide.bo.IdeUiDefineBO;
import com.ejianc.foundation.front.business.ide.bo.IdeUiFolderBO;
import com.ejianc.foundation.front.business.ide.bo.IdeUiLibraryBO;
import com.ejianc.foundation.front.business.ide.entity.IdeUiLibrary;
import com.ejianc.foundation.front.business.ide.repository.IdeUiLibraryRepo;
import com.ejianc.foundation.front.business.ide.service.IdeUiFolderService;
import com.ejianc.foundation.front.business.ide.service.IdeUiLibraryService;

@Service
public class IdeUiLibraryServiceImpl implements IdeUiLibraryService {
    @Autowired
    private IdeUiLibraryRepo repo;
    @Autowired
    private IdeUiFolderService folderService;

    /**
     * 创建库
     *
     * @param library 库
     * @return 库id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public String create(IdeUiLibraryBO library) {
        IdeUiLibrary entity = new IdeUiLibrary();
        entity.setName(library.getName());
        repo.save(entity);
        return entity.getId().toString();
    }

    /**
     * 更新库
     *
     * @param library 库
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(IdeUiLibraryBO library) {
        IdeUiLibrary entity = repo.findOne(library.getId());
        entity.setName(library.getName());
        repo.save(entity);
    }

    /**
     * 删除库
     *
     * @param id id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(String id) {
        repo.delete(id);
        // 删除库下文件夹
        folderService.deleteByLibraryId(id);
    }

    /**
     * 查询库列表
     *
     * @return 库列表
     */
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    @Override
    public List<IdeUiLibraryBO> queryList() {
        List<IdeUiLibrary> entities = repo.findAll();

        return entities.stream().map(item -> {
            IdeUiLibraryBO vo = new IdeUiLibraryBO();
            BeanUtils.copyProperties(item, vo);

            vo.setFolders(folderService.findLibraryFolders(String.valueOf(item.getId())));

            return vo;
        }).collect(Collectors.toList());
    }

    /**
     * 查询库详情
     *
     * @param id id
     * @return 库详情
     */
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    @Override
    public IdeUiLibraryBO queryDetail(String id, String searchText) {
        IdeUiLibrary entity = repo.findOne(id);

        if (entity == null) {
            return null;
        }

        IdeUiLibraryBO vo = new IdeUiLibraryBO();
        BeanUtils.copyProperties(entity, vo);

        List<IdeUiFolderBO> folders = folderService.findByLibraryId(id, searchText);
        List<IdeUiDefineBO> defines = new ArrayList<>();

        if (!CollectionUtils.isEmpty(folders)) {
            folders.forEach(item -> {
                defines.addAll(item.getDefines());
                item.setDefines(null);
            });
        }

        vo.setFolders(folders);
        vo.setDefines(defines);

        return vo;
    }
}
