package com.ejianc.foundation.front.business.ide.utils;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.poi.openxml4j.exceptions.OpenXML4JException;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.xssf.extractor.XSSFEventBasedExcelExtractor;
import org.apache.poi.xwpf.extractor.XWPFWordExtractor;
import org.apache.poi.xwpf.usermodel.XWPFDocument;
import org.apache.xmlbeans.XmlException;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.support.idworker.util.IdWorker;

public class PageParser {
    private static final String LABEL = "label";
    private static final String TYPE = "type";
    private static final String PATTERN_PREFIX = "\\$[TF]?(\\.[a-zA-Z]+)?\\{(.*?)}@";
    private static final Map<String, String> typeMap = new HashMap<>();

    static {
        typeMap.put("text", "input");
        typeMap.put("textarea", "textarea");
        typeMap.put("date", "date");
        typeMap.put("time", "time");
        typeMap.put("number", "number");
        typeMap.put("refer", "inputrefer");
        typeMap.put("switch", "switch");
        typeMap.put("enumselect", "enumselect");
        typeMap.put("currency", "currency");
        typeMap.put("attach", "attachupload");
        typeMap.put("radio", "radiogroup");
        typeMap.put("checkbox", "checkboxgroup");
    }

    public static Map<String, Map<String, List<Map<String, String>>>> parse(InputStream is, FileType fileType) {
        String fileText = null;
        try {
            switch (fileType) {
                case WORD:
                    fileText = new XWPFWordExtractor(new XWPFDocument(is)).getText();
                    break;
                case EXCEL:
                    fileText = new XSSFEventBasedExcelExtractor(OPCPackage.open(is)).getText();
                    break;
                default:
                    break;
            }
        } catch (IOException | OpenXML4JException | XmlException e) {
            throw new BusinessException("解析文件内容异常");
        }

        final String content = fileText;
        Map<String, List<Map<String, String>>> forms = new HashMap<>();
        Map<String, List<Map<String, String>>> tables = new HashMap<>();
        typeMap.keySet().forEach(key -> {
            Pattern pattern = Pattern.compile(PATTERN_PREFIX + "(" + key + ")");
            Matcher matcher = pattern.matcher(content);
            while (matcher.find()) {
                String text = matcher.group(0);
                String label = matcher.group(2);
                String type = typeMap.get(matcher.group(3));
                Map<String, String> item = new HashMap<>();
                item.put(LABEL, label);
                item.put(TYPE, type);

                if (text.startsWith("${") || text.startsWith("$F{")) {
                    List<Map<String, String>> formItems = forms.getOrDefault("default", new ArrayList<>());
                    formItems.add(item);
                    forms.put("default", formItems);
                } else if (text.startsWith("$T{")) {
                    List<Map<String, String>> tableItems = tables.getOrDefault("default", new ArrayList<>());
                    tableItems.add(item);
                    tables.put("default", tableItems);
                } else if (text.startsWith("$F.")) {
                    String itemKey = matcher.group(1).substring(1);
                    List<Map<String, String>> formItems = forms.getOrDefault(itemKey, new ArrayList<>());
                    formItems.add(item);
                    forms.put(itemKey, formItems);
                } else if (text.startsWith("$T.")) {
                    String itemKey = matcher.group(1).substring(1);
                    List<Map<String, String>> tableItems = tables.getOrDefault(itemKey, new ArrayList<>());
                    tableItems.add(item);
                    tables.put(itemKey, tableItems);
                }
            }
        });

        Map<String, Map<String, List<Map<String, String>>>> result = new HashMap<>();
        result.put("forms", forms);
        result.put("tables", tables);

        return result;
    }

    public static Map<String, Object> parse(InputStream is, FileType fileType,
                                            Function<Map<String, Map<String, List<Map<String, String>>>>, Map<String, Object>> function) {
        return function.apply(parse(is, fileType));
    }

    public static Map<String, Object> basicParse(InputStream is, FileType fileType) {
        Map<String, Object> labelCol = new HashMap<>();
        labelCol.put("span", 9);
        Map<String, Object> wrapperCol = new HashMap<>();
        wrapperCol.put("span", 15);

        return parse(is, fileType, result -> {
            List<Map<String, Object>> cardChildren = new ArrayList<>();
            result.getOrDefault("forms", new HashMap<>()).forEach((key, value) -> {
                Map<String, Object> form = new HashMap<>();
                form.put("nid", "nid" + "_" + IdWorker.getId());
                form.put("uikey", "form:" + key);
                form.put("uitype", "FormWidget");
                form.put("uititle", "表单:" + key);
                form.put("labelCol", labelCol);
                form.put("wrapperCol", wrapperCol);
                form.put("colnumber", 3);
                form.put("children", value.stream().map(item -> {
                    Map<String, Object> formItem = new HashMap<>();
                    formItem.put("nid", "nid" + "_" + IdWorker.getId());
                    formItem.put("uitype", "FormItemWidget");
                    formItem.put("uisubtype", item.get(TYPE));
                    formItem.put("uititle", item.get(LABEL));
                    formItem.put("label", item.get(LABEL));
                    formItem.put("labelCol", labelCol);
                    formItem.put("wrapperCol", wrapperCol);
                    return formItem;
                }).collect(Collectors.toList()));

                Map<String, Object> cardChild = new HashMap<>();
                cardChild.put("nid", "nid" + "_" + IdWorker.getId());
                cardChild.put("uikey", "card:" + key);
                cardChild.put("uitype", "YYAccordion");
                cardChild.put("uititle", key);
                cardChild.put("header", key);
                cardChild.put("isOpen", true);
                cardChild.put("enableStepTarget", true);
                cardChild.put("children", Collections.singletonList(form));

                cardChildren.add(cardChild);
            });

            Map<String, Object> pageChild = new HashMap<>();
            pageChild.put("nid", "ncardaccordions");
            pageChild.put("uikey", "CPCardBody");
            pageChild.put("uitype", "CPCardBody");
            pageChild.put("uititle", "多卡片");
            pageChild.put("header", "多卡片");
            pageChild.put("children", cardChildren);

            Map<String, Object> pageContent = new HashMap<>();
            pageContent.put("nid", "ncardpage");
            pageContent.put("uikey", "CPCardPage");
            pageContent.put("uitype", "CPCardPage");
            pageContent.put("uititle", "卡片页");
            pageContent.put("children", Collections.singletonList(pageChild));

            return pageContent;
        });
    }

    public enum FileType {
        WORD,
        EXCEL,
    }

    public static void main(String[] args) throws IOException {
        System.out.println(JSONObject.toJSONString(parse(new FileInputStream("E:\\yonyou\\合同模板.docx"), FileType.WORD)));
        System.out.println(JSONObject.toJSONString(parse(new FileInputStream("E:\\yonyou\\上线申请单.xlsx"), FileType.EXCEL)));
        System.out.println(JSONObject.toJSONString(basicParse(new FileInputStream("E:\\yonyou\\合同模板.docx"), FileType.WORD)
                , SerializerFeature.DisableCircularReferenceDetect));
    }
}