package com.ejianc.foundation.front.business.ide.service;

import java.util.List;

import com.ejianc.foundation.front.business.ide.bo.IdeModuleBo;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import com.ejianc.foundation.front.business.ide.bo.IdeTeamBo;
import com.ejianc.foundation.front.business.ide.bo.IdeUserBo;
import com.ejianc.foundation.front.business.ide.entity.IdeTeam;
import com.ejianc.framework.core.exception.BusinessException;

/**
 * Created by cdr_c on 2017/7/1.
 */
public interface IdeTeamService {

    /**
     * 保存团队
     *
     * @param team 团队
     * @return 团队
     */
    IdeTeam save(IdeTeam team);

    /**
     * 更新团队服务地址配置
     *
     * @param id 主键
     * @param serverUrls 服务地址列表
     * @return 团队
     */
    IdeTeam updateServerUrls(String id, String serverUrls);

    /**
     * 获取团队
     *
     * @param id 主键
     * @return 团队
     */
    IdeTeam findById(String id);

    /**
     * 删除团队
     *
     * @param id 主键
     */
    void deleteById(String id);

    /**
     * 分页查询团队
     *
     * @param key      关键字
     * @param userId   用户id
     * @param prodType 产品类型
     * @param pageable 分页信息
     * @return 团队分页列表
     */
    Page<IdeTeamBo> queryForPage(String key, String userId, int prodType, Pageable pageable);

    /**
     * 查询用户归属的团队
     *
     * @param userId   用户id
     * @param prodType 产品类型
     * @return 团队分页列表
     */
    List<IdeTeamBo> queryUserTeams(String userId, int prodType);

    /**
     * 获取团队下用户集合
     *
     * @param teamId 团队id
     * @return 用户集合
     */
    List<IdeUserBo> findUsers(String teamId);

    /**
     * 申请加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     */
    void applyToJoin(String teamId, String userId);

    /**
     * 同意用户加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     */
    void agreeToJoin(String teamId, String userId);

    /**
     * 拒绝用户加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     */
    void refuseToJoin(String teamId, String userId);

    /**
     * 同意用户加入团队（通过账号）
     *
     * @param teamId 团队id
     * @param code   账号
     */
    void agreeToJoinByCode(String teamId, String code);

    /**
     * 从团队移除用户
     *
     * @param teamId  团队id
     * @param userIds 用户id数组
     */
    void removeUsers(String teamId, String[] userIds);

    /**
     * 提升用户为创建者
     *
     * @param teamId  团队id
     * @param userId 用户id
     */
    void updateTeamCreator(String teamId, String userId);

    /**
     * 提升用户为管理员
     *
     * @param teamId  团队id
     * @param userIds 用户id数组
     */
    void setTeamManagers(String teamId, String[] userIds);

    /**
     * 撤销用户管理员角色
     *
     * @param teamId  团队id
     * @param userIds 用户id数组
     */
    void cancelTeamManagers(String teamId, String[] userIds);
    
    /**
     * 分页查询所有团队
     *
     * @param searchText 关键字
     * @param prodType 产品类型
     * @param pageable 分页信息
     * @return 团队分页列表
     */
    Page<IdeTeamBo> findPage(String searchText,int prodType,boolean isDelete, Pageable pageable);
    
    /**
     * 根据产品类型查询所有团队(应用管理的团队树)
     */
    List<IdeTeam> findTeamByProdType(int prodType);

    List<IdeModuleBo> searchByAppName(String searchText,int maxCount);

    //删除（或彻底删除）指定团队
    void deleteByIds(List<String> ids,boolean isRemove) throws BusinessException,Exception;
    
    //恢复数据
    void recoverByIds(List<String> ids) throws BusinessException,Exception;
    
    /**
     * 分页获取团队下用户集合
     *
     * @param teamId 团队id
     * @return 用户集合
     */
    Page<IdeUserBo> findUsersByTeamId(String searchText,boolean isDelete,String teamId,Pageable pageable);
    
    //删除（或彻底删除）指定团队下的指定用户
    void deleteUsersByTeamId(List<String>ids,String teamId,boolean isRemove) throws BusinessException,Exception;
    
    //恢复用户数据
    void recoverUsersByIds(List<String> ids,String teamId) throws BusinessException,Exception;
    
    
}
