package com.ejianc.foundation.front.business.ide.controller;

import java.util.Arrays;

import com.ejianc.foundation.front.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.ejianc.foundation.front.business.ide.entity.IdeTeam;
import com.ejianc.foundation.front.business.ide.service.IdeTeamService;
import com.ejianc.foundation.front.util.JsonBackData;
import com.ejianc.framework.core.exception.BusinessException;

/**
 * Created by cdr_c on 2017/7/1.
 */
@RestController
@RequestMapping(value = "/ide/team")
public class IdeTeamController {
    private static final Logger LOGGER = LoggerFactory.getLogger(IdeTeamController.class);
    @Autowired
    private IdeTeamService teamService;

    /**
     * 保存团队
     *
     * @param team 团队
     * @return {@link JsonBackData}
     */
    @RequestMapping(method = RequestMethod.POST)
    public JsonBackData save(@RequestBody IdeTeam team) {
        JsonBackData backData = new JsonBackData();
        try {
            backData.setBackData(teamService.save(team));
            backData.setBackMsg("保存团队成功");
        } catch (BusinessException e) {
            LOGGER.error("保存团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("保存团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("保存团队失败");
        }
        return backData;
    }

    /**
     * 更新团队服务地址配置
     *
     * @param id         主键
     * @param serverUrls 服务地址列表
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "updateServerUrls", method = RequestMethod.POST)
    public JsonBackData updateServerUrls(String id, String serverUrls) {
        JsonBackData backData = new JsonBackData();
        try {
            if(StringUtils.isEmpty(serverUrls)){
                throw new BusinessException("未选择可发布的环境！");
            }
            backData.setBackData(teamService.updateServerUrls(id, serverUrls));
            backData.setBackMsg("更新团队服务地址配置成功");
        } catch (BusinessException e) {
            LOGGER.error("更新团队服务地址配置失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("更新团队服务地址配置失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("更新团队服务地址配置失败");
        }
        return backData;
    }

    /**
     * 获取团队
     *
     * @param id 主键
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "{id}", method = RequestMethod.GET)
    public JsonBackData get(@PathVariable String id) {
        JsonBackData backData = new JsonBackData();
        try {
            backData.setBackData(teamService.findById(id));
            backData.setBackMsg("获取团队成功");
        } catch (BusinessException e) {
            LOGGER.error("获取团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("获取团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("获取团队失败");
        }
        return backData;
    }

    /**
     * 删除团队
     *
     * @param id 主键
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "{id}", method = RequestMethod.DELETE)
    public JsonBackData delete(@PathVariable String id) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.deleteById(id);
            backData.setBackMsg("删除团队成功");
        } catch (BusinessException e) {
            LOGGER.error("删除团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("删除团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("删除团队失败");
        }
        return backData;
    }

    /**
     * 分页查询全部团队
     *
     * @param key        关键字
     * @param createId   用户id
     * @param prodType   产品类型
     * @param pageNumber 页索引
     * @param pageSize   页大小
     * @return {@link JsonBackData}
     */
    @SuppressWarnings("deprecation")
	@RequestMapping(value = "page", method = RequestMethod.GET)
    public JsonBackData queryTeams(String key, String createId,
                                   @RequestParam(value = "prodType", defaultValue = "0") int prodType,
                                   @RequestParam(value = "pageNumber", defaultValue = "0") int pageNumber,
                                   @RequestParam(value = "pageSize", defaultValue = "10") int pageSize) {
        JsonBackData backData = new JsonBackData();
        try {
            backData.setBackData(teamService.queryForPage(key, createId, prodType, new PageRequest(pageNumber, pageSize,
                    new Sort(Arrays.asList(new Sort.Order(Sort.Direction.DESC, "roleType"),
                            new Sort.Order(Sort.Direction.DESC, "code"))))));
            backData.setBackMsg("分页查询团队成功");
        } catch (BusinessException e) {
            LOGGER.error("分页查询团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("分页查询团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("分页查询团队失败");
        }
        return backData;
    }

    /**
     * 查询用户归属的团队
     *
     * @param createId 用户id
     * @param prodType 产品类型
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "teams/{createId}", method = RequestMethod.GET)
    public JsonBackData queryUserTeams(@PathVariable("createId") String createId,
                                       @RequestParam(value = "prodType", defaultValue = "0") int prodType) {
        JsonBackData backData = new JsonBackData();
        try {
            backData.setBackData(teamService.queryUserTeams(createId, prodType));
            backData.setBackMsg("查询用户归属的团队成功");
        } catch (BusinessException e) {
            LOGGER.error("查询用户归属的团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("查询用户归属的团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("查询用户归属的团队失败");
        }
        return backData;
    }

    /**
     * 获取团队下所有用户
     *
     * @param id 主键
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "users/{id}", method = RequestMethod.GET)
    public JsonBackData queryTeamUsers(@PathVariable String id) {
        JsonBackData backData = new JsonBackData();
        try {
            backData.setBackData(teamService.findUsers(id));
            backData.setBackMsg("获取团队下所有用户成功");
        } catch (BusinessException e) {
            LOGGER.error("获取团队下所有用户失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("获取团队下所有用户失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("获取团队下所有用户失败");
        }
        return backData;
    }

    /**
     * 申请加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "applyToJoin", method = RequestMethod.POST)
    public JsonBackData applyToJoin(String teamId, String userId) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.applyToJoin(teamId, userId);
            backData.setBackMsg("申请加入团队成功");
        } catch (BusinessException e) {
            LOGGER.error("申请加入团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("申请加入团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("申请加入团队失败");
        }
        return backData;
    }

    /**
     * 同意用户加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "agreeToJoin", method = RequestMethod.POST)
    public JsonBackData agreeToJoin(String teamId, String userId) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.agreeToJoin(teamId, userId);
            backData.setBackMsg("同意用户加入团队成功");
        } catch (BusinessException e) {
            LOGGER.error("同意用户加入团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("同意用户加入团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("同意用户加入团队失败");
        }
        return backData;
    }

    /**
     * 拒绝用户加入团队
     *
     * @param teamId 团队id
     * @param userId 用户id
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "refuseToJoin", method = RequestMethod.POST)
    public JsonBackData refuseToJoin(String teamId, String userId) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.refuseToJoin(teamId, userId);
            backData.setBackMsg("拒绝用户加入团队成功");
        } catch (BusinessException e) {
            LOGGER.error("拒绝用户加入团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("拒绝用户加入团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("拒绝用户加入团队失败");
        }
        return backData;
    }

    /**
     * 同意用户加入团队（通过账号）
     *
     * @param teamId 团队id
     * @param code   账号
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "agreeToJoinByCode", method = RequestMethod.POST)
    public JsonBackData agreeToJoinByCode(String teamId, String code) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.agreeToJoinByCode(teamId, code);
            backData.setBackMsg("同意用户加入团队成功");
        } catch (BusinessException e) {
            LOGGER.error("同意用户加入团队失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("同意用户加入团队失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("同意用户加入团队失败");
        }
        return backData;
    }

    /**
     * 从团队中移除用户
     *
     * @param teamId  团队id
     * @param userIds 用户id数组
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "removeUsers", method = RequestMethod.POST)
    public JsonBackData removeUsers(String teamId, String[] userIds) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.removeUsers(teamId, userIds);
            backData.setBackMsg("从团队中移除用户成功");
        } catch (BusinessException e) {
            LOGGER.error("从团队中移除用户失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("从团队中移除用户失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("从团队中移除用户失败");
        }
        return backData;
    }

    /**
     * 提升用户为创建者
     *
     * @param teamId 团队id
     * @param userId 用户id
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "updateTeamCreator")
    public JsonBackData updateTeamCreator(String teamId, String userId) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.updateTeamCreator(teamId, userId);
            backData.setBackMsg("设为创建者成功");
        } catch (BusinessException e) {
            LOGGER.error("设为创建者失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("设为创建者失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("设为创建者失败");
        }
        return backData;
    }

    /**
     * 提升用户为管理员
     *
     * @param teamId  团队id
     * @param userIds 用户id数组
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "setTeamManagers")
    public JsonBackData setTeamManagers(String teamId, String[] userIds) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.setTeamManagers(teamId, userIds);
            backData.setBackMsg("设为管理员成功");
        } catch (BusinessException e) {
            LOGGER.error("设为管理员失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("设为管理员失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("提升用户为管理员失败");
        }
        return backData;
    }

    /**
     * 撤销用户管理员角色
     *
     * @param teamId  团队id
     * @param userIds 用户id数组
     * @return {@link JsonBackData}
     */
    @RequestMapping(value = "cancelTeamManagers")
    public JsonBackData cancelTeamManagers(String teamId, String[] userIds) {
        JsonBackData backData = new JsonBackData();
        try {
            teamService.cancelTeamManagers(teamId, userIds);
            backData.setBackMsg("撤销用户管理员角色成功");
        } catch (BusinessException e) {
            LOGGER.error("撤销用户管理员角色失败：", e.getMessage());
            backData.setSuccess(false);
            backData.setBackMsg(e.getMessage());
        } catch (Exception e) {
            LOGGER.error("撤销用户管理员角色失败", e);
            backData.setSuccess(false);
            backData.setBackMsg("撤销用户管理员角色失败");
        }
        return backData;
    }
}
