package com.ejianc.poc.gyy.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.gyypoc.vo.PurchaseProcessVO;
import com.ejianc.business.gyypoc.vo.SoftwareNeedsListVO;
import com.ejianc.foundation.bpm.api.IBpmApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.poc.gyy.bean.PurchaseProcessEntity;
import com.ejianc.poc.gyy.bean.SoftwareNeedsListEntity;
import com.ejianc.poc.gyy.service.IPurchaseProcessService;
import com.ejianc.poc.gyy.service.ISoftwareNeedsListService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 流程主表控制器
 *
 * @author CJ
 * @Description: 流程主表控制器
 * @date 2021/3/23 15:52
 */
@RestController
@RequestMapping(value = "/purchaseProcess/")
public class PurchaseProcessController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IPurchaseProcessService purchaseProcessService;

    @Autowired
    private IBillCodeApi billCodeApi;

    private final static String BILL_CODE= "GYYPOC_PURCHASE_PROCESS";

    @Autowired
    private ISoftwareNeedsListService softwareNeedsListService;

    private final static String PC_URL = "ejc-gyypoc-frontend/#/purchaseBill?id={id}";
    private final static String MA_URL = "ejc-gyypoc-mobile/#/apply?id={id}";

    @Autowired
    private IBpmApi bpmApi;

    /**
     * 新生成一个采购流程主单据并将单据提交
     *
     * @param purchaseProcessVO
     * @return
     */
    @PostMapping(value = "startPurchaseProcess")
    public CommonResponse<PurchaseProcessVO> addAndSubmit(@RequestBody PurchaseProcessVO purchaseProcessVO) {
        PurchaseProcessVO vo = null;
        //1、创建新的主流程单据
        CommonResponse<String> codeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
        if(!codeResp.isSuccess()) {
            return CommonResponse.error("保存失败，获取自动编码失败！");
        }
        purchaseProcessVO.setBillCode(codeResp.getData());
        purchaseProcessVO.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
        purchaseProcessVO.setProcessState(PurchaseProcessVO.PROCESS_STATE_INIT);

        PurchaseProcessEntity entity = BeanMapper.map(purchaseProcessVO, PurchaseProcessEntity.class);
        purchaseProcessService.saveOrUpdate(entity);

        //2、单据流程提交。
        Map<String, Object> submitParam = new HashMap<>(20);
        submitParam.put("userId", InvocationInfoProxy.getUserid());
        submitParam.put("billtypeId", BILL_CODE);
        submitParam.put("businessKey", entity.getId());
        submitParam.put("assignAble", false);
        submitParam.put("messageType", new String[]{"sys","weixin"});
        submitParam.put("billOrgId", InvocationInfoProxy.getOrgId());
        submitParam.put("procInstName", entity.getName());
        submitParam.put("formurl", PC_URL);
        submitParam.put("maurl", MA_URL);
        logger.info("采购流程提交流程参数：{}", JSONObject.toJSONString(submitParam));
        CommonResponse<String> submitResp = bpmApi.doSubmit(submitParam);
        logger.info("采购流程提交流程结果：{}", JSONObject.toJSONString(submitResp));
        if(!submitResp.isSuccess()) {
            logger.error("采购流程提交审批失败，原因：{}", submitResp.getMsg());
            throw new BusinessException("采购流程启动失败！");
        }

        vo = BeanMapper.map(entity, PurchaseProcessVO.class);
        return CommonResponse.success("流程启动成功！", vo);
    }

    /**
     * 分页查询物资总计划列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<IPage<PurchaseProcessVO>> pageList(@RequestBody QueryParam queryParam) {
        queryParam.getFuzzyFields().add("name");
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);

        IPage<PurchaseProcessEntity> pageData = purchaseProcessService.queryPage(queryParam);

        IPage<PurchaseProcessVO> resp = new Page<>();
        resp.setSize(pageData.getSize());
        resp.setRecords(BeanMapper.mapList(pageData.getRecords(), PurchaseProcessVO.class));
        resp.setCurrent(pageData.getCurrent());
        resp.setTotal(pageData.getTotal());
        resp.setPages(pageData.getPages());

        return CommonResponse.success("分页查询采购流程列表成功", resp);
    }

    /**
     * 根据采购流程单据审批详情
     *
     * @param id
     * @return
     */
    @GetMapping("queryDetail")
    public CommonResponse<PurchaseProcessVO> queryDetail(@RequestParam(value = "id") Long id) {
        PurchaseProcessVO vo = null;

        PurchaseProcessEntity dbEntity = purchaseProcessService.selectById(id);
        if(null != dbEntity) {
            vo = BeanMapper.map(dbEntity, PurchaseProcessVO.class);

            QueryWrapper<SoftwareNeedsListEntity> softQuery = new QueryWrapper<>();
            softQuery.eq("purchase_process_id", id);
            softQuery.orderByAsc("purchase_bill_id");

            List<SoftwareNeedsListEntity> softList = softwareNeedsListService.list(softQuery);
            List<SoftwareNeedsListVO> softVos = new ArrayList<>();

            if(CollectionUtils.isNotEmpty(softList)) {
                BigDecimal zero = BigDecimal.ZERO.setScale(8);
                BigDecimal applyBudgetAmt = zero;
                BigDecimal reviewBudgetAmt = zero;
                for(SoftwareNeedsListEntity soft : softList) {
                    reviewBudgetAmt = reviewBudgetAmt.add(null != soft.getReviewBudgetAmt() ? soft.getReviewBudgetAmt() : zero);
                    applyBudgetAmt = applyBudgetAmt.add(null != soft.getBudgetAmt() ? soft.getBudgetAmt() : zero);
                    softVos.add(BeanMapper.map(soft, SoftwareNeedsListVO.class));
                }
                vo.setReviewBudgetAmt(reviewBudgetAmt);
                vo.setApplyBudgetAmt(applyBudgetAmt);
                vo.setSoftwareNeedsList(BeanMapper.mapList(softList, SoftwareNeedsListVO.class));
            }
        }
        return CommonResponse.success("查询购流程单据详情成功！", vo);
    }


    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<PurchaseProcessVO> saveOrUpdate(@RequestBody PurchaseProcessVO purchaseProcessVO) {
        PurchaseProcessEntity dbEntity = purchaseProcessService.selectById(purchaseProcessVO.getId());

        BigDecimal reviewBudgetAmt = BigDecimal.ZERO.setScale(8);
        BigDecimal applyBudgetAmt = BigDecimal.ZERO.setScale(8);
        List<SoftwareNeedsListEntity> saveList = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(purchaseProcessVO.getSoftwareNeedsList())) {
            for(SoftwareNeedsListVO soft : purchaseProcessVO.getSoftwareNeedsList()) {
                reviewBudgetAmt = reviewBudgetAmt.add(soft.getReviewBudgetAmt());
                applyBudgetAmt = applyBudgetAmt.add(soft.getBudgetAmt());
                saveList.add(BeanMapper.map(soft, SoftwareNeedsListEntity.class));
            }

            dbEntity.setReviewBudgetAmt(reviewBudgetAmt);
            dbEntity.setApplyBudgetAmt(applyBudgetAmt);
        }
        dbEntity.setMeetingSummary(purchaseProcessVO.getMeetingSummary());

        purchaseProcessVO = purchaseProcessService.savePurChaseProcess(dbEntity, saveList);
        return CommonResponse.success("保存成功！", purchaseProcessVO);
    }
}
