package com.ejianc.framework.idmclient.cas.logout;

import java.util.Arrays;
import java.util.List;
import java.util.zip.Inflater;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.jasig.cas.client.session.HashMapBackedSessionMappingStorage;
import org.jasig.cas.client.session.SessionMappingStorage;
import org.jasig.cas.client.util.CommonUtils;
import org.jasig.cas.client.util.XmlUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.ejianc.framework.idmclient.cas.logout.entity.TenantUser;
import com.ejianc.framework.idmclient.sdk.RedisUtils;

public final class SingleSignOutHandler {
	public static final String DEFAULT_ARTIFACT_PARAMETER_NAME = "ticket";
	public static final String DEFAULT_LOGOUT_PARAMETER_NAME = "logoutRequest";
	public static final String DEFAULT_FRONT_LOGOUT_PARAMETER_NAME = "SAMLRequest";
	public static final String DEFAULT_RELAY_STATE_PARAMETER_NAME = "RelayState";
	private static final int DECOMPRESSION_FACTOR = 10;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private SessionMappingStorage sessionMappingStorage = new HashMapBackedSessionMappingStorage();
	private String artifactParameterName = "ticket";
	private String logoutParameterName = "logoutRequest";
	private String frontLogoutParameterName = "SAMLRequest";
	private String relayStateParameterName = "RelayState";
	private String casServerUrlPrefix;
	private boolean artifactParameterOverPost = false;
	private List<String> safeParameters;

	public void setSessionMappingStorage(SessionMappingStorage storage) {
		this.sessionMappingStorage = storage;
	}

	public void setArtifactParameterOverPost(boolean artifactParameterOverPost) {
		this.artifactParameterOverPost = artifactParameterOverPost;
	}

	public SessionMappingStorage getSessionMappingStorage() {
		return this.sessionMappingStorage;
	}

	public void setArtifactParameterName(String name) {
		this.artifactParameterName = name;
	}

	public void setLogoutParameterName(String name) {
		this.logoutParameterName = name;
	}

	public void setCasServerUrlPrefix(String casServerUrlPrefix) {
		this.casServerUrlPrefix = casServerUrlPrefix;
	}

	public void setFrontLogoutParameterName(String name) {
		this.frontLogoutParameterName = name;
	}

	public void setRelayStateParameterName(String name) {
		this.relayStateParameterName = name;
	}

	public synchronized void init() {
		if (this.safeParameters == null) {
			CommonUtils.assertNotNull(this.artifactParameterName, "artifactParameterName cannot be null.");
			CommonUtils.assertNotNull(this.logoutParameterName, "logoutParameterName cannot be null.");
			CommonUtils.assertNotNull(this.frontLogoutParameterName, "frontLogoutParameterName cannot be null.");
			CommonUtils.assertNotNull(this.sessionMappingStorage, "sessionMappingStorage cannot be null.");
			CommonUtils.assertNotNull(this.relayStateParameterName, "relayStateParameterName cannot be null.");
			CommonUtils.assertNotNull(this.casServerUrlPrefix, "casServerUrlPrefix cannot be null.");
			if (this.artifactParameterOverPost) {
				this.safeParameters = Arrays.asList(new String[]{this.logoutParameterName, this.artifactParameterName});
			} else {
				this.safeParameters = Arrays.asList(new String[]{this.logoutParameterName});
			}
		}

	}

	private boolean isTokenRequest(HttpServletRequest request) {
		return CommonUtils.isNotBlank(CommonUtils.safeGetParameter(request, this.artifactParameterName, this.safeParameters));
	}

	private boolean isBackChannelLogoutRequest(HttpServletRequest request) {
		return "POST".equals(request.getMethod()) && !this.isMultipartRequest(request) && CommonUtils.isNotBlank(CommonUtils.safeGetParameter(request, this.logoutParameterName, this.safeParameters));
	}

	private boolean isFrontChannelLogoutRequest(HttpServletRequest request) {
		return "GET".equals(request.getMethod())&& CommonUtils.isNotBlank(CommonUtils.safeGetParameter(request, this.frontLogoutParameterName));
	}

	public boolean process(HttpServletRequest request, HttpServletResponse response) {
		if (this.isTokenRequest(request)) {
			this.logger.trace("Received a token request");
			this.recordToken(request, response);
			return true;
		} else if (this.isBackChannelLogoutRequest(request)) {
			this.logger.trace("Received a back channel logout request");
			this.destroyToken(request, response);
			return false;
		} else if (this.isFrontChannelLogoutRequest(request)) {
			this.logger.trace("Received a front channel logout request");
			this.destroyToken(request, response);
			String redirectionUrl = this.computeRedirectionToServer(request);
			if (redirectionUrl != null) {
				CommonUtils.sendRedirect(response, redirectionUrl);
			}
			return false;
		} else {
			this.logger.trace("Ignoring URI for logout: {}", request.getRequestURI());
			return true;
		}
	}

	private void recordToken(HttpServletRequest request, HttpServletResponse response) {
		String token = CommonUtils.safeGetParameter(request, this.artifactParameterName, this.safeParameters);
		this.logger.debug("Recording session for token {}", token);
	}

	private String uncompressLogoutMessage(String originalMessage) {
		byte[] binaryMessage = Base64.decodeBase64(originalMessage);
		Inflater decompresser = null;

		String arg5;
		try {
			decompresser = new Inflater();
			decompresser.setInput(binaryMessage);
			byte[] e = new byte[binaryMessage.length * DECOMPRESSION_FACTOR];
			int resultLength = decompresser.inflate(e);
			arg5 = new String(e, 0, resultLength, "UTF-8");
		} catch (Exception arg9) {
			this.logger.error("Unable to decompress logout message", arg9);
			throw new RuntimeException(arg9);
		} finally {
			if (decompresser != null) {
				decompresser.end();
			}

		}

		return arg5;
	}

	private void destroyToken(HttpServletRequest request, HttpServletResponse response) {
		String logoutMessage;
		if (this.isFrontChannelLogoutRequest(request)) {
			logoutMessage = this.uncompressLogoutMessage(CommonUtils.safeGetParameter(request, this.frontLogoutParameterName));
		} else {
			logoutMessage = CommonUtils.safeGetParameter(request, this.logoutParameterName, this.safeParameters);
		}

		this.logger.trace("Logout request:\n{}", logoutMessage);
		String token = XmlUtils.getTextForElement(logoutMessage, "SessionIndex");
		String userId = XmlUtils.getTextForElement(logoutMessage, "UserId");
		if (CommonUtils.isNotBlank(userId)) {
			this.destroyToken(token, userId, request, response);
			HttpSession session = request.getSession(false);
			if (session != null) {
				session.invalidate();
			}
		}

	}

	private void destroyToken(String ticket, String userId, HttpServletRequest request, HttpServletResponse response) {
		if (!StringUtils.isBlank(userId)) {
			String saveUserid = ticket + "__" + userId;
			TenantUser user = RedisUtils.getUserCache("user.info.login.tenant:" + saveUserid);
			Cookie[] cookies = request.getCookies();
			if (cookies != null) {
				Cookie[] arr$ = cookies;
				int len$ = cookies.length;

				for (int i$ = 0; i$ < len$; ++i$) {
					Cookie c = arr$[i$];
					if (c.getName().equals("tenant_token") || c.getName().equals("tenant_username")) {
						c.setValue((String) null);
						c.setMaxAge(-1);
						c.setHttpOnly(true);
						c.setPath("/");
						response.addCookie(c);
					}
				}
			}

			if (user != null) {
				RedisUtils.disCacheUser(saveUserid);
			}

			RedisUtils.removeSessionCacheAttribute("tenant_assertion", saveUserid);
		}

	}

	private String computeRedirectionToServer(HttpServletRequest request) {
		String relayStateValue = CommonUtils.safeGetParameter(request, this.relayStateParameterName);
		if (StringUtils.isNotBlank(relayStateValue)) {
			StringBuilder buffer = new StringBuilder();
			buffer.append(this.casServerUrlPrefix);
			if (!this.casServerUrlPrefix.endsWith("/")) {
				buffer.append("/");
			}

			buffer.append("logout?_eventId=next&");
			buffer.append(this.relayStateParameterName);
			buffer.append("=");
			buffer.append(CommonUtils.urlEncode(relayStateValue));
			String redirectUrl = buffer.toString();
			this.logger.debug("Redirection url to the CAS server: {}", redirectUrl);
			return redirectUrl;
		} else {
			return null;
		}
	}

	private boolean isMultipartRequest(HttpServletRequest request) {
		return request.getContentType() != null && request.getContentType().toLowerCase().startsWith("multipart");
	}
}