package com.ejianc.framework.idmclient.sdk;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.nio.charset.Charset;

import org.springside.modules.nosql.redis.JedisTemplate;
import org.springside.modules.nosql.redis.JedisUtils;
import org.springside.modules.nosql.redis.pool.JedisPool;

import com.ejianc.framework.idmclient.cas.logout.entity.TenantUser;
import com.ejianc.framework.idmclient.esapi.EncryptException;
import com.ejianc.framework.idmclient.utils.ContextUtils;
import com.ejianc.framework.idmclient.utils.TokenGenerator;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.exceptions.JedisConnectionException;

public class UserCacheManagerUtils {
	public static final String TOKEN_SEED = "token_seed";
	public static final String DEFAULT_CHARSET = "UTF-8";
	private int sessionTimeout;
	private JedisTemplate jedisTemplate;

	private static UserCacheManagerUtils instance;

	public static UserCacheManagerUtils getInstance() {
		if (UserCacheManagerUtils.instance == null) {
			synchronized (UserCacheManagerUtils.class) {
				if (UserCacheManagerUtils.instance == null) {
					instance = new UserCacheManagerUtils();
					if (instance.jedisTemplate == null) {
						instance = null;
					}
				}
			}
		}
		return instance;
	}

	private UserCacheManagerUtils() {
		try {
			jedisTemplate = ContextUtils.getBean(JedisTemplate.class);
			this.sessionTimeout = 3600;
		} catch (Exception err) {
			err.printStackTrace();
		}
	}


	public TenantUser getUserCache(String userName) {
		Jedis jedis = null;
		boolean borrowOrOprSuccess = true;
		JedisPool pool = this.jedisTemplate.getJedisPool();
		byte[] value = null;
		byte[] userNameBytes = userName.getBytes(Charset.forName("UTF-8"));
		try {
			jedis = pool.getResource();
			value = jedis.get(userNameBytes);
			if (value != null && value.length > 0) {
				jedis.expire(userNameBytes, sessionTimeout);
			}
		} catch (JedisConnectionException e) {
			borrowOrOprSuccess = false;
			e.printStackTrace();
		} finally {
			closeResource(pool, jedis, borrowOrOprSuccess);
		}

		if (value == null) {
			return null;
		}
		
		return (TenantUser) byteToObject(value);
	}

	private String createUserCacheKey(String uname) {
		return "user.info.login.tenant" + ":" + uname;
	}

	private void closeResource(JedisPool pool, Jedis jedis, boolean borrowOrOprSuccess) {
		try {
			if (jedis != null) {
				if (borrowOrOprSuccess) {
					pool.returnResource(jedis);
				} else {
					pool.returnBrokenResource(jedis);
				}
			}
		} catch (Exception e) {
			if (jedis != null) {
				JedisUtils.destroyJedis(jedis);
			}
			e.printStackTrace();
		}
	}

	public Object byteToObject(byte[] bytes) {
		Object obj = null;
		try {
			ByteArrayInputStream bi = new ByteArrayInputStream(bytes);
			ObjectInputStream oi = new ObjectInputStream(bi);
			obj = oi.readObject();
			bi.close();
			oi.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return obj;
	}

	public String findSeed() {
		String seed = jedisTemplate.get(TOKEN_SEED);
		if (org.apache.commons.lang.StringUtils.isBlank(seed)) {
			try {
				seed = TokenGenerator.genSeed();
			} catch (EncryptException e) {
				e.printStackTrace();
				return null;
			}
			this.jedisTemplate.set(TOKEN_SEED, seed);
		}
		return seed;
	}

	@SuppressWarnings("unchecked")
	public <T extends Serializable> T getSessionCacheAttribute(String sid, String key) {
		Jedis jedis = null;
		boolean borrowOrOprSuccess = true;
		JedisPool pool = this.jedisTemplate.getJedisPool();
		byte[] value = null;
		byte[] keyBytes = sid.getBytes(Charset.forName("UTF-8"));
		byte[] fieldBytes = key.getBytes(Charset.forName("UTF-8"));

		try {
			jedis = pool.getResource();
			value = jedis.hget(keyBytes, fieldBytes);
			if (value != null && value.length > 0)
				jedis.expire(sid, sessionTimeout);

		} catch (JedisConnectionException e) {
			borrowOrOprSuccess = false;
			e.printStackTrace();
		} finally {
			closeResource(pool, jedis, borrowOrOprSuccess);
		}

		if (value == null) {
			return null;
		}
		
		return (T) byteToObject(value);
	}

	public <T> void cacheUser(String uname, T user) {
		Jedis jedis = null;
		boolean borrowOrOprSuccess = true;
		JedisPool pool = this.jedisTemplate.getJedisPool();

		byte[] keyBytes = createUserCacheKey(uname).getBytes(Charset.forName("UTF-8"));
		byte[] valueBytes = objectToByte(user);

		try {
			jedis = pool.getResource();
			jedis.setex(keyBytes, sessionTimeout, valueBytes);

		} catch (JedisConnectionException e) {
			borrowOrOprSuccess = false;
			e.printStackTrace();
		} finally {
			closeResource(pool, jedis, borrowOrOprSuccess);
		}
	}

	static byte[] objectToByte(Object obj) {
		byte[] bytes = null;
		try {
			ByteArrayOutputStream bo = new ByteArrayOutputStream();
			ObjectOutputStream oo = new ObjectOutputStream(bo);
			oo.writeObject(obj);
			bytes = bo.toByteArray();
			bo.close();
			oo.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return bytes;
	}

	public <T> void putSessionCacheAttribute(String sid, String key, T value) {
		Jedis jedis = null;
		boolean borrowOrOprSuccess = true;
		JedisPool pool = this.jedisTemplate.getJedisPool();

		byte[] keyBytes = sid.getBytes(Charset.forName("UTF-8"));
		byte[] fieldBytes = key.getBytes(Charset.forName("UTF-8"));
		byte[] valueBytes = objectToByte(value);

		try {
			jedis = pool.getResource();
			jedis.hset(keyBytes, fieldBytes, valueBytes);
			jedis.expire(sid, sessionTimeout);

		} catch (JedisConnectionException e) {
			borrowOrOprSuccess = false;
			e.printStackTrace();
		} finally {
			closeResource(pool, jedis, borrowOrOprSuccess);
		}
	}

	public void disCacheUser(String uname) {
		Jedis jedis = null;
		boolean borrowOrOprSuccess = true;
		JedisPool pool = this.jedisTemplate.getJedisPool();

		try {
			jedis = pool.getResource();
			jedis.del(createUserCacheKey(uname));

		} catch (JedisConnectionException e) {
			borrowOrOprSuccess = false;
			e.printStackTrace();
		} finally {
			closeResource(pool, jedis, borrowOrOprSuccess);
		}
	}

	public void removeSessionCacheAttribute(String sid, String key) {
		Jedis jedis = null;
		boolean borrowOrOprSuccess = true;
		JedisPool pool = this.jedisTemplate.getJedisPool();

		try {
			jedis = pool.getResource();
			jedis.hdel(sid, key);

		} catch (JedisConnectionException e) {
			borrowOrOprSuccess = false;
			e.printStackTrace();
		} finally {
			closeResource(pool, jedis, borrowOrOprSuccess);
		}
	}

}
