package com.ejianc.foundation.cache;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.ejianc.foundation.workbench.vo.MenuItemVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.cache.serializer.Serializer;
import com.ejianc.framework.cache.serializer.impl.DefaultJDKSerializer;

@Component
public class MenuItemCacheManage {

	public static final String DEFAULT_CHARSET = "UTF-8";
	// 所有菜单缓存
	private final static String MENUITEM_LIST_KEY = "menuitem_list_key:";

	@Autowired
	private CacheManager cacheManager;

	private Serializer serializer = new DefaultJDKSerializer();

	/**
	 * 把菜单list放入缓存
	 * 
	 * @param menuItemList
	 */
	public void putMenuItemCache(Long tenantId, List<MenuItemVO> menuItemList) {
		if (menuItemList != null && menuItemList.size() > 0) {
			cacheManager.piplineExecute(pipeline -> {
				pipeline.multi();
				String key = MENUITEM_LIST_KEY + tenantId;
				for (MenuItemVO menuItemVo : menuItemList) {
					byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
					byte[] fieldBytes = menuItemVo.getId().toString().getBytes(Charset.forName(DEFAULT_CHARSET));
					byte[] contentBytes = serializer.marshalToByte(JSON.toJSONString(menuItemVo));
					pipeline.hset(keyBytes, fieldBytes, contentBytes);
				}
				pipeline.exec();
			});
		}
	}

	/**
	 * 获取redis中的全部菜单
	 * 
	 * @return
	 */
	public List<MenuItemVO> getMenuItemsCache(Long tenantId) {
		List<MenuItemVO> menuItemList = new ArrayList<MenuItemVO>();
		String key = MENUITEM_LIST_KEY + tenantId;
		Map<byte[], byte[]> resultMap = cacheManager.hgetAll(key);
		if(resultMap != null && !resultMap.isEmpty()) {
			MenuItemVO menuItemVo = null;
			for(Map.Entry<byte[], byte[]> entry:resultMap.entrySet()) {
				byte[] content = entry.getValue();
				String menuItemJson = (String) serializer.unMarshal(content);
				menuItemVo = JSON.parseObject(menuItemJson, MenuItemVO.class);
				menuItemList.add(menuItemVo);
			}
		}
		return menuItemList;
	}

	/**
	 * 删除redis中的某个菜单
	 * 
	 * @param menuItemVo
	 */
	public void removeMenuItemCache(Long tenantId, MenuItemVO menuItemVo) {
		String key = MENUITEM_LIST_KEY + tenantId;
		cacheManager.hdel(key, menuItemVo.getId().toString());
	}

}
